package io.dyte.core.controllers.stage

import io.dyte.core.controllers.DyteStageStatus
import io.dyte.core.controllers.ParticipantController
import io.dyte.core.controllers.SelfController
import io.dyte.core.events.EventEmitter
import io.dyte.core.events.InternalEvents
import io.dyte.core.feat.ParticipantFlags
import io.dyte.core.host.IHostController
import io.dyte.core.models.DyteJoinedMeetingParticipant
import io.dyte.core.models.DyteSelfParticipant
import io.dyte.core.observability.DyteLogger
import io.dyte.core.socket.socketservice.ISockratesSocketService
import io.dyte.core.socket.socketservice.SocketServiceEventListener
import io.dyte.core.socket.socketservice.SocketServiceUtils
import kotlinx.coroutines.CoroutineScope
import socket.livestreaming.DenyStageAccessRequest
import socket.livestreaming.GetStageRequestsResponse
import socket.livestreaming.GrantStageAccessRequest
import socket.livestreaming.LeaveStageRequest
import socket.room.PeerStatusUpdate

internal class StageSocketServiceController(
  val self: DyteSelfParticipant,
  private val socketService: ISockratesSocketService,
  scope: CoroutineScope,
  private val participantController: ParticipantController,
  private val selfController: SelfController,
  private val hostController: IHostController,
  private val internalEventsEmitter: EventEmitter<InternalEvents>
) : BaseStageController(self, scope) {

  // we dont show any viewers in case of livestream stage. this is only to support webinar
  override val viewers: List<DyteJoinedMeetingParticipant>
    get() = emptyList()

  private val socketEventListener =
    object : SocketServiceEventListener {
      override fun onEvent(event: Int, eventId: String?, payload: ByteArray?) {
        when (event) {
          /** triggered whenever there is any change in stage access requests */
          SocketServiceUtils.RoomEvent.GET_STAGE_REQUESTS.id -> {
            payload?.let {
              val response = GetStageRequestsResponse.ADAPTER.decode(payload)
              _accessRequests.clear()
              response.stage_requests.forEach { requestPeer ->
                val participant =
                  DyteJoinedMeetingParticipant(
                    requestPeer.peer_id,
                    requestPeer.user_id,
                    requestPeer.display_name,
                    null,
                    false,
                    null,
                    ParticipantFlags(false, false, false),
                    "",
                    participantController,
                    hostController,
                    self.selfPermissions
                  )
                _accessRequests.add(participant)
              }
            }
            emitEvent { it.onStageRequestsUpdated(_accessRequests) }
          }
          SocketServiceUtils.RoomEvent.PEER_STAGE_STATUS_UPDATE.id -> {
            payload?.let {
              val response = PeerStatusUpdate.ADAPTER.decode(payload)
              val stageStatusChangeUserId = response.user_id
              if (stageStatusChangeUserId == self.userId) {
                val dyteStage = response.stage_type
                dyteStage?.let {
                  val stageStatus = DyteStageStatus.fromStageType(dyteStage)
                  self._stageStatus = stageStatus
                  selfController.emitEvent { it.onStageStatusUpdated(stageStatus) }
                  emitEvent { it.onStageStatusUpdated(stageStatus) }
                }
              } else {
                val peer = accessRequests.firstOrNull { it.id == response.user_id }
                // TODO : emit participant stage status change callback here
              }
            }
              ?: run {
                DyteLogger.error(
                  "StageSocketServiceController::empty payload for PEER_STAGE_STATUS_UPDATE"
                )
              }
          }
          else -> {}
        }
      }
    }

  override fun setupEvents() {
    socketService.subscribe(
      SocketServiceUtils.RoomEvent.REQUEST_STAGE_ACCESS.id,
      socketEventListener
    )
    socketService.subscribe(
      SocketServiceUtils.RoomEvent.PEER_STAGE_STATUS_UPDATE.id,
      socketEventListener
    )
    socketService.subscribe(SocketServiceUtils.RoomEvent.GET_STAGE_REQUESTS.id, socketEventListener)
  }

  override suspend fun requestAccess() {
    socketService.send(SocketServiceUtils.RoomEvent.REQUEST_STAGE_ACCESS.id, null)
    self._stageStatus = DyteStageStatus.REQUESTED_TO_JOIN_STAGE
  }

  override suspend fun denyAccess(id: String) {
    socketService.send(
      SocketServiceUtils.RoomEvent.DENY_STAGE_REQUEST.id,
      DenyStageAccessRequest(listOf(id)).encode()
    )
  }

  override suspend fun denyAccessAll() {
    val ids = _accessRequests.map { it.userId }
    socketService.send(
      SocketServiceUtils.RoomEvent.DENY_STAGE_REQUEST.id,
      DenyStageAccessRequest(ids).encode()
    )
  }

  override suspend fun grantAccess(id: String) {
    socketService.send(
      SocketServiceUtils.RoomEvent.GRANT_STAGE_REQUEST.id,
      GrantStageAccessRequest(listOf(id)).encode()
    )
  }

  override suspend fun grantAccessAll() {
    val ids = _accessRequests.map { it.userId }
    socketService.send(
      SocketServiceUtils.RoomEvent.GRANT_STAGE_REQUEST.id,
      GrantStageAccessRequest(ids).encode()
    )
  }

  override suspend fun join() {
    socketService.send(SocketServiceUtils.RoomEvent.JOIN_STAGE.id, null)
    self._stageStatus = DyteStageStatus.ON_STAGE
    internalEventsEmitter.emitEvent { it.connectToRoomNode() }
    // internalEventsEmitter.emitEvent { it.connectMedia() }
    selfController.emitEvent { it.onStageStatusUpdated(DyteStageStatus.ON_STAGE) }
    emitEvent { it.onStageStatusUpdated(DyteStageStatus.ON_STAGE) }
  }

  override suspend fun leave() {
    internalEventsEmitter.emitEvent { it.disconnectFromRoomNode() }
    val request = LeaveStageRequest(listOf(self.userId)).encode()
    socketService.send(SocketServiceUtils.RoomEvent.LEAVE_STAGE.id, request)
    self._stageStatus = DyteStageStatus.OFF_STAGE
    selfController.emitEvent { it.onStageStatusUpdated(DyteStageStatus.OFF_STAGE) }
    emitEvent { it.onStageStatusUpdated(DyteStageStatus.OFF_STAGE) }
  }

  override suspend fun kick(id: String) {
    val leaveStageRequest = LeaveStageRequest(listOf(id)).encode()
    socketService.send(SocketServiceUtils.RoomEvent.LEAVE_STAGE.id, leaveStageRequest)
  }

  override suspend fun cancelRequestAccess() {
    socketService.send(SocketServiceUtils.RoomEvent.CANCEL_STAGE_REQUEST.id, null)
  }
}
