/*
 * Copyright (c) 2017 Henry Lin @zxcpoiu
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
package io.dyte.core.incallmanager

import android.annotation.SuppressLint
import android.app.Activity
import android.content.BroadcastReceiver
import android.content.Context
import android.content.Intent
import android.content.IntentFilter
import android.content.pm.PackageManager
import android.media.AudioAttributes
import android.media.AudioAttributes.Builder
import android.media.AudioDeviceInfo
import android.media.AudioFocusRequest
import android.media.AudioManager
import android.media.AudioManager.OnAudioFocusChangeListener
import android.media.MediaPlayer
import android.media.ToneGenerator
import android.net.Uri
import android.os.Build.VERSION
import android.os.Build.VERSION_CODES
import android.os.Bundle
import android.os.Handler
import android.os.PowerManager
import android.provider.Settings.System
import android.util.Log
import android.view.KeyEvent
import android.view.WindowManager
import android.view.WindowManager.LayoutParams
import androidx.annotation.RequiresApi
import io.dyte.core.incallmanager.InCallManagerModule.AudioDevice.BLUETOOTH
import io.dyte.core.incallmanager.InCallManagerModule.AudioDevice.EARPIECE
import io.dyte.core.incallmanager.InCallManagerModule.AudioDevice.NONE
import io.dyte.core.incallmanager.InCallManagerModule.AudioDevice.SPEAKER_PHONE
import io.dyte.core.incallmanager.InCallManagerModule.AudioDevice.WIRED_HEADSET
import io.dyte.core.incallmanager.apprtc.AppRTCBluetoothManager
import io.dyte.core.incallmanager.apprtc.AppRTCBluetoothManager.Companion.create
import io.dyte.core.incallmanager.apprtc.AppRTCBluetoothManager.State.HEADSET_AVAILABLE
import io.dyte.core.incallmanager.apprtc.AppRTCBluetoothManager.State.HEADSET_UNAVAILABLE
import io.dyte.core.incallmanager.apprtc.AppRTCBluetoothManager.State.SCO_CONNECTED
import io.dyte.core.incallmanager.apprtc.AppRTCBluetoothManager.State.SCO_CONNECTING
import io.dyte.core.incallmanager.apprtc.AppRTCBluetoothManager.State.SCO_DISCONNECTING
import java.io.File
import java.util.Collections
import okhttp3.internal.notify

private const val maxWaitTimeMs = 3600000 // 1 hour fairly enough
private const val loadBufferWaitTimeMs = 20
private const val toneVolume = 100

// The volume of the tone, given in percentage of maximum volume (from 0-100).
// --- constant in ToneGenerator all below 100
const val BEEP = 101
const val BUSY = 102
const val CALLEND = 103
const val CALLWAITING = 104
const val RINGBACK = 105
const val SILENT = 106

class InCallManagerModule(
  activity: Activity,
  inCallManagerEventCallbacks: InCallManagerEventCallbacks
) : OnAudioFocusChangeListener {
  private val mPackageName = "io.dyte.core.sample"
  private val activity: Activity?

  // --- Screen Manager
  private val mPowerManager: PowerManager
  private var lastLayoutParams: LayoutParams? = null
  private val mWindowManager: WindowManager

  // --- AudioRouteManager
  private val audioManager: AudioManager
  private var audioManagerActivated = false
  private var isAudioFocused = false

  // private final Object mAudioFocusLock = new Object();
  private var isOrigAudioSetupStored = false
  private var origIsSpeakerPhoneOn = false
  private var origIsMicrophoneMute = false
  private var origAudioMode = AudioManager.MODE_INVALID
  private var defaultSpeakerOn = false
  private val defaultAudioMode = AudioManager.MODE_IN_COMMUNICATION
  private var forceSpeakerOn = 0
  private var automatic = true
  private var isProximityRegistered = false
  private val proximityIsNear = false
  private var wiredHeadsetReceiver: BroadcastReceiver? = null
  private var noisyAudioReceiver: BroadcastReceiver? = null
  private var mediaButtonReceiver: BroadcastReceiver? = null
  private var mAudioAttributes: AudioAttributes? = null
  private var mAudioFocusRequest: AudioFocusRequest? = null

  // --- same as: RingtoneManager.getActualDefaultRingtoneUri(reactContext,
  // RingtoneManager.TYPE_RINGTONE);
  private val defaultRingtoneUri = System.DEFAULT_RINGTONE_URI
  private val defaultRingbackUri = System.DEFAULT_RINGTONE_URI
  private val defaultBusytoneUri = System.DEFAULT_NOTIFICATION_URI

  // private Uri defaultAlarmAlertUri = Settings.System.DEFAULT_ALARM_ALERT_URI; // --- too annoying
  private val bundleRingtoneUri: Uri? = null
  private val bundleRingbackUri: Uri? = null
  private val bundleBusytoneUri: Uri? = null
  private val audioUriMap: MutableMap<String, Uri?>
  private var mRingtone: MyPlayerInterface? = null
  private var mRingback: MyPlayerInterface? = null
  private var mBusytone: MyPlayerInterface? = null
  private var mRingtoneCountDownHandler: Handler? = null
  private var media = "audio"
  private val inCallManagerEventCallbacks: InCallManagerEventCallbacks

  /** AudioDevice is the names of possible audio devices that we currently support. */
  enum class AudioDevice {
    SPEAKER_PHONE,
    WIRED_HEADSET,
    EARPIECE,
    BLUETOOTH,
    NONE
  }

  /** AudioManager state. */
  enum class AudioManagerState {
    UNINITIALIZED,
    PREINITIALIZED,
    RUNNING
  }

  private val savedAudioMode = AudioManager.MODE_INVALID
  private val savedIsSpeakerPhoneOn = false
  private val savedIsMicrophoneMute = false
  private var hasWiredHeadset = false

  // Default audio device; speaker phone for video calls or earpiece for audio
  // only calls.
  private var defaultAudioDevice = NONE

  /** Returns the currently selected audio device. */
  // Contains the currently selected audio device.
  // This device is changed automatically using a certain scheme where e.g.
  // a wired headset "wins" over speaker phone. It is also possible for a
  // user to explicitly select a device (and overrid any predefined scheme).
  // See |userSelectedAudioDevice| for details.
  var selectedAudioDevice: AudioDevice? = null
    private set

  // Contains the user-selected audio device which overrides the predefined
  // selection scheme.
  // TODO(henrika): always set to AudioDevice.NONE today. Add support for
  // explicit selection based on choice by userSelectedAudioDevice.
  private var userSelectedAudioDevice: AudioDevice? = null

  // Contains speakerphone setting: auto, true or false
  private val useSpeakerphone = SPEAKERPHONE_AUTO

  // Handles all tasks related to Bluetooth headset devices.
  private var bluetoothManager: AppRTCBluetoothManager? = null
  private val proximityManager: InCallProximityManager
  private val wakeLockUtils: InCallWakeLockUtils

  // Contains a list of available audio devices. A Set collection is used to
  // avoid duplicate elements.
  private var audioDevices: MutableSet<AudioDevice> = HashSet()

  internal interface MyPlayerInterface {
    val isMediaPlaying: Boolean

    fun startPlay(data: Map<String?, Any?>?)

    fun stopPlay()
  }

  private fun manualTurnScreenOff() {
    Log.d(TAG, "manualTurnScreenOff()")
    if (activity == null) {
      Log.d(TAG, "ReactContext doesn't hava any Activity attached.")
      return
    }
    val window = activity.window
    val params = window.attributes
    lastLayoutParams = params // --- store last param
    params.screenBrightness =
      LayoutParams
        .BRIGHTNESS_OVERRIDE_OFF // --- Dim as dark as possible. see BRIGHTNESS_OVERRIDE_OFF
    window.attributes = params
    window.clearFlags(LayoutParams.FLAG_KEEP_SCREEN_ON)
  }

  private fun manualTurnScreenOn() {
    Log.d(TAG, "manualTurnScreenOn()")
    if (activity == null) {
      Log.d(TAG, "ReactContext doesn't hava any Activity attached.")
      return
    }
    val window = activity.window
    if (lastLayoutParams != null) {
      window.attributes = lastLayoutParams
    } else {
      val params = window.attributes
      params.screenBrightness = -1f // --- Dim to preferable one
      window.attributes = params
    }
    window.addFlags(LayoutParams.FLAG_KEEP_SCREEN_ON)
  }

  @SuppressLint("WrongConstant")
  private fun storeOriginalAudioSetup() {
    Log.d(TAG, "storeOriginalAudioSetup()")
    if (!isOrigAudioSetupStored) {
      origAudioMode = audioManager.mode
      origIsSpeakerPhoneOn = audioManager.isSpeakerphoneOn
      origIsMicrophoneMute = audioManager.isMicrophoneMute
      isOrigAudioSetupStored = true
    }
  }

  @SuppressLint("WrongConstant")
  private fun restoreOriginalAudioSetup() {
    Log.d(TAG, "restoreOriginalAudioSetup()")
    if (isOrigAudioSetupStored) {
      setSpeakerphoneOn(origIsSpeakerPhoneOn)
      setMicrophoneMute(origIsMicrophoneMute)
      audioManager.mode = origAudioMode
      if (activity != null) {
        activity.volumeControlStream = AudioManager.USE_DEFAULT_STREAM_TYPE
      }
      isOrigAudioSetupStored = false
    }
  }

  private fun startWiredHeadsetEvent() {
    if (wiredHeadsetReceiver == null) {
      Log.d(TAG, "startWiredHeadsetEvent()")
      val filter = IntentFilter(ACTION_HEADSET_PLUG)
      wiredHeadsetReceiver =
        object : BroadcastReceiver() {
          override fun onReceive(context: Context, intent: Intent) {
            if (ACTION_HEADSET_PLUG == intent.action) {
              hasWiredHeadset = intent.getIntExtra("state", 0) == 1
              updateAudioRoute()
              var deviceName = intent.getStringExtra("name")
              if (deviceName == null) {
                deviceName = ""
              }
              val data = Bundle()
              data.putBoolean("isPlugged", intent.getIntExtra("state", 0) == 1)
              data.putBoolean("hasMic", intent.getIntExtra("microphone", 0) == 1)
              data.putString("deviceName", deviceName)
              sendEvent("WiredHeadset", data)
            } else {
              hasWiredHeadset = false
            }
          }
        }
      if (activity != null) {
        if (VERSION.SDK_INT >= VERSION_CODES.TIRAMISU) {
          activity.registerReceiver(wiredHeadsetReceiver, filter, Context.RECEIVER_NOT_EXPORTED)
        } else {
          activity.registerReceiver(wiredHeadsetReceiver, filter)
        }
      } else {
        Log.d(TAG, "startWiredHeadsetEvent() reactContext is null")
      }
    }
  }

  private fun stopWiredHeadsetEvent() {
    if (wiredHeadsetReceiver != null) {
      Log.d(TAG, "stopWiredHeadsetEvent()")
      unregisterReceiver(wiredHeadsetReceiver)
      wiredHeadsetReceiver = null
    }
  }

  private fun startNoisyAudioEvent() {
    if (noisyAudioReceiver == null) {
      Log.d(TAG, "startNoisyAudioEvent()")
      val filter = IntentFilter(AudioManager.ACTION_AUDIO_BECOMING_NOISY)
      noisyAudioReceiver =
        object : BroadcastReceiver() {
          override fun onReceive(context: Context, intent: Intent) {
            if (AudioManager.ACTION_AUDIO_BECOMING_NOISY == intent.action) {
              updateAudioRoute()
              sendEvent("NoisyAudio", null)
            }
          }
        }
      if (activity != null) {
        if (VERSION.SDK_INT >= VERSION_CODES.TIRAMISU) {
          activity.registerReceiver(noisyAudioReceiver, filter, Context.RECEIVER_NOT_EXPORTED)
        } else {
          activity.registerReceiver(noisyAudioReceiver, filter)
        }
      } else {
        Log.d(TAG, "startNoisyAudioEvent() reactContext is null")
      }
    }
  }

  private fun stopNoisyAudioEvent() {
    if (noisyAudioReceiver != null) {
      Log.d(TAG, "stopNoisyAudioEvent()")
      unregisterReceiver(noisyAudioReceiver)
      noisyAudioReceiver = null
    }
  }

  @Suppress("DEPRECATION")
  private fun startMediaButtonEvent() {
    if (mediaButtonReceiver == null) {
      Log.d(TAG, "startMediaButtonEvent()")
      val filter = IntentFilter(Intent.ACTION_MEDIA_BUTTON)
      mediaButtonReceiver =
        object : BroadcastReceiver() {
          override fun onReceive(context: Context, intent: Intent) {
            if (Intent.ACTION_MEDIA_BUTTON == intent.action) {
              val event =
                if (VERSION.SDK_INT >= VERSION_CODES.TIRAMISU) {
                  intent.getParcelableExtra(Intent.EXTRA_KEY_EVENT, KeyEvent::class.java)
                } else {
                  intent.getParcelableExtra(Intent.EXTRA_KEY_EVENT)
                }
              val keyCode = event!!.keyCode
              val keyText =
                when (keyCode) {
                  KeyEvent.KEYCODE_MEDIA_PLAY -> "KEYCODE_MEDIA_PLAY"
                  KeyEvent.KEYCODE_MEDIA_PAUSE -> "KEYCODE_MEDIA_PAUSE"
                  KeyEvent.KEYCODE_MEDIA_PLAY_PAUSE -> "KEYCODE_MEDIA_PLAY_PAUSE"
                  KeyEvent.KEYCODE_MEDIA_NEXT -> "KEYCODE_MEDIA_NEXT"
                  KeyEvent.KEYCODE_MEDIA_PREVIOUS -> "KEYCODE_MEDIA_PREVIOUS"
                  KeyEvent.KEYCODE_MEDIA_CLOSE -> "KEYCODE_MEDIA_CLOSE"
                  KeyEvent.KEYCODE_MEDIA_EJECT -> "KEYCODE_MEDIA_EJECT"
                  KeyEvent.KEYCODE_MEDIA_RECORD -> "KEYCODE_MEDIA_RECORD"
                  KeyEvent.KEYCODE_MEDIA_STOP -> "KEYCODE_MEDIA_STOP"
                  else -> "KEYCODE_UNKNOW"
                }
              val data = Bundle()
              data.putString("eventText", keyText)
              data.putInt("eventCode", keyCode)
              sendEvent("MediaButton", data)
            }
          }
        }
      if (activity != null) {
        if (VERSION.SDK_INT >= VERSION_CODES.TIRAMISU) {
          activity.registerReceiver(mediaButtonReceiver, filter, Context.RECEIVER_NOT_EXPORTED)
        } else {
          activity.registerReceiver(mediaButtonReceiver, filter)
        }
      } else {
        Log.d(TAG, "startMediaButtonEvent() reactContext is null")
      }
    }
  }

  private fun stopMediaButtonEvent() {
    if (mediaButtonReceiver != null) {
      Log.d(TAG, "stopMediaButtonEvent()")
      unregisterReceiver(mediaButtonReceiver)
      mediaButtonReceiver = null
    }
  }

  fun onProximitySensorChangedState(isNear: Boolean) {
    if (automatic && selectedAudioDevice == EARPIECE) {
      if (isNear) {
        turnScreenOff()
      } else {
        turnScreenOn()
      }
      updateAudioRoute()
    }
    val data = Bundle()
    data.putBoolean("isNear", isNear)
    sendEvent("Proximity", data)
  }

  fun startProximitySensor() {
    if (!proximityManager.isProximitySupported) {
      Log.d(TAG, "Proximity Sensor is not supported.")
      return
    }
    if (isProximityRegistered) {
      Log.d(TAG, "Proximity Sensor is already registered.")
      return
    }
    // --- SENSOR_DELAY_FASTEST(0 milisecs), SENSOR_DELAY_GAME(20 milisecs), SENSOR_DELAY_UI(60
    // milisecs), SENSOR_DELAY_NORMAL(200 milisecs)
    if (!proximityManager.start()) {
      Log.d(TAG, "proximityManager.start() failed. return false")
      return
    }
    Log.d(TAG, "startProximitySensor()")
    isProximityRegistered = true
  }

  fun stopProximitySensor() {
    if (!proximityManager.isProximitySupported) {
      Log.d(TAG, "Proximity Sensor is not supported.")
      return
    }
    if (!isProximityRegistered) {
      Log.d(TAG, "Proximity Sensor is not registered.")
      return
    }
    Log.d(TAG, "stopProximitySensor()")
    proximityManager.stop()
    isProximityRegistered = false
  }

  // --- see: https://developer.android.com/reference/android/media/AudioManager
  override fun onAudioFocusChange(focusChange: Int) {
    val focusChangeStr: String
    focusChangeStr =
      when (focusChange) {
        AudioManager.AUDIOFOCUS_GAIN -> "AUDIOFOCUS_GAIN"
        AudioManager.AUDIOFOCUS_GAIN_TRANSIENT -> "AUDIOFOCUS_GAIN_TRANSIENT"
        AudioManager.AUDIOFOCUS_GAIN_TRANSIENT_EXCLUSIVE -> "AUDIOFOCUS_GAIN_TRANSIENT_EXCLUSIVE"
        AudioManager.AUDIOFOCUS_GAIN_TRANSIENT_MAY_DUCK -> "AUDIOFOCUS_GAIN_TRANSIENT_MAY_DUCK"
        AudioManager.AUDIOFOCUS_LOSS -> "AUDIOFOCUS_LOSS"
        AudioManager.AUDIOFOCUS_LOSS_TRANSIENT -> "AUDIOFOCUS_LOSS_TRANSIENT"
        AudioManager.AUDIOFOCUS_LOSS_TRANSIENT_CAN_DUCK -> "AUDIOFOCUS_LOSS_TRANSIENT_CAN_DUCK"
        AudioManager.AUDIOFOCUS_NONE -> "AUDIOFOCUS_NONE"
        else -> "AUDIOFOCUS_UNKNOW"
      }
    Log.d(TAG, "onAudioFocusChange(): $focusChange - $focusChangeStr")
    val data = Bundle()
    data.putString("eventText", focusChangeStr)
    data.putInt("eventCode", focusChange)
    sendEvent("onAudioFocusChange", data)
  }

  /*
      // --- TODO: AudioDeviceCallBack android sdk 23+
      if (android.os.Build.VERSION.SDK_INT >= 23) {
          private class MyAudioDeviceCallback extends AudioDeviceCallback {
              public void onAudioDevicesAdded(AudioDeviceInfo[] addedDevices) {
                  mAddCallbackCalled = true;
              }
              public void onAudioDevicesRemoved(AudioDeviceInfo[] removedDevices) {
                  mRemoveCallbackCalled = true;
              }
          }

          // --- Specifies the Handler object for the thread on which to execute the callback. If null, the Handler associated with the main Looper will be used.
          public void test_deviceCallback() {
              AudioDeviceCallback callback =  new EmptyDeviceCallback();
              mAudioManager.registerAudioDeviceCallback(callback, null);
          }

          // --- get all audio devices by flags
          //public AudioDeviceInfo[] getDevices (int flags)
          //Returns an array of AudioDeviceInfo objects corresponding to the audio devices currently connected to the system and meeting the criteria specified in the flags parameter.
          //flags    int: A set of bitflags specifying the criteria to test.
      }

      // --- TODO: adjust valume if needed.
      if (android.os.Build.VERSION.SDK_INT >= 21) {
          isVolumeFixed ()

          // The following APIs have no effect when volume is fixed:
          adjustVolume(int, int)
          adjustSuggestedStreamVolume(int, int, int)
          adjustStreamVolume(int, int, int)
          setStreamVolume(int, int, int)
          setRingerMode(int)
          setStreamSolo(int, boolean)
          setStreamMute(int, boolean)
      }

      // -- TODO: bluetooth support
  */
  private fun sendEvent(eventName: String, params: Any?) {
    try {
      inCallManagerEventCallbacks.onEvent(eventName, params)
    } catch (e: RuntimeException) {
      Log.e(
        TAG,
        "sendEvent(): java.lang.RuntimeException: Trying to invoke JS before CatalystInstance has been set!"
      )
    }
  }

  fun start(_media: String, auto: Boolean, ringbackUriType: String) {
    media = _media
    defaultSpeakerOn = media == "video"
    automatic = auto
    if (!audioManagerActivated) {
      audioManagerActivated = true
      Log.d(TAG, "start audioRouteManager")
      wakeLockUtils.acquirePartialWakeLock()
      if (mRingtone != null && mRingtone!!.isMediaPlaying) {
        Log.d(TAG, "stop ringtone")
        stopRingtone() // --- use brandnew instance
      }
      storeOriginalAudioSetup()
      requestAudioFocus()
      startEvents()
      bluetoothManager!!.start()

      // TODO: even if not acquired focus, we can still play sounds. but need figure out which is
      // better.
      // getCurrentActivity().setVolumeControlStream(AudioManager.STREAM_VOICE_CALL);
      audioManager.mode = defaultAudioMode
      setSpeakerphoneOn(defaultSpeakerOn)
      setMicrophoneMute(false)
      forceSpeakerOn = 0
      hasWiredHeadset = hasWiredHeadset()
      defaultAudioDevice =
        if (defaultSpeakerOn) SPEAKER_PHONE else if (hasEarpiece()) EARPIECE else SPEAKER_PHONE
      userSelectedAudioDevice = NONE
      selectedAudioDevice = NONE
      audioDevices.clear()
      updateAudioRoute()
      if (!ringbackUriType.isEmpty()) {
        startRingback(ringbackUriType)
      }
    } else {
      Log.d(TAG, "not starting")
    }
  }

  @JvmOverloads
  fun stop(busytoneUriType: String = "") {
    if (audioManagerActivated) {
      stopRingback()
      if (!busytoneUriType.isEmpty() && startBusytone(busytoneUriType)) {
        // play busytone first, and call this func again when finish
        Log.d(TAG, "play busytone before stop InCallManager")
        return
      } else {
        Log.d(TAG, "stop() InCallManager")
        stopBusytone()
        stopEvents()
        setSpeakerphoneOn(false)
        setMicrophoneMute(false)
        forceSpeakerOn = 0
        bluetoothManager!!.stop()
        restoreOriginalAudioSetup()
        abandonAudioFocus()
        audioManagerActivated = false
      }
      wakeLockUtils.releasePartialWakeLock()
    }
  }

  private fun startEvents() {
    startWiredHeadsetEvent()
    startNoisyAudioEvent()
    startMediaButtonEvent()
    startProximitySensor() // --- proximity event always enable, but only turn screen off when audio
    // is routing to earpiece.
    setKeepScreenOn(true)
  }

  private fun stopEvents() {
    stopWiredHeadsetEvent()
    stopNoisyAudioEvent()
    stopMediaButtonEvent()
    stopProximitySensor()
    setKeepScreenOn(false)
    turnScreenOn()
  }

  private fun requestAudioFocus(): String {
    val requestAudioFocusResStr =
      if (VERSION.SDK_INT >= 26) requestAudioFocusV26() else requestAudioFocusOld()
    Log.d(TAG, "requestAudioFocus(): res = $requestAudioFocusResStr")
    return requestAudioFocusResStr
  }

  @RequiresApi(api = VERSION_CODES.O)
  private fun requestAudioFocusV26(): String {
    if (isAudioFocused) {
      return ""
    }
    if (mAudioAttributes == null) {
      mAudioAttributes =
        Builder()
          .setUsage(AudioAttributes.USAGE_VOICE_COMMUNICATION)
          .setContentType(AudioAttributes.CONTENT_TYPE_SPEECH)
          .build()
    }
    if (mAudioFocusRequest == null) {
      mAudioFocusRequest =
        AudioFocusRequest.Builder(AudioManager.AUDIOFOCUS_GAIN_TRANSIENT)
          .setAudioAttributes(mAudioAttributes!!)
          .setAcceptsDelayedFocusGain(false)
          .setWillPauseWhenDucked(false)
          .setOnAudioFocusChangeListener(this)
          .build()
    }
    val requestAudioFocusRes = audioManager.requestAudioFocus(mAudioFocusRequest!!)
    val requestAudioFocusResStr: String
    when (requestAudioFocusRes) {
      AudioManager.AUDIOFOCUS_REQUEST_FAILED ->
        requestAudioFocusResStr = "AUDIOFOCUS_REQUEST_FAILED"
      AudioManager.AUDIOFOCUS_REQUEST_GRANTED -> {
        isAudioFocused = true
        requestAudioFocusResStr = "AUDIOFOCUS_REQUEST_GRANTED"
      }
      AudioManager.AUDIOFOCUS_REQUEST_DELAYED ->
        requestAudioFocusResStr = "AUDIOFOCUS_REQUEST_DELAYED"
      else -> requestAudioFocusResStr = "AUDIOFOCUS_REQUEST_UNKNOWN"
    }
    return requestAudioFocusResStr
  }

  @Suppress("DEPRECATION")
  private fun requestAudioFocusOld(): String {
    if (isAudioFocused) {
      return ""
    }
    val requestAudioFocusRes =
      audioManager.requestAudioFocus(
        this,
        AudioManager.STREAM_VOICE_CALL,
        AudioManager.AUDIOFOCUS_GAIN_TRANSIENT
      )
    val requestAudioFocusResStr: String
    when (requestAudioFocusRes) {
      AudioManager.AUDIOFOCUS_REQUEST_FAILED ->
        requestAudioFocusResStr = "AUDIOFOCUS_REQUEST_FAILED"
      AudioManager.AUDIOFOCUS_REQUEST_GRANTED -> {
        isAudioFocused = true
        requestAudioFocusResStr = "AUDIOFOCUS_REQUEST_GRANTED"
      }
      else -> requestAudioFocusResStr = "AUDIOFOCUS_REQUEST_UNKNOWN"
    }
    return requestAudioFocusResStr
  }

  private fun abandonAudioFocus(): String {
    val abandonAudioFocusResStr =
      if (VERSION.SDK_INT >= 26) abandonAudioFocusV26() else abandonAudioFocusOld()
    Log.d(TAG, "abandonAudioFocus(): res = $abandonAudioFocusResStr")
    return abandonAudioFocusResStr
  }

  @RequiresApi(api = VERSION_CODES.O)
  private fun abandonAudioFocusV26(): String {
    if (!isAudioFocused || mAudioFocusRequest == null) {
      return ""
    }
    val abandonAudioFocusRes = audioManager.abandonAudioFocusRequest(mAudioFocusRequest!!)
    val abandonAudioFocusResStr: String
    when (abandonAudioFocusRes) {
      AudioManager.AUDIOFOCUS_REQUEST_FAILED ->
        abandonAudioFocusResStr = "AUDIOFOCUS_REQUEST_FAILED"
      AudioManager.AUDIOFOCUS_REQUEST_GRANTED -> {
        isAudioFocused = false
        abandonAudioFocusResStr = "AUDIOFOCUS_REQUEST_GRANTED"
      }
      else -> abandonAudioFocusResStr = "AUDIOFOCUS_REQUEST_UNKNOWN"
    }
    return abandonAudioFocusResStr
  }

  @Suppress("DEPRECATION")
  private fun abandonAudioFocusOld(): String {
    if (!isAudioFocused) {
      return ""
    }
    val abandonAudioFocusRes = audioManager.abandonAudioFocus(this)
    val abandonAudioFocusResStr: String
    when (abandonAudioFocusRes) {
      AudioManager.AUDIOFOCUS_REQUEST_FAILED ->
        abandonAudioFocusResStr = "AUDIOFOCUS_REQUEST_FAILED"
      AudioManager.AUDIOFOCUS_REQUEST_GRANTED -> {
        isAudioFocused = false
        abandonAudioFocusResStr = "AUDIOFOCUS_REQUEST_GRANTED"
      }
      else -> abandonAudioFocusResStr = "AUDIOFOCUS_REQUEST_UNKNOWN"
    }
    return abandonAudioFocusResStr
  }

  fun turnScreenOn() {
    if (proximityManager.isProximityWakeLockSupported) {
      Log.d(TAG, "turnScreenOn(): use proximity lock.")
      proximityManager.releaseProximityWakeLock(true)
    } else {
      Log.d(TAG, "turnScreenOn(): proximity lock is not supported. try manually.")
      manualTurnScreenOn()
    }
  }

  fun turnScreenOff() {
    if (proximityManager.isProximityWakeLockSupported) {
      Log.d(TAG, "turnScreenOff(): use proximity lock.")
      proximityManager.acquireProximityWakeLock()
    } else {
      Log.d(TAG, "turnScreenOff(): proximity lock is not supported. try manually.")
      manualTurnScreenOff()
    }
  }

  fun setKeepScreenOn(enable: Boolean) {
    Log.d(TAG, "setKeepScreenOn() $enable")
    if (activity == null) {
      Log.d(TAG, "ReactContext doesn't hava any Activity attached.")
      return
    }
    val window = activity.window
    if (enable) {
      window.addFlags(LayoutParams.FLAG_KEEP_SCREEN_ON)
    } else {
      window.clearFlags(LayoutParams.FLAG_KEEP_SCREEN_ON)
    }
  }

  fun setSpeakerphoneOn(enable: Boolean) {
    if (enable != audioManager.isSpeakerphoneOn) {
      Log.d(TAG, "setSpeakerphoneOn(): $enable")
      audioManager.isSpeakerphoneOn = enable
    }
  }
  // --- TODO (zxcpoiu): These two api name is really confusing. should be changed.
  /** flag: Int 0: use default action 1: force speaker on -1: force speaker off */
  fun setForceSpeakerphoneOn(flag: Int) {
    if (flag < -1 || flag > 1) {
      return
    }
    Log.d(TAG, "setForceSpeakerphoneOn() flag: $flag")
    forceSpeakerOn = flag

    // --- will call updateAudioDeviceState()
    // --- Note: in some devices, it may not contains specified route thus will not be effected.
    if (flag == 1) {
      selectAudioDevice(SPEAKER_PHONE)
    } else if (flag == -1) {
      selectAudioDevice(EARPIECE) // --- use the most common earpiece to force `speaker off`
    } else {
      selectAudioDevice(
        NONE
      ) // --- NONE will follow default route, the default route of `video` call is speaker.
    }
  }

  // --- TODO (zxcpoiu): Implement api to let user choose audio devices
  fun setMicrophoneMute(enable: Boolean) {
    if (enable != audioManager.isMicrophoneMute) {
      Log.d(TAG, "setMicrophoneMute(): $enable")
      audioManager.isMicrophoneMute = enable
    }
  }

  /** This is part of start() process. ringbackUriType must not empty. empty means do not play. */
  fun startRingback(ringbackUriType: String) {
    if (ringbackUriType.isEmpty()) {
      return
    }
    try {
      Log.d(TAG, "startRingback(): UriType=$ringbackUriType")
      if (mRingback != null) {
        if (mRingback!!.isMediaPlaying) {
          Log.d(TAG, "startRingback(): is already playing")
          return
        }
        stopRingback() // --- use brandnew instance
      }
      val ringbackUri: Uri?
      val data: MutableMap<String, Any> = HashMap<String, Any>()
      data["name"] = "mRingback"

      // --- use ToneGenerator instead file uri
      if (ringbackUriType == "_DTMF_") {
        mRingback = DyteToneGenerator(RINGBACK)
        (mRingback as DyteToneGenerator).startPlay(data.toMap())
        return
      }
      ringbackUri = getRingbackUri(ringbackUriType)
      if (ringbackUri == null) {
        Log.d(TAG, "startRingback(): no available media")
        return
      }
      mRingback = DyteMediaPlayer()
      data["sourceUri"] = ringbackUri
      data["setLooping"] = true

      // data.put("audioStream", AudioManager.STREAM_VOICE_CALL); // --- lagacy
      // --- The ringback doesn't have to be a DTMF.
      // --- Should use VOICE_COMMUNICATION for sound during call or it may be silenced.
      data["audioUsage"] = AudioAttributes.USAGE_VOICE_COMMUNICATION
      data["audioContentType"] = AudioAttributes.CONTENT_TYPE_MUSIC
      setMediaPlayerEvents(mRingback as MediaPlayer, "mRingback")
      (mRingback as DyteMediaPlayer).startPlay(data.toMap())
    } catch (e: Exception) {
      Log.d(TAG, "startRingback() failed", e)
    }
  }

  fun stopRingback() {
    try {
      if (mRingback != null) {
        mRingback!!.stopPlay()
        mRingback = null
      }
    } catch (e: Exception) {
      Log.d(TAG, "stopRingback() failed")
    }
  }

  /**
   * This is part of start() process. busytoneUriType must not empty. empty means do not play.
   * return false to indicate play tone failed and should be stop() immediately otherwise, it will
   * stop() after a tone completed.
   */
  fun startBusytone(busytoneUriType: String): Boolean {
    return if (busytoneUriType.isEmpty()) {
      false
    } else
      try {
        Log.d(TAG, "startBusytone(): UriType=$busytoneUriType")
        if (mBusytone != null) {
          if (mBusytone!!.isMediaPlaying) {
            Log.d(TAG, "startBusytone(): is already playing")
            return false
          }
          stopBusytone() // --- use brandnew instance
        }
        val busytoneUri: Uri?
        val data: MutableMap<String, Any> = HashMap<String, Any>()
        data["name"] = "mBusytone"

        // --- use ToneGenerator instead file uri
        if (busytoneUriType == "_DTMF_") {
          mBusytone = DyteToneGenerator(BUSY)
          (mBusytone as DyteToneGenerator).startPlay(data.toMap())
          return true
        }
        busytoneUri = getBusytoneUri(busytoneUriType)
        if (busytoneUri == null) {
          Log.d(TAG, "startBusytone(): no available media")
          return false
        }
        mBusytone = DyteMediaPlayer()
        data["sourceUri"] = busytoneUri
        data["setLooping"] = false
        // data.put("audioStream", AudioManager.STREAM_VOICE_CALL); // --- lagacy
        // --- Should use VOICE_COMMUNICATION for sound during a call or it may be silenced.
        data["audioUsage"] = AudioAttributes.USAGE_VOICE_COMMUNICATION
        data["audioContentType"] =
          AudioAttributes.CONTENT_TYPE_SONIFICATION // --- CONTENT_TYPE_MUSIC?
        setMediaPlayerEvents(mBusytone as MediaPlayer, "mBusytone")
        (mBusytone as DyteMediaPlayer).startPlay(data.toMap())
        true
      } catch (e: Exception) {
        Log.d(TAG, "startBusytone() failed", e)
        false
      }
  }

  private fun stopBusytone() {
    try {
      if (mBusytone != null) {
        mBusytone!!.stopPlay()
        mBusytone = null
      }
    } catch (e: Exception) {
      Log.d(TAG, "stopBusytone() failed")
    }
  }

  private fun stopRingtone() {
    val thread: Thread =
      object : Thread() {
        override fun run() {
          try {
            if (mRingtone != null) {
              mRingtone!!.stopPlay()
              mRingtone = null
              restoreOriginalAudioSetup()
            }
            if (mRingtoneCountDownHandler != null) {
              mRingtoneCountDownHandler!!.removeCallbacksAndMessages(null)
              mRingtoneCountDownHandler = null
            }
          } catch (e: Exception) {
            Log.d(TAG, "stopRingtone() failed")
          }
          wakeLockUtils.releasePartialWakeLock()
        }
      }
    thread.start()
  }

  private fun setMediaPlayerEvents(mp: MediaPlayer, name: String) {
    mp.setOnErrorListener { _, what, extra ->
      // http://developer.android.com/reference/android/media/MediaPlayer.OnErrorListener.html
      Log.d(TAG, String.format("MediaPlayer %s onError(). what: %d, extra: %d", name, what, extra))
      // return True if the method handled the error
      // return False, or not having an OnErrorListener at all, will cause the OnCompletionListener
      // to be called. Get news & tips
      true
    }
    mp.setOnInfoListener { _, what, extra ->

      // http://developer.android.com/reference/android/media/MediaPlayer.OnInfoListener.html
      Log.d(TAG, String.format("MediaPlayer %s onInfo(). what: %d, extra: %d", name, what, extra))
      // return True if the method handled the info
      // return False, or not having an OnInfoListener at all, will cause the info to be discarded.
      true
    }
    mp.setOnPreparedListener {
      Log.d(
        TAG,
        String.format(
          "MediaPlayer %s onPrepared(), start play, isSpeakerPhoneOn %b",
          name,
          audioManager.isSpeakerphoneOn
        )
      )
      when (name) {
        "mBusytone" -> {
          audioManager.mode = AudioManager.MODE_IN_COMMUNICATION
        }
        "mRingback" -> {
          audioManager.mode = AudioManager.MODE_IN_COMMUNICATION
        }
        "mRingtone" -> {
          audioManager.mode = AudioManager.MODE_RINGTONE
        }
      }
      updateAudioRoute()
      mp.start()
    }
    mp.setOnCompletionListener {
      Log.d(TAG, String.format("MediaPlayer %s onCompletion()", name))
      if (name == "mBusytone") {
        Log.d(TAG, "MyMediaPlayer(): invoke stop()")
        stop()
      }
    }
  }

  private fun getRingtoneUri(_type: String): Uri? {
    val fileBundle = "incallmanager_ringtone"
    val fileBundleExt = "mp3"
    val fileSysPath =
      "/system/media/audio/ui" // --- every devices all ships with different in ringtone. maybe ui
    // sounds are more "stock"
    // --- _type MAY be empty
    val type: String =
      if (_type == "_DEFAULT_" || _type.isEmpty()) {
        // type = fileSysWithExt;
        return getDefaultUserUri("defaultRingtoneUri")
      } else {
        _type
      }
    return fileBundleExt.getAudioUri(
      type,
      fileBundle,
      fileSysPath,
      "bundleRingtoneUri",
      "defaultRingtoneUri"
    )
  }

  private fun getRingbackUri(_type: String): Uri? {
    val fileBundle = "incallmanager_ringback"
    val fileBundleExt = "mp3"
    val fileSysPath =
      "/system/media/audio/ui" // --- every devices all ships with different in ringtone. maybe ui
    // sounds are more "stock"
    // --- _type would never be empty here. just in case.
    val type: String =
      if (_type == "_DEFAULT_" || _type.isEmpty()) {
        // type = fileSysWithExt;
        return getDefaultUserUri("defaultRingbackUri")
      } else {
        _type
      }
    return fileBundleExt.getAudioUri(
      type,
      fileBundle,
      fileSysPath,
      "bundleRingbackUri",
      "defaultRingbackUri"
    )
  }

  private fun getBusytoneUri(_type: String): Uri? {
    val fileBundle = "incallmanager_busytone"
    val fileBundleExt = "mp3"
    val fileSysPath =
      "/system/media/audio/ui" // --- every devices all ships with different in ringtone. maybe ui
    // sounds are more "stock"
    // --- _type would never be empty here. just in case.
    val type: String =
      if (_type == "_DEFAULT_" || _type.isEmpty()) {
        // type = fileSysWithExt; // ---
        return getDefaultUserUri("defaultBusytoneUri")
      } else {
        _type
      }
    return fileBundleExt.getAudioUri(
      type,
      fileBundle,
      fileSysPath,
      "bundleBusytoneUri",
      "defaultBusytoneUri"
    )
  }

  @SuppressLint("DiscouragedApi")
  private fun String.getAudioUri(
    _type: String,
    fileBundle: String,
    fileSysPath: String,
    uriBundle: String,
    uriDefault: String
  ): Uri? {
    if (_type == "_BUNDLE_") {
      return if (audioUriMap[uriBundle] == null) {
        var res = 0
        if (activity != null) {
          res = activity.resources.getIdentifier(fileBundle, "raw", mPackageName)
        } else {
          Log.d(TAG, "getAudioUri() reactContext is null")
        }
        if (res <= 0) {
          Log.d(TAG, String.format("getAudioUri() %s.%s not found in bundle.", fileBundle, this))
          audioUriMap[uriBundle] = null
          // type = fileSysWithExt;
          getDefaultUserUri(
            uriDefault
          ) // --- if specified bundle but not found, use default directlly
        } else {
          audioUriMap[uriBundle] =
            Uri.parse("android.resource://" + mPackageName + "/" + res.toString())
          // bundleRingtoneUri = Uri.parse("android.resource://" + reactContext.getPackageName() +
          // "/" + R.raw.incallmanager_ringtone);
          // bundleRingtoneUri = Uri.parse("android.resource://" + reactContext.getPackageName() +
          // "/raw/incallmanager_ringtone");
          Log.d(TAG, "getAudioUri() using: $_type")
          audioUriMap[uriBundle]
        }
      } else {
        Log.d(TAG, "getAudioUri() using: $_type")
        audioUriMap[uriBundle]
      }
    }

    // --- Check file every time in case user deleted.
    val target = "$fileSysPath/$_type"
    val _uri = getSysFileUri(target)
    return if (_uri == null) {
      Log.d(TAG, "getAudioUri() using user default")
      getDefaultUserUri(uriDefault)
    } else {
      Log.d(TAG, "getAudioUri() using internal: $target")
      audioUriMap[uriDefault] = _uri
      _uri
    }
  }

  private fun getSysFileUri(target: String): Uri? {
    val file = File(target)
    return if (file.isFile) {
      Uri.fromFile(file)
    } else null
  }

  private fun getDefaultUserUri(type: String): Uri {
    // except ringtone, it doesn't suppose to be go here. and every android has different files
    // unlike apple;
    return if (type == "defaultRingtoneUri") {
      System.DEFAULT_RINGTONE_URI
    } else if (type == "defaultRingbackUri") {
      System.DEFAULT_RINGTONE_URI
    } else if (type == "defaultBusytoneUri") {
      System.DEFAULT_NOTIFICATION_URI // --- DEFAULT_ALARM_ALERT_URI
    } else {
      System.DEFAULT_NOTIFICATION_URI
    }
  }

  // ===== File Uri End =====
  // ===== Internal Classes Start =====
  private inner class DyteToneGenerator(private val toneCategory: Int) :
    Thread(), MyPlayerInterface {
    private var toneType = 0
    override var isMediaPlaying = false
      private set

    var customWaitTimeMs = maxWaitTimeMs
    var caller: String? = null

    override fun startPlay(data: Map<String?, Any?>?) {
      val name = data?.get("name") as String?
      caller = name
      start()
    }

    override fun stopPlay() {
      synchronized(this) {
        if (isMediaPlaying) {
          notify()
        }
        isMediaPlaying = false
      }
    }

    override fun run() {
      val toneWaitTimeMs: Int
      when (toneCategory) {
        SILENT -> {
          // toneType = ToneGenerator.TONE_CDMA_SIGNAL_OFF;
          toneType = ToneGenerator.TONE_CDMA_ANSWER
          toneWaitTimeMs = 1000
        }
        BUSY -> {
          // toneType = ToneGenerator.TONE_SUP_BUSY;
          // toneType = ToneGenerator.TONE_SUP_CONGESTION;
          // toneType = ToneGenerator.TONE_SUP_CONGESTION_ABBREV;
          // toneType = ToneGenerator.TONE_CDMA_NETWORK_BUSY;
          // toneType = ToneGenerator.TONE_CDMA_NETWORK_BUSY_ONE_SHOT;
          toneType = ToneGenerator.TONE_SUP_RADIO_NOTAVAIL
          toneWaitTimeMs = 4000
        }
        RINGBACK -> {
          // toneType = ToneGenerator.TONE_SUP_RINGTONE;
          toneType = ToneGenerator.TONE_CDMA_NETWORK_USA_RINGBACK
          toneWaitTimeMs = maxWaitTimeMs // [STOP MANUALLY]
        }
        CALLEND -> {
          toneType = ToneGenerator.TONE_PROP_PROMPT
          toneWaitTimeMs = 200 // plays when call ended
        }
        CALLWAITING -> {
          // toneType = ToneGenerator.TONE_CDMA_NETWORK_CALLWAITING;
          toneType = ToneGenerator.TONE_SUP_CALL_WAITING
          toneWaitTimeMs = maxWaitTimeMs // [STOP MANUALLY]
        }
        BEEP -> {
          // toneType = ToneGenerator.TONE_SUP_PIP;
          // toneType = ToneGenerator.TONE_CDMA_PIP;
          // toneType = ToneGenerator.TONE_SUP_RADIO_ACK;
          // toneType = ToneGenerator.TONE_PROP_BEEP;
          toneType = ToneGenerator.TONE_PROP_BEEP2
          toneWaitTimeMs = 1000 // plays when call ended
        }
        else -> {
          // --- use ToneGenerator internal type.
          Log.d(TAG, "myToneGenerator: use internal tone type: $toneCategory")
          toneType = toneCategory
          toneWaitTimeMs = customWaitTimeMs
        }
      }
      Log.d(
        TAG,
        String.format(
          "myToneGenerator: toneCategory: %d ,toneType: %d, toneWaitTimeMs: %d",
          toneCategory,
          toneType,
          toneWaitTimeMs
        )
      )
      val tg: ToneGenerator? =
        try {
          ToneGenerator(AudioManager.STREAM_VOICE_CALL, toneVolume)
        } catch (e: RuntimeException) {
          Log.d(TAG, "myToneGenerator: Exception caught while creating ToneGenerator: $e")
          null
        }
      if (tg != null) {
        synchronized(this) {
          if (!isMediaPlaying) {
            isMediaPlaying = true

            // --- make sure audio routing, or it will be wired when switch suddenly
            when (caller) {
              "mBusytone" -> {
                audioManager.mode = AudioManager.MODE_IN_COMMUNICATION
              }
              "mRingback" -> {
                audioManager.mode = AudioManager.MODE_IN_COMMUNICATION
              }
              "mRingtone" -> {
                audioManager.mode = AudioManager.MODE_RINGTONE
              }
            }
            updateAudioRoute()
            tg.startTone(toneType)
            try {
              sleep((toneWaitTimeMs + loadBufferWaitTimeMs).toLong())
            } catch (e: InterruptedException) {
              Log.d(TAG, "myToneGenerator stopped. toneType: $toneType")
            }
            tg.stopTone()
          }
          isMediaPlaying = false
          tg.release()
        }
      }
      Log.d(TAG, "MyToneGenerator(): play finished. caller=$caller")
      if (caller == "mBusytone") {
        Log.d(TAG, "MyToneGenerator(): invoke stop()")
        this@InCallManagerModule.stop()
      }
    }
  }

  private inner class DyteMediaPlayer : MediaPlayer(), MyPlayerInterface {
    override var isMediaPlaying = false
      private set

    override fun stopPlay() {
      stop()
      reset()
      release()
    }

    override fun startPlay(data: Map<String?, Any?>?) {
      try {
        setDataSource(activity!!, (data?.get("sourceUri") as Uri?)!!)
        isLooping = (data?.get("setLooping") as Boolean?)!!

        // --- the `minSdkVersion` is 21 since RN 64,
        // --- if you want to suuport api < 21, comment out `setAudioAttributes` and use
        // `setAudioStreamType((Integer) data.get("audioStream"))` instead
        setAudioAttributes(
          Builder()
            .setUsage((data?.get("audioUsage") as Int?)!!)
            .setContentType((data?.get("audioContentType") as Int?)!!)
            .build()
        )

        // -- will start at onPrepared() event
        prepareAsync()
      } catch (e: Exception) {
        Log.d(TAG, "startPlay() failed", e)
      }
    }
  }

  private fun updateAudioRoute() {
    if (!automatic) {
      return
    }
    updateAudioDeviceState()
  }
  // ===== NOTE: below functions is based on appRTC DEMO M64 ===== //
  /** Changes selection of the currently active audio device. */
  private fun setAudioDeviceInternal(device: AudioDevice) {
    Log.d(TAG, "setAudioDeviceInternal(device=$device)")
    if (!audioDevices.contains(device)) {
      Log.e(TAG, "specified audio device does not exist")
      return
    }
    when (device) {
      SPEAKER_PHONE -> setSpeakerphoneOn(true)
      EARPIECE -> setSpeakerphoneOn(false)
      WIRED_HEADSET -> setSpeakerphoneOn(false)
      BLUETOOTH -> setSpeakerphoneOn(false)
      else -> Log.e(TAG, "Invalid audio device selection")
    }
    selectedAudioDevice = device
  }

  /**
   * Changes default audio device.
   *
   * TODO(henrika): add usage of this method in the AppRTCMobile client.
   */
  fun setDefaultAudioDevice(defaultDevice: AudioDevice?) {
    when (defaultDevice) {
      SPEAKER_PHONE -> defaultAudioDevice = defaultDevice
      EARPIECE ->
        defaultAudioDevice =
          if (hasEarpiece()) {
            defaultDevice
          } else {
            SPEAKER_PHONE
          }
      else -> Log.e(TAG, "Invalid default audio device selection")
    }
    Log.d(TAG, "setDefaultAudioDevice(device=$defaultAudioDevice)")
    updateAudioDeviceState()
  }

  /** Changes selection of the currently active audio device. */
  fun selectAudioDevice(device: AudioDevice) {
    if (device != NONE && !audioDevices.contains(device)) {
      Log.e(TAG, "selectAudioDevice() Can not select $device from available $audioDevices")
      return
    }
    userSelectedAudioDevice = device
    updateAudioDeviceState()
  }

  /** Returns current set of available/selectable audio devices. */
  fun getAudioDevices(): Set<AudioDevice> {
    return Collections.unmodifiableSet(HashSet(audioDevices))
  }

  /** Helper method for unregistration of an existing receiver. */
  private fun unregisterReceiver(receiver: BroadcastReceiver?) {
    if (activity != null) {
      try {
        activity.unregisterReceiver(receiver)
      } catch (e: Exception) {
        Log.d(TAG, "unregisterReceiver() failed")
      }
    } else {
      Log.d(TAG, "unregisterReceiver() reactContext is null")
    }
  }
  /** Sets the speaker phone mode. */
  /*
  private void setSpeakerphoneOn(boolean on) {
      boolean wasOn = audioManager.isSpeakerphoneOn();
      if (wasOn == on) {
          return;
      }
      audioManager.setSpeakerphoneOn(on);
  }
  */
  /** Sets the microphone mute state. */
  /*
  private void setMicrophoneMute(boolean on) {
      boolean wasMuted = audioManager.isMicrophoneMute();
      if (wasMuted == on) {
          return;
      }
      audioManager.setMicrophoneMute(on);
  }
  */
  /** Gets the current earpiece state. */
  private fun hasEarpiece(): Boolean {
    return activity!!.packageManager.hasSystemFeature(PackageManager.FEATURE_TELEPHONY)
  }

  /**
   * Checks whether a wired headset is connected or not. This is not a valid indication that audio
   * playback is actually over the wired headset as audio routing depends on other conditions. We
   * only use it as an early indicator (during initialization) of an attached wired headset.
   */
  @Suppress("DEPRECATION")
  private fun hasWiredHeadset(): Boolean {
    return if (VERSION.SDK_INT < VERSION_CODES.M) {
      audioManager.isWiredHeadsetOn
    } else {
      val devices = audioManager.getDevices(AudioManager.GET_DEVICES_INPUTS)
      for (device in devices) {
        val type = device.type
        when (type) {
          AudioDeviceInfo.TYPE_WIRED_HEADSET -> {
            Log.d(TAG, "hasWiredHeadset: found wired headset")
            return true
          }
          AudioDeviceInfo.TYPE_USB_DEVICE -> {
            Log.d(TAG, "hasWiredHeadset: found USB audio device")
            return true
          }
          AudioDeviceInfo.TYPE_WIRED_HEADPHONES -> {
            Log.d(TAG, "hasWiredHeadset: found wired headphones")
            return true
          }
        }
      }
      false
    }
  }

  /** Updates list of possible audio devices and make new device selection. */
  fun updateAudioDeviceState() {
    Log.d(
      TAG,
      "--- updateAudioDeviceState: " +
        "wired headset=" +
        hasWiredHeadset +
        ", " +
        "BT state=" +
        bluetoothManager!!.state
    )
    Log.d(
      TAG,
      "Device status: " +
        "available=" +
        audioDevices +
        ", " +
        "selected=" +
        selectedAudioDevice +
        ", " +
        "user selected=" +
        userSelectedAudioDevice
    )

    // Check if any Bluetooth headset is connected. The internal BT state will
    // change accordingly.
    // TODO(henrika): perhaps wrap required state into BT manager.
    if (
      bluetoothManager!!.state === HEADSET_AVAILABLE ||
        bluetoothManager!!.state === HEADSET_UNAVAILABLE ||
        bluetoothManager!!.state === SCO_DISCONNECTING
    ) {
      bluetoothManager!!.updateDevice()
    }

    // Update the set of available audio devices.
    val newAudioDevices: MutableSet<AudioDevice> = HashSet()

    // always assume device has speaker phone
    newAudioDevices.add(SPEAKER_PHONE)
    if (
      bluetoothManager!!.state === SCO_CONNECTED ||
        bluetoothManager!!.state === SCO_CONNECTING ||
        bluetoothManager!!.state === HEADSET_AVAILABLE
    ) {
      newAudioDevices.add(BLUETOOTH)
    }
    if (hasWiredHeadset) {
      newAudioDevices.add(WIRED_HEADSET)
    }
    if (hasEarpiece()) {
      newAudioDevices.add(EARPIECE)
    }

    // --- check whether user selected audio device is available
    if (
      userSelectedAudioDevice != null &&
        userSelectedAudioDevice != NONE &&
        !newAudioDevices.contains(userSelectedAudioDevice)
    ) {
      userSelectedAudioDevice = NONE
    }

    // Store state which is set to true if the device list has changed.
    var audioDeviceSetUpdated = audioDevices != newAudioDevices
    // Update the existing audio device set.
    audioDevices = newAudioDevices
    var newAudioDevice = preferredAudioDevice

    // --- stop bluetooth if needed
    if (
      selectedAudioDevice == BLUETOOTH &&
        newAudioDevice != BLUETOOTH &&
        (bluetoothManager!!.state === SCO_CONNECTED || bluetoothManager!!.state === SCO_CONNECTING)
    ) {
      bluetoothManager!!.stopScoAudio()
      bluetoothManager!!.updateDevice()
    }

    // --- start bluetooth if needed
    if (
      selectedAudioDevice != BLUETOOTH &&
        newAudioDevice == BLUETOOTH &&
        bluetoothManager!!.state === HEADSET_AVAILABLE
    ) {
      // Attempt to start Bluetooth SCO audio (takes a few second to start).
      if (!bluetoothManager!!.startScoAudio()) {
        // Remove BLUETOOTH from list of available devices since SCO failed.
        audioDevices.remove(BLUETOOTH)
        audioDeviceSetUpdated = true
        if (userSelectedAudioDevice == BLUETOOTH) {
          userSelectedAudioDevice = NONE
        }
        newAudioDevice = preferredAudioDevice
      }
    }
    if (newAudioDevice == BLUETOOTH && bluetoothManager!!.state !== SCO_CONNECTED) {
      newAudioDevice = getPreferredAudioDevice(true) // --- skip bluetooth
    }

    // Switch to new device but only if there has been any changes.
    if (newAudioDevice != selectedAudioDevice || audioDeviceSetUpdated) {

      // Do the required device switch.
      setAudioDeviceInternal(newAudioDevice)
      Log.d(
        TAG,
        "New device status: " + "available=" + audioDevices + ", " + "selected=" + newAudioDevice
      )
      /*if (audioManagerEvents != null) {
          // Notify a listening client that audio device has been changed.
          audioManagerEvents.onAudioDeviceChanged(selectedAudioDevice, audioDevices);
      }*/ sendEvent("onAudioDeviceChanged", audioDeviceStatusMap)
    }
    Log.d(TAG, "--- updateAudioDeviceState done")
  }

  // --- strip the last `,`
  private val audioDeviceStatusMap: Bundle
    get() {
      val data = Bundle()
      var audioDevicesJson = "["
      for (s in audioDevices) {
        audioDevicesJson += "\"" + s.name + "\","
      }

      // --- strip the last `,`
      if (audioDevicesJson.length > 1) {
        audioDevicesJson = audioDevicesJson.substring(0, audioDevicesJson.length - 1)
      }
      audioDevicesJson += "]"
      data.putString("availableAudioDeviceList", audioDevicesJson)
      data.putString(
        "selectedAudioDevice",
        if (selectedAudioDevice == null) "" else selectedAudioDevice!!.name
      )
      return data
    }

  private val preferredAudioDevice: AudioDevice
    get() = getPreferredAudioDevice(false)

  private fun getPreferredAudioDevice(skipBluetooth: Boolean): AudioDevice {
    val newAudioDevice: AudioDevice =
      if (userSelectedAudioDevice != null && userSelectedAudioDevice != NONE) {
        userSelectedAudioDevice!!
      } else if (!skipBluetooth && audioDevices.contains(BLUETOOTH)) {
        // If a Bluetooth is connected, then it should be used as output audio
        // device. Note that it is not sufficient that a headset is available;
        // an active SCO channel must also be up and running.
        BLUETOOTH
      } else if (audioDevices.contains(WIRED_HEADSET)) {
        // If a wired headset is connected, but Bluetooth is not, then wired headset is used as
        // audio device.
        WIRED_HEADSET
      } else if (audioDevices.contains(defaultAudioDevice)) {
        defaultAudioDevice
      } else {
        SPEAKER_PHONE
      }
    return newAudioDevice
  }

  companion object {
    private const val REACT_NATIVE_MODULE_NAME = "InCallManager"
    private const val TAG = REACT_NATIVE_MODULE_NAME
    private const val ACTION_HEADSET_PLUG = AudioManager.ACTION_HEADSET_PLUG
    private const val SPEAKERPHONE_AUTO = "auto"
  }

  init {
    this.activity = activity
    this.inCallManagerEventCallbacks = inCallManagerEventCallbacks
    mWindowManager = activity.getSystemService(Context.WINDOW_SERVICE) as WindowManager
    mPowerManager = activity.getSystemService(Context.POWER_SERVICE) as PowerManager
    audioManager = activity.getSystemService(Context.AUDIO_SERVICE) as AudioManager
    audioUriMap = HashMap()
    audioUriMap["defaultRingtoneUri"] = defaultRingtoneUri
    audioUriMap["defaultRingbackUri"] = defaultRingbackUri
    audioUriMap["defaultBusytoneUri"] = defaultBusytoneUri
    audioUriMap["bundleRingtoneUri"] = bundleRingtoneUri
    audioUriMap["bundleRingbackUri"] = bundleRingbackUri
    audioUriMap["bundleBusytoneUri"] = bundleBusytoneUri
    wakeLockUtils = InCallWakeLockUtils(activity)
    proximityManager = InCallProximityManager.create(activity, this)
    bluetoothManager = create(activity, this)
    Log.d(TAG, "InCallManager initialized")
  }
}
