package io.dyte.core.host

import io.dyte.core.DyteError
import io.dyte.core.HostControlErrorCodes.Kick_All
import io.dyte.core.HostControlErrorCodes.Kick_Peer
import io.dyte.core.HostControlErrorCodes.Mute_All_Audio
import io.dyte.core.HostControlErrorCodes.Mute_All_Video
import io.dyte.core.HostControlErrorCodes.Mute_Audio
import io.dyte.core.HostControlErrorCodes.Mute_Video
import io.dyte.core.HostControlErrorCodes.Pin_Peer
import io.dyte.core.HostControlErrorCodes.Unpin_Peer
import io.dyte.core.Result
import io.dyte.core.Result.Failure
import io.dyte.core.Result.Success
import io.dyte.core.socket.IRoomNodeSocketService
import io.dyte.core.socket.events.OutboundMeetingEventType
import kotlinx.serialization.json.JsonElement
import kotlinx.serialization.json.JsonObject
import kotlinx.serialization.json.JsonPrimitive

internal class HostRoomNodeController(
  private val socketController: IRoomNodeSocketService,
) : IHostController {
  override suspend fun kickPeer(peerId: String): Result<Unit, DyteError> {
    val content = HashMap<String, JsonElement>()
    content["id"] = JsonPrimitive(peerId)
    val result =
      socketController.sendPacket(OutboundMeetingEventType.KICK_PEER, JsonObject(content))
    return evaluate(result, DyteError(Kick_Peer))
  }

  override suspend fun kickAll(): Result<Unit, DyteError> {
    val result = socketController.sendPacket(OutboundMeetingEventType.KICK_ALL, null)
    return evaluate(result, DyteError(Kick_All))
  }

  override suspend fun muteAllAudio(allowUnmute: Boolean?): Result<Unit, DyteError> {
    val content = HashMap<String, JsonElement>()
    content["allowUnMute"] = JsonPrimitive(allowUnmute)
    val result =
      socketController.sendPacket(OutboundMeetingEventType.MUTE_ALL_AUDIO, JsonObject(content))
    return evaluate(result, DyteError(Mute_All_Audio))
  }

  override suspend fun muteAllVideo(): Result<Unit, DyteError> {
    val result = socketController.sendPacket(OutboundMeetingEventType.MUTE_ALL_VIDEO, null)
    return evaluate(result, DyteError(Mute_All_Video))
  }

  override suspend fun muteAudio(peerId: String): Result<Unit, DyteError> {
    val content = HashMap<String, JsonElement>()
    content["id"] = JsonPrimitive(peerId)
    val result =
      socketController.sendPacket(OutboundMeetingEventType.MUTE_PEER_AUDIO, JsonObject(content))
    return evaluate(result, DyteError(Mute_Audio))
  }

  override suspend fun muteVideo(peerId: String): Result<Unit, DyteError> {
    val content = HashMap<String, JsonElement>()
    content["id"] = JsonPrimitive(peerId)
    val result =
      socketController.sendPacket(OutboundMeetingEventType.MUTE_PEER_VIDEO, JsonObject(content))
    return evaluate(result, DyteError(Mute_Video))
  }

  override suspend fun pinPeer(peerId: String): Result<Unit, DyteError> {
    val content = HashMap<String, JsonElement>()
    content["id"] = JsonPrimitive(peerId)
    val result =
      socketController.sendPacket(OutboundMeetingEventType.PIN_PARTICIPANT, JsonObject(content))
    return evaluate(result, DyteError(Pin_Peer))
  }

  override suspend fun unpinPeer(): Result<Unit, DyteError> {
    val content = HashMap<String, JsonElement>()
    val nullString: String? = null
    content["id"] = JsonPrimitive(nullString)
    val result =
      socketController.sendPacket(OutboundMeetingEventType.PIN_PARTICIPANT, JsonObject(content))
    return evaluate(result, DyteError(Unpin_Peer))
  }

  private fun evaluate(
    result: Result<String, DyteError>,
    localError: DyteError
  ): Result<Unit, DyteError> {
    when (result) {
      is Failure -> {
        return Result.Failure(localError)
      }
      is Success -> {
        return Result.Success(Unit)
      }
    }
  }
}
