package io.dyte.core.controllers

import io.dyte.core.events.EventEmitter
import io.dyte.core.events.InternalEvents
import io.dyte.core.feat.ParticipantFlags
import io.dyte.core.host.IHostController
import io.dyte.core.listeners.DyteSelfEventsListener
import io.dyte.core.media.IDyteSFUUtils
import io.dyte.core.models.DyteAudioDevice
import io.dyte.core.models.DyteMeetingType
import io.dyte.core.models.DyteSelfParticipant
import io.dyte.core.models.DyteVideoDevice
import io.dyte.core.models.VideoDeviceType
import io.dyte.core.models.WaitListStatus
import io.dyte.core.network.info.ParticipantInfo
import io.dyte.core.network.info.SelfPermissions
import io.dyte.core.observability.DyteLogger
import io.dyte.core.platform.IDyteMediaUtils
import io.dyte.core.platform.IDyteVideoUtils
import io.dyte.core.platform.VideoView
import io.dyte.core.socket.RoomNodeSocketService
import io.dyte.core.socket.SocketMessageEventListener
import io.dyte.core.socket.events.InboundMeetingEventType
import io.dyte.core.socket.events.OutboundMeetingEventType.MUTE_SELF_AUDIO
import io.dyte.core.socket.events.OutboundMeetingEventType.UN_MUTE_SELF_AUDIO
import io.dyte.core.socket.events.payloadmodel.InboundMeetingEvent
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketPeerMuteModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketRoomMessage
import io.dyte.core.socket.events.payloadmodel.outbound.WebSocketJoinRoomModel
import io.dyte.core.socket.socketservice.SockratesSocketService
import io.dyte.core.utils.DyteUtils.supportsAudioProduction
import io.dyte.core.utils.DyteUtils.supportsVideoProduction
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.launch
import kotlinx.serialization.json.jsonObject

internal class SelfController(
  private val permissionController: PermissionController,
  private val internalEventsEmitter: EventEmitter<InternalEvents>,
  meetingUserPeerId: String,
  private val isHiveMeeting: Boolean,
  private val mediaSoupController: MediaSoupController?,
  private val roomNodeSocket: RoomNodeSocketService,
  private val mediaUtils: IDyteMediaUtils?,
  private val participantInfo: ParticipantInfo,
  private val participantController: ParticipantController,
  private val metaController: IMetaController,
  hostController: IHostController,
  private val sockratesSocketService: SockratesSocketService,
  private val scope: CoroutineScope,
  private val videoUtils: IDyteVideoUtils?
) : EventEmitter<DyteSelfEventsListener>(), InternalEvents {

  internal var sfuUtils: IDyteSFUUtils? = null

  private val meetingType: DyteMeetingType = metaController.getMeetingType()

  val selfParticipant =
    DyteSelfParticipant(
      meetingUserPeerId,
      participantInfo.id,
      participantInfo.name,
      participantInfo.picture,
      false,
      participantInfo.clientSpecificId,
      ParticipantFlags(false, false, false),
      participantInfo.presetInfo.name,
      this,
      participantInfo,
      participantController,
      metaController,
      hostController,
      permissionController,
    )

  internal var _roomJoined: Boolean = false

  val selfPermissions: SelfPermissions
    get() = participantInfo.presetInfo.permissions

  override fun onRoomJoined(webSocketJoinRoomModel: WebSocketJoinRoomModel) {
    super.onRoomJoined(webSocketJoinRoomModel)
    if (meetingType.supportsVideoProduction) {
      val isCameraPermissionGranted = permissionController.isPermissionGrated(PermissionType.CAMERA)
      if (isCameraPermissionGranted.not()) {
        emitEvent { it.onMeetingRoomJoinedWithoutCameraPermission() }
      }
    }

    if (meetingType.supportsAudioProduction) {
      val isMicrophonePermissionGranted =
        permissionController.isPermissionGrated(PermissionType.MICROPHONE)
      if (isMicrophonePermissionGranted.not()) {
        emitEvent { it.onMeetingRoomJoinedWithoutMicPermission() }
      }
    }
  }

  override fun onWaitlistEntryAccepted() {
    emitEvent { it.onWaitListStatusUpdate(WaitListStatus.ACCEPTED) }
  }

  override fun onWaitlistEntryRejected() {
    emitEvent { it.onWaitListStatusUpdate(WaitListStatus.REJECTED) }
  }

  init {
    roomNodeSocket.addMessageEventListener(
      InboundMeetingEventType.WEB_SOCKET_PEER_MUTED,
      object : SocketMessageEventListener {
        override suspend fun onMessageEvent(event: InboundMeetingEvent) {
          if ((event.payload as WebSocketPeerMuteModel).peerId == selfParticipant.id) {
            selfParticipant._audioEnabled = false
            emitEvent { it.onAudioUpdate(false) }
            participantController.onAudioUpdate(selfParticipant)
          }
        }
      }
    )

    roomNodeSocket.addMessageEventListener(
      InboundMeetingEventType.WEB_SOCKET_PEER_UNMUTED,
      object : SocketMessageEventListener {
        override suspend fun onMessageEvent(event: InboundMeetingEvent) {
          if ((event.payload as WebSocketPeerMuteModel).peerId == selfParticipant.id) {
            selfParticipant._audioEnabled = true
            emitEvent { it.onAudioUpdate(true) }
            participantController.onAudioUpdate(selfParticipant)
          }
        }
      }
    )

    roomNodeSocket.addMessageEventListener(
      InboundMeetingEventType.WEB_SOCKET_PRODUCER_CONNECT,
      object : SocketMessageEventListener {
        override suspend fun onMessageEvent(event: InboundMeetingEvent) {
          selfParticipant._videoEnabled = true
          emitEvent { it.onVideoUpdate(true) }
          participantController.onVideoUpdate(selfParticipant)
        }
      }
    )

    roomNodeSocket.addMessageEventListener(
      InboundMeetingEventType.WEB_SOCKET_PRODUCER_CLOSED,
      object : SocketMessageEventListener {
        override suspend fun onMessageEvent(event: InboundMeetingEvent) {
          mediaSoupController?.destroyCurrentVideo()
          selfParticipant._videoEnabled = false
          emitEvent { it.onVideoUpdate(false) }
          participantController.onVideoUpdate(selfParticipant)
        }
      }
    )

    roomNodeSocket.addMessageEventListener(
      InboundMeetingEventType.WEB_SOCKET_MUTE_ALL_VIDEO,
      object : SocketMessageEventListener {
        override suspend fun onMessageEvent(event: InboundMeetingEvent) {
          disableVideo()
        }
      }
    )

    roomNodeSocket.addMessageEventListener(
      InboundMeetingEventType.WEB_SOCKET_DISABLE_VIDEO,
      object : SocketMessageEventListener {
        override suspend fun onMessageEvent(event: InboundMeetingEvent) {
          disableVideo()
        }
      }
    )

    roomNodeSocket.addMessageEventListener(
      InboundMeetingEventType.WEB_SOCKET_MUTE_ALL_AUDIO,
      object : SocketMessageEventListener {
        override suspend fun onMessageEvent(event: InboundMeetingEvent) {
          disableAudio()
        }
      }
    )

    roomNodeSocket.addMessageEventListener(
      InboundMeetingEventType.WEB_SOCKET_DISABLE_AUDIO,
      object : SocketMessageEventListener {
        override suspend fun onMessageEvent(event: InboundMeetingEvent) {
          disableAudio()
        }
      }
    )

    /*
     * TODO: Include timestamp in the emitted roomMessage and fix formatting.
     *  Reference: https://github.com/dyte-in/web-core/blob/df7c518b091d1cde51f959fed60f21cfc65f5ead/src/controllers/ParticipantController.ts#L986-L1003
     * */
    roomNodeSocket.addMessageEventListener(
      InboundMeetingEventType.WEB_SOCKET_ROOM_MESSAGE,
      object : SocketMessageEventListener {
        override suspend fun onMessageEvent(event: InboundMeetingEvent) {
          val roomMessage = (event.payload as WebSocketRoomMessage).payload.jsonObject
          val roomMessageType = roomMessage["type"].toString()
          val roomMessagePayload = roomMessage["payload"]?.jsonObject
          val parsedPayload = hashMapOf<String, Any>()
          roomMessagePayload?.entries?.forEach { parsedPayload[it.key] = it.value }
          emitEvent { it.onRoomMessage(roomMessageType, parsedPayload) }
        }
      }
    )

    roomNodeSocket.addMessageEventListener(
      InboundMeetingEventType.WEB_SOCKET_KICKED,
      object : SocketMessageEventListener {
        override suspend fun onMessageEvent(event: InboundMeetingEvent) {
          onRemovedFromMeeting()
        }
      }
    )
  }

  fun onRemovedFromMeeting() {
    DyteLogger.info("self::removed from meeting")
    internalEventsEmitter.emitEvent { it.disconnectMedia() }
    roomNodeSocket.disconnect()
    scope.launch { sockratesSocketService.disconnect() }
    emitEvent { it.onRemovedFromMeeting() }
  }

  fun getSelf(): DyteSelfParticipant {
    return selfParticipant
  }

  suspend fun disableAudio() {
    DyteLogger.info("DyteSelf.disableAudio")
    if (!meetingType.supportsAudioProduction) {
      DyteLogger.info("SelfController::disableAudio::cannot disable audio in a $meetingType")
      return
    }

    if (!selfParticipant.audioEnabled) {
      return
    }

    sfuUtils?.muteSelfAudio()

    if (_roomJoined) {
      if (!isHiveMeeting) {
        roomNodeSocket.sendMessage(MUTE_SELF_AUDIO, null)
      }
    } else {
      selfParticipant._audioEnabled = false
      emitEvent { it.onAudioUpdate(false) }
      participantController.onAudioUpdate(selfParticipant)
    }
  }

  suspend fun enableAudio() {
    DyteLogger.info("DyteSelf.enableAudio")
    if (!meetingType.supportsAudioProduction) {
      DyteLogger.info("SelfController::enableAudio::cannot enable audio in a $meetingType")
      return
    }

    if (selfParticipant.audioEnabled) {
      DyteLogger.info("DyteSelf.enableAudio::already enabled")
      return
    }

    if (!canPublishAudio()) {
      DyteLogger.info("DyteSelf.enableAudio::audio not allowed")
      return
    }

    if (_roomJoined) {
      sfuUtils?.unmuteSelfAudio()
      if (!isHiveMeeting) roomNodeSocket.sendMessage(UN_MUTE_SELF_AUDIO, null)
    } else {
      selfParticipant._audioEnabled = true
      emitEvent { it.onAudioUpdate(true) }
      participantController.onAudioUpdate(selfParticipant)
    }
  }

  fun canPublishAudio(): Boolean {
    return selfPermissions.canPublishAudio(selfParticipant.stageStatus)
  }

  fun canPublishVideo(): Boolean {
    return selfPermissions.canPublishVideo(selfParticipant.stageStatus)
  }

  suspend fun disableVideo() {
    DyteLogger.info("DyteSelf.disableVideo")
    if (!meetingType.supportsVideoProduction) {
      DyteLogger.info("SelfController::disableVideo::cannot disable video in a $meetingType")
      return
    }

    if (!selfParticipant.videoEnabled) {
      return
    }
    sfuUtils?.muteSelfVideo()
    selfParticipant._videoEnabled = false
    selfParticipant._videoTrack = null
    emitEvent { it.onVideoUpdate(false) }
    participantController.onVideoUpdate(selfParticipant)
  }

  fun getSelfPreview(): VideoView {
    if (!meetingType.supportsVideoProduction) {
      DyteLogger.warn(
        "SelfController::getSelfPreview::cannot create selfPreview in a $meetingType meeting"
      )
      error("SelfPreview not available in a $meetingType meeting")
    }

    return videoUtils?.getSelfPreview()
      ?: kotlin.run {
        DyteLogger.error("SelfController::getSelfPreview::videoUtils is null")
        error("Failed to get SelfPreview")
      }
  }

  suspend fun enableVideo() {
    DyteLogger.info("DyteSelf.enableVideo")
    if (!meetingType.supportsVideoProduction) {
      DyteLogger.info("SelfController::enableVideo::cannot enable video in a $meetingType")
      return
    }

    if (selfParticipant.videoEnabled) {
      DyteLogger.info("DyteSelf.enableVideo::already enabled")
      return
    }

    if (!canPublishVideo()) {
      DyteLogger.info("DyteSelf.enableVideo::not permitted")
      return
    }

    selfParticipant._videoEnabled = true
    sfuUtils?.unmuteSelfVideo()
    mediaUtils?.resumeVideo()
    emitEvent { it.onVideoUpdate(true) }
    participantController.onVideoUpdate(selfParticipant)
  }

  fun setDevice(dyteAudioDevice: DyteAudioDevice) {
    DyteLogger.info("DyteSelf.setDevice.audio.${dyteAudioDevice.type.displayName}")
    mediaUtils?.setAudioDevice(dyteAudioDevice)
  }

  fun getAudioDevices(): List<DyteAudioDevice> {
    if (!meetingType.supportsAudioProduction) {
      DyteLogger.warn(
        "SelfController::getAudioDevices::audio devices not available in a $meetingType meeting"
      )
      error("AudioDevices not available in a $meetingType meeting")
    }

    return mediaUtils?.getAudioDevices()
      ?: kotlin.run {
        DyteLogger.error("SelfController::getAudioDevices::mediaUtils is null")
        error("Failed to get AudioDevices")
      }
  }

  fun getSelectedAudioDevice(): DyteAudioDevice? {
    if (!meetingType.supportsAudioProduction) {
      DyteLogger.warn(
        "SelfController::getSelectedAudioDevice::audio devices not available in a $meetingType meeting"
      )
      return null
    }

    return mediaUtils?.getSelectedAudioDevice()
  }

  fun getVideoDevices(): List<DyteVideoDevice> {
    if (!meetingType.supportsVideoProduction) {
      DyteLogger.error(
        "SelfController::getVideoDevices::video devices not available in a $meetingType meeting"
      )
      error("Video devices not available in a $meetingType meeting")
    }

    return mediaUtils?.getVideoDevices()
      ?: kotlin.run {
        DyteLogger.error("SelfController::getVideoDevices::mediaUtils is null")
        error("Failed to get Video devices")
      }
  }

  fun setDevice(dyteVideoDevice: DyteVideoDevice) {
    if (!canPublishVideo()) {
      DyteLogger.info("DyteSelf.setDevice::not permitted")
      return
    }

    DyteLogger.info("DyteSelf.setDevice.video.${dyteVideoDevice.type.displayName}")
    mediaUtils?.setVideoDevice(dyteVideoDevice)
    emitEvent {
      it.onVideoDeviceChanged(
        getSelectedVideoDevice() ?: DyteVideoDevice("", VideoDeviceType.UNKNOWN)
      )
    }
  }

  fun getSelectedVideoDevice(): DyteVideoDevice? {
    if (!meetingType.supportsVideoProduction) {
      DyteLogger.warn(
        "SelfController::getSelectedVideoDevice::video devices not available in a $meetingType meeting"
      )
      return null
    }

    return mediaUtils?.getSelectedVideoDevice()
  }

  override fun emitEvent(event: (listener: DyteSelfEventsListener) -> Unit) {
    super.emitEvent(event)
    super.emitEvent { it.onUpdate(selfParticipant) }
  }
}
