package io.dyte.core.controllers

import io.dyte.core.controllers.DyteEventType.OnLiveStreamEnded
import io.dyte.core.controllers.DyteEventType.OnLiveStreamEnding
import io.dyte.core.controllers.DyteEventType.OnLiveStreamErrored
import io.dyte.core.controllers.DyteEventType.OnLiveStreamStarted
import io.dyte.core.controllers.DyteEventType.OnLiveStreamStarting
import io.dyte.core.controllers.DyteEventType.OnMeetingRoomInitCompleted
import io.dyte.core.controllers.DyteEventType.OnMeetingRoomInitFailed
import io.dyte.core.controllers.DyteEventType.OnMeetingRoomInitStarted
import io.dyte.core.controllers.DyteEventType.OnMeetingRoomJoinFailed
import io.dyte.core.controllers.DyteEventType.OnMeetingRoomJoinStarted
import io.dyte.core.controllers.DyteEventType.OnMeetingRoomJoined
import io.dyte.core.controllers.DyteEventType.OnMeetingRoomLeave
import io.dyte.core.controllers.DyteEventType.OnMeetingRoomLeaveStarted
import io.dyte.core.controllers.DyteEventType.OnMeetingStateChanged
import io.dyte.core.controllers.DyteEventType.OnMetaUpdate
import io.dyte.core.controllers.DyteEventType.OnPeerUpdate
import io.dyte.core.controllers.DyteEventType.OnPluginsUpdates
import io.dyte.core.controllers.DyteEventType.OnScreenShareUpdate
import io.dyte.core.controllers.DyteEventType.OnSelfAudioUpdate
import io.dyte.core.controllers.DyteEventType.OnSelfVideoUpdate
import io.dyte.core.controllers.DyteEventType.OnViewerCountUpdated
import io.dyte.core.feat.DytePlugin
import io.dyte.core.listeners.DyteDataUpdateListener
import io.dyte.core.listeners.DyteLiveStreamEventsListener
import io.dyte.core.listeners.DyteMeetingRoomEventsListener
import io.dyte.core.listeners.DyteParticipantUpdateListener
import io.dyte.core.listeners.DyteWaitlistEventsListener
import io.dyte.core.models.DyteJoinedMeetingParticipant
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext

internal class EventController(controllerContainer: IControllerContainer) :
  BaseController(controllerContainer), IEventController {
  private val roomEventsListeners = arrayListOf<DyteMeetingRoomEventsListener>()
  private val liveStreamEventsListeners = arrayListOf<DyteLiveStreamEventsListener>()
  private val waitlistEventsListeners = arrayListOf<DyteWaitlistEventsListener>()
  private val dataUpdateListeners = arrayListOf<DyteDataUpdateListener>()

  private val participantUpdateListeners =
    hashMapOf<String, ArrayList<DyteParticipantUpdateListener>>()

  override fun triggerEvent(eventType: DyteEventType) {
    serialScope.launch { withContext(Dispatchers.Main) { triggerEventInternal(eventType) } }
  }

  private fun triggerEventInternal(eventType: DyteEventType) {
    when (eventType) {
      OnMeetingRoomInitStarted -> {
        ArrayList(roomEventsListeners).forEach { roomEventsListener ->
          roomEventsListener.onMeetingInitStarted()
        }
      }
      OnMeetingRoomInitCompleted -> {
        ArrayList(dataUpdateListeners).forEach { dataUpdateListener ->
          dataUpdateListener.onSelfPermissionsUpdate(
            controllerContainer.selfController.getSelf().permissions
          )
        }
        ArrayList(dataUpdateListeners).forEach { dataUpdateListener ->
          dataUpdateListener.onPluginsUpdates(controllerContainer.pluginsController.activePlugins)
        }
        ArrayList(roomEventsListeners).forEach { roomEventsListener ->
          roomEventsListener.onMeetingInitCompleted()
        }
      }
      is OnMeetingRoomInitFailed -> {
        ArrayList(roomEventsListeners).forEach { roomEventsListener ->
          roomEventsListener.onMeetingInitFailed(eventType.exception)
        }
      }
      OnMeetingRoomLeaveStarted -> {
        ArrayList(roomEventsListeners).forEach { roomEventsListener ->
          roomEventsListener.onMeetingRoomLeaveStarted()
        }
      }
      OnMeetingRoomLeave -> {
        ArrayList(roomEventsListeners).forEach { roomEventsListener ->
          roomEventsListener.onMeetingRoomLeaveCompleted()
        }
        dispose()
      }
      is OnMeetingStateChanged -> {
        controllerContainer.connectionController.onSocketStateChanged(eventType.state)
      }
      OnMeetingRoomJoinStarted -> {
        ArrayList(roomEventsListeners).forEach { roomEventsListener ->
          roomEventsListener.onMeetingRoomJoinStarted()
        }
      }
      is OnMeetingRoomJoined -> {
        ArrayList(roomEventsListeners).forEach { roomEventsListener ->
          roomEventsListener.onMeetingRoomJoinCompleted()
        }
        ArrayList(dataUpdateListeners).forEach { dataUpdateListener ->
          dataUpdateListener.onSelfPermissionsUpdate(
            controllerContainer.selfController.getSelf().permissions
          )
        }
      }
      is OnMeetingRoomJoinFailed -> {
        ArrayList(roomEventsListeners).forEach { roomEventsListener ->
          roomEventsListener.onMeetingRoomJoinFailed(eventType.exception)
        }
      }
      is OnSelfAudioUpdate -> {
        val updateListeners =
          participantUpdateListeners[controllerContainer.selfController.getSelf().id]
        updateListeners?.let {
          ArrayList(updateListeners).forEach { updateListener ->
            updateListener.onAudioUpdate(controllerContainer.selfController.getSelf().audioEnabled)
            updateListener.onUpdate(controllerContainer.selfController.getSelf())
          }
        }
      }
      is OnSelfVideoUpdate -> {
        val updateListeners =
          participantUpdateListeners[controllerContainer.selfController.getSelf().id]
        updateListeners?.let {
          ArrayList(updateListeners).forEach { updateListener ->
            updateListener.onVideoUpdate(controllerContainer.selfController.getSelf().videoEnabled)
            updateListener.onUpdate(controllerContainer.selfController.getSelf())
          }
        }
      }
      is OnPeerUpdate -> {
        val updateListeners = participantUpdateListeners[eventType.participant.id]
        updateListeners?.let {
          ArrayList(updateListeners).forEach { updateListener ->
            updateListener.onUpdate(eventType.participant)
          }
        }
      }
      OnLiveStreamStarting -> {
        ArrayList(liveStreamEventsListeners).forEach { liveStreamEventsListener ->
          liveStreamEventsListener.onLiveStreamStarting()
        }
      }
      OnLiveStreamStarted -> {
        ArrayList(liveStreamEventsListeners).forEach { liveStreamEventsListener ->
          liveStreamEventsListener.onLiveStreamStateUpdate(
            controllerContainer.liveStreamController.livestreamData
          )
        }
        ArrayList(liveStreamEventsListeners).forEach { liveStreamEventsListener ->
          liveStreamEventsListener.onLiveStreamStarted()
        }
      }
      OnLiveStreamEnding -> {
        ArrayList(liveStreamEventsListeners).forEach { liveStreamEventsListener ->
          liveStreamEventsListener.onLiveStreamEnding()
        }
      }
      OnLiveStreamEnded -> {
        ArrayList(liveStreamEventsListeners).forEach { liveStreamEventsListener ->
          liveStreamEventsListener.onLiveStreamStateUpdate(
            controllerContainer.liveStreamController.livestreamData
          )
        }
        ArrayList(liveStreamEventsListeners).forEach { liveStreamEventsListener ->
          liveStreamEventsListener.onLiveStreamEnded()
        }
      }
      OnLiveStreamErrored -> {
        ArrayList(liveStreamEventsListeners).forEach { liveStreamEventsListener ->
          liveStreamEventsListener.onLiveStreamStateUpdate(
            controllerContainer.liveStreamController.livestreamData
          )
        }

        ArrayList(liveStreamEventsListeners).forEach { liveStreamEventsListener ->
          liveStreamEventsListener.onLiveStreamErrored()
        }
      }
      is OnViewerCountUpdated -> {
        ArrayList(liveStreamEventsListeners).forEach { liveStreamEventsListener ->
          liveStreamEventsListener.onViewerCountUpdated(eventType.count)
        }
      }
      OnMetaUpdate -> {
        ArrayList(dataUpdateListeners).forEach { dataUpdateListener ->
          dataUpdateListener.onMetaUpdate(
            controllerContainer.metaController.getRoomName(),
            controllerContainer.metaController.getMeetingTitle(),
            controllerContainer.metaController.getMeetingStatedTimestamp(),
            controllerContainer.metaController.getMeetingType().name
          )
        }
      }
      is OnScreenShareUpdate -> {
        ArrayList(dataUpdateListeners).forEach { dataUpdateListener ->
          dataUpdateListener.onScreenShareUpdate(eventType.screenShares)
        }
      }
      is OnPluginsUpdates -> {
        dataUpdateListeners.forEach { it.onPluginsUpdates(eventType.activePlugins) }
      }
      else -> {}
    }
  }

  override fun addRoomEventListener(roomEventsListener: DyteMeetingRoomEventsListener) {
    this.roomEventsListeners.add(roomEventsListener)
  }

  override fun removeRoomEventListener(roomEventsListener: DyteMeetingRoomEventsListener) {
    if (this.roomEventsListeners.contains(roomEventsListener)) {
      this.roomEventsListeners.remove(roomEventsListener)
    }
  }

  override fun addLiveStreamEventListener(liveStreamEventsListener: DyteLiveStreamEventsListener) {
    this.liveStreamEventsListeners.add(liveStreamEventsListener)
  }

  override fun removeLiveStreamEventListener(
    liveStreamEventsListener: DyteLiveStreamEventsListener
  ) {
    if (this.liveStreamEventsListeners.contains(liveStreamEventsListener)) {
      this.liveStreamEventsListeners.remove(liveStreamEventsListener)
    }
  }

  override fun addWaitlistEventListener(waitlistEventsListener: DyteWaitlistEventsListener) {
    if (this.waitlistEventsListeners.contains(waitlistEventsListener)) {
      return
    }

    this.waitlistEventsListeners.add(waitlistEventsListener)
  }

  override fun removeWaitlistEventListener(waitlistEventsListener: DyteWaitlistEventsListener) {
    if (this.waitlistEventsListeners.contains(waitlistEventsListener)) {
      waitlistEventsListeners.remove(waitlistEventsListener)
    }
  }

  override fun addDataUpdateListener(dataUpdateListener: DyteDataUpdateListener) {
    dataUpdateListeners.add(dataUpdateListener)
  }

  override fun removeDataUpdateListener(dataUpdateListener: DyteDataUpdateListener) {
    dataUpdateListeners.remove(dataUpdateListener)
  }

  override fun dispose() {
    this.roomEventsListeners.clear()
    this.participantUpdateListeners.clear()
    this.dataUpdateListeners.clear()
  }
}

interface IEventController {

  fun addRoomEventListener(roomEventsListener: DyteMeetingRoomEventsListener)

  fun removeRoomEventListener(roomEventsListener: DyteMeetingRoomEventsListener)

  fun addLiveStreamEventListener(liveStreamEventsListener: DyteLiveStreamEventsListener)

  fun removeLiveStreamEventListener(liveStreamEventsListener: DyteLiveStreamEventsListener)

  fun addWaitlistEventListener(waitlistEventsListener: DyteWaitlistEventsListener)

  fun removeWaitlistEventListener(waitlistEventsListener: DyteWaitlistEventsListener)

  fun addDataUpdateListener(dataUpdateListener: DyteDataUpdateListener)

  fun removeDataUpdateListener(dataUpdateListener: DyteDataUpdateListener)

  fun dispose()

  fun triggerEvent(eventType: DyteEventType)
}

sealed class DyteEventType {
  object OnMeetingRoomInitStarted : DyteEventType()

  object OnMeetingRoomInitCompleted : DyteEventType()

  class OnMeetingRoomInitFailed(val exception: Exception) : DyteEventType()

  object OnMeetingRoomLeave : DyteEventType()

  object OnMeetingRoomLeaveStarted : DyteEventType()

  object OnMeetingRoomJoinStarted : DyteEventType()

  class OnMeetingRoomJoined(val participant: DyteJoinedMeetingParticipant) : DyteEventType()

  class OnMeetingRoomJoinFailed(val exception: Exception) : DyteEventType()

  class OnMeetingStateChanged(val state: String, val isReconnected: Boolean = false) :
    DyteEventType()

  class OnPeerUpdate(val participant: DyteJoinedMeetingParticipant) : DyteEventType()

  object OnSelfAudioUpdate : DyteEventType()

  object OnSelfVideoUpdate : DyteEventType()

  object OnAudioDevicesChanged : DyteEventType()

  object OnLiveStreamStarting : DyteEventType()

  object OnLiveStreamStarted : DyteEventType()

  object OnLiveStreamEnding : DyteEventType()

  object OnLiveStreamEnded : DyteEventType()

  object OnLiveStreamErrored : DyteEventType()

  class OnViewerCountUpdated(val count: Int) : DyteEventType()

  object OnMetaUpdate : DyteEventType()

  class OnRoomMessage(val type: String, val payload: Map<String, *>) : DyteEventType()

  class OnScreenShareUpdate(val screenShares: List<DyteJoinedMeetingParticipant>) : DyteEventType()

  class OnPluginsUpdates(val activePlugins: List<DytePlugin>) : DyteEventType()
}
