package io.dyte.core.plugins.socketservice

import kotlinx.coroutines.flow.Flow
import kotlinx.serialization.json.JsonElement

internal interface PluginsSocketHandler {
  suspend fun addPlugin(pluginId: String, staggered: Boolean)

  suspend fun removePlugin(pluginId: String)

  suspend fun getActivePlugins(): List<ActivePlugin>

  suspend fun sendCustomPluginEventToRoom(pluginId: String, data: JsonElement, messageId: String?)

  suspend fun sendCustomPluginEventToPeers(
    pluginId: String,
    peerIds: List<String>,
    data: JsonElement,
    messageId: String?
  )

  suspend fun enablePluginForRoom(pluginId: String, messageId: String?)

  suspend fun disablePluginForRoom(pluginId: String, messageId: String?)

  suspend fun enablePluginForPeers(pluginId: String, peerIds: List<String>, messageId: String?)

  suspend fun disablePluginForPeers(pluginId: String, peerIds: List<String>, messageId: String?)

  suspend fun storeInsertKeys(
    pluginId: String,
    store: String,
    insertKeys: List<PluginStoreKeyEntry>,
    messageId: String?
  )

  suspend fun storeGetKeys(
    pluginId: String,
    store: String,
    getKeys: List<String>,
    messageId: String?
  )

  suspend fun storeDeleteKeys(
    pluginId: String,
    store: String,
    deleteKeys: List<String>,
    messageId: String?
  )

  suspend fun deleteStore(pluginId: String, store: String, messageId: String?)

  fun listenToPluginSocketMessages(): Flow<PluginSocketMessage>

  data class PluginStoreKeyEntry(val key: String, val payload: JsonElement?)

  data class ActivePlugin(val id: String, val enabledBy: String)
}

internal sealed class PluginSocketMessage(val event: Int, open val id: String? = null) {
  internal data class AddPlugin(val pluginId: String, val enabledBy: String) :
    PluginSocketMessage(event = PluginsSocketEvent.ADD_PLUGIN.id)

  internal data class RemovePlugin(val pluginId: String, val disabledBy: String) :
    PluginSocketMessage(event = PluginsSocketEvent.REMOVE_PLUGIN.id)

  internal data class EnablePluginForPeers(
    override val id: String? = null,
    val pluginId: String,
    val enabledBy: String
  ) : PluginSocketMessage(event = PluginsSocketEvent.ENABLE_PLUGIN_FOR_PEERS.id, id = id)

  internal data class EnablePluginForRoom(
    override val id: String? = null,
    val pluginId: String,
    val enabledBy: String
  ) : PluginSocketMessage(event = PluginsSocketEvent.ENABLE_PLUGIN_FOR_ROOM.id, id = id)

  internal data class DisablePluginForPeers(
    override val id: String? = null,
    val pluginId: String,
    val disabledBy: String
  ) : PluginSocketMessage(event = PluginsSocketEvent.DISABLE_PLUGIN_FOR_PEERS.id, id = id)

  internal data class DisablePluginForRoom(
    override val id: String? = null,
    val pluginId: String,
    val disabledBy: String
  ) : PluginSocketMessage(event = PluginsSocketEvent.DISABLE_PLUGIN_FOR_ROOM.id, id = id)

  internal data class CustomPluginEventToPeers(
    override val id: String? = null,
    val pluginId: String,
    val pluginData: JsonElement
  ) : PluginSocketMessage(event = PluginsSocketEvent.CUSTOM_PLUGIN_EVENT_TO_PEERS.id, id = id)

  internal data class CustomPluginEventToRoom(
    override val id: String? = null,
    val pluginId: String,
    val pluginData: JsonElement
  ) : PluginSocketMessage(event = PluginsSocketEvent.CUSTOM_PLUGIN_EVENT_TO_ROOM.id, id = id)

  internal data class StoreInsertKeys(
    override val id: String? = null,
    val pluginStore: PluginStore
  ) : PluginSocketMessage(event = PluginsSocketEvent.STORE_INSERT_KEYS.id, id = id)

  internal data class StoreGetKeys(override val id: String? = null, val pluginStore: PluginStore) :
    PluginSocketMessage(event = PluginsSocketEvent.STORE_GET_KEYS.id, id = id)

  internal data class StoreDeleteKeys(
    override val id: String? = null,
    val pluginStore: PluginStore
  ) : PluginSocketMessage(event = PluginsSocketEvent.STORE_DELETE_KEYS.id, id = id)

  internal data class StoreDelete(
    override val id: String? = null,
    val pluginId: String,
    val storeName: String
  ) : PluginSocketMessage(event = PluginsSocketEvent.STORE_DELETE.id, id = id)
}

internal data class PluginStore(
  val pluginId: String,
  val storeName: String,
  val storeItems: List<PluginStoreItem>
) {
  data class PluginStoreItem(
    val timestamp: String,
    val peerId: String,
    val key: String,
    val payload: JsonElement
  )
}
