package io.dyte.core.controllers.chat

import io.dyte.core.feat.DyteChatMessage
import io.dyte.core.feat.DyteFileMessage
import io.dyte.core.feat.DyteImageMessage
import io.dyte.core.feat.DyteTextMessage
import io.dyte.core.network.PresignedUrlApiService
import io.dyte.core.observability.DyteLogger
import io.dyte.core.platform.IDytePlatformUtilsProvider
import io.dyte.core.socket.socketservice.SocketServiceEventListener
import io.dyte.core.socket.socketservice.SocketServiceUtils
import io.dyte.core.socket.socketservice.SockratesSocketService
import kotlinx.serialization.encodeToString
import kotlinx.serialization.json.Json
import kotlinx.serialization.json.jsonObject
import kotlinx.serialization.json.jsonPrimitive
import kotlinx.serialization.json.long
import socket.chat.ChatMessage
import socket.chat.GetChatMessagesResponse
import socket.chat.SendChatMessageToRoomRequest
import socket.chat.SendChatMessageToRoomResponse

internal class ChatSocketServiceController(
  private val platformUtilsProvider: IDytePlatformUtilsProvider,
  private val presignedUrlApiService: PresignedUrlApiService,
  private val canSendChatText: Boolean,
  private val canSendChatFile: Boolean,
  private val socketService: SockratesSocketService
) :
  BaseChatController(
    platformUtilsProvider,
    presignedUrlApiService,
    canSendChatText,
    canSendChatFile,
  ) {
  private val roomMessageEventListener =
    object : SocketServiceEventListener {
      override fun onEvent(event: Int, eventId: String?, payload: ByteArray?) {
        payload?.let { onSocketServiceMessageToRoom(payload) }
      }
    }

  init {
    setupEvents()
  }

  override suspend fun getChatMessages(): List<DyteChatMessage> {
    val response =
      socketService.requestResponse(event = ChatSocketEvent.GET_MESSAGES.id, payload = null)

    response?.let {
      val getChatMessagesResponse = GetChatMessagesResponse.ADAPTER.decode(response)
      val socketChatMessages = mutableListOf<DyteChatMessage>()
      for (socketServiceChatMessage in getChatMessagesResponse.messages) {
        try {
          val socketChatMessage = fromSocketServiceChatMessage(socketServiceChatMessage)
          socketChatMessages.add(socketChatMessage)
        } catch (e: Exception) {
          DyteLogger.error("ChatSocketHandler::getChatMessages::error", e)
        }
      }
      return socketChatMessages.toList()
    }
    return emptyList()
  }

  override suspend fun sendTextMessageSocket(message: String) {
    sendMessageToRoom(MessageType.TEXT.type, message)
  }

  override suspend fun sendImageMessageSocket(path: String) {
    sendMessageToRoom(MessageType.IMAGE.type, path)
  }

  private fun fromSocketServiceChatMessage(socketServerChatMessage: ChatMessage): DyteChatMessage {
    when (socketServerChatMessage.payload_type) {
      0 -> {
        return DyteTextMessage(
          message = socketServerChatMessage.payload,
          userId = socketServerChatMessage.user_id,
          displayName = socketServerChatMessage.display_name,
          time = parseTime(socketServerChatMessage.created_at),
          pluginId = null,
          read = false,
        )
      }
      1 -> {
        return DyteImageMessage(
          link = socketServerChatMessage.payload,
          userId = socketServerChatMessage.user_id,
          displayName = socketServerChatMessage.display_name,
          time = parseTime(socketServerChatMessage.created_at),
          pluginId = null,
          read = false,
        )
      }
      2 -> {
        val fileMessagePayload = Json.parseToJsonElement(socketServerChatMessage.payload).jsonObject

        return DyteFileMessage(
          name = fileMessagePayload.getValue("name").jsonPrimitive.content,
          size = fileMessagePayload.getValue("size").jsonPrimitive.long,
          link = fileMessagePayload.getValue("link").jsonPrimitive.content,
          userId = socketServerChatMessage.user_id,
          displayName = socketServerChatMessage.display_name,
          time = parseTime(socketServerChatMessage.created_at),
          pluginId = null,
          read = false,
        )
      }
      else -> {
        throw UnsupportedOperationException(
          "Message type ${socketServerChatMessage.payload_type} not supported"
        )
      }
    }
  }

  override suspend fun sendFileMessageSocket(fileUrl: String, fileName: String, fileSize: Long) {
    val message = SocketServiceFileMessage(fileUrl, fileName, fileSize)
    sendMessageToRoom(MessageType.FILE.type, Json.encodeToString(message))
  }

  private suspend fun sendMessageToRoom(messageType: Int, message: String) {
    val sendChatMessageToRoomRequest =
      SendChatMessageToRoomRequest(payload_type = messageType, payload = message)
    socketService.send(
      event = ChatSocketEvent.SEND_MESSAGE_TO_ROOM.id,
      payload = SendChatMessageToRoomRequest.ADAPTER.encode(sendChatMessageToRoomRequest)
    )
  }

  private fun unSubscribeToSocketServiceChatEvents() {
    unSubscribeToSendMessageToRoomEvent()
  }

  private fun setupEvents() {
    socketService.subscribe(
      event = ChatSocketEvent.SEND_MESSAGE_TO_ROOM.id,
      listener = roomMessageEventListener
    )
  }

  private fun unSubscribeToSendMessageToRoomEvent() {
    socketService.unsubscribe(
      event = ChatSocketEvent.SEND_MESSAGE_TO_ROOM.id,
      listener = roomMessageEventListener
    )
  }

  private fun onSocketServiceMessageToRoom(payload: ByteArray) {
    try {
      val sendChatMessageToRoomResponse = SendChatMessageToRoomResponse.ADAPTER.decode(payload)
      val message = sendChatMessageToRoomResponse.message ?: return
      val convertedMessage = fromSocketServiceChatMessage(message)
      handleMessage(convertedMessage)
    } catch (e: Exception) {
      DyteLogger.error("ChatSocketHandler::getChatMessagesSocketService::error", e)
    }
  }

  @kotlinx.serialization.Serializable
  private data class SocketServiceFileMessage(val link: String, val name: String, val size: Long)
}

internal enum class ChatSocketEvent(slug: Int) {
  GET_MESSAGES(0),
  SEND_MESSAGE_TO_ROOM(1),
  SEND_MESSAGE_TO_PEERS(2),
  EDIT_MESSAGE(3),
  DELETE_MESSAGE(4);

  private val base: Int = 1
  val id: Int = SocketServiceUtils.generateSocketEventId(base, slug)
}
