package io.dyte.core.models

import io.dyte.core.controllers.ParticipantController
import io.dyte.core.controllers.WaitlistController
import io.dyte.core.feat.DyteMeetingParticipant
import io.dyte.core.feat.DyteWaitlistedParticipant
import io.dyte.core.host.IHostController
import kotlinx.coroutines.*

/**
 * Dyte meeting participants
 *
 * @property waitlisted List of participant objects that are waiting to be let in
 * @property joined List of participant objects that are in the meeting right now
 * @property active List of participant objects that should be rendered on the screen right now
 * @property pinned List of participant objects that are marked pinned
 */
typealias DyteParticipants = DyteRoomParticipants

class DyteRoomParticipants
internal constructor(
  private val participantController: ParticipantController,
  private val hostController: IHostController,
  private val waitlistController: WaitlistController
) {

  val serialScope = CoroutineScope(newSingleThreadContext("DyteRoomParticipants"))

  val pinned: DyteJoinedMeetingParticipant?
    get() = runBlocking {
      withContext(serialScope.coroutineContext) { participantController.pinnedParticipant }
    }

  val active: List<DyteJoinedMeetingParticipant>
    get() = participantController.activeParticipants.distinctBy { it.id }

  val joined: List<DyteJoinedMeetingParticipant>
    get() = participantController.joinedParticipants

  val activeSpeaker: DyteJoinedMeetingParticipant?
    get() = runBlocking {
      withContext(serialScope.coroutineContext) { participantController.activeSpeaker }
    }

  @Deprecated("Please use screenShares")
  val screenshares: List<DyteScreenShareMeetingParticipant>
    get() = participantController.screenshareParticipants

  val screenShares: List<DyteJoinedMeetingParticipant>
    get() = participantController.screenshareParticipants

  val waitlisted: List<DyteWaitlistedParticipant>
    get() = waitlistController.waitlistedParticipants

  val currentPageNumber: Int
    get() = participantController.getGridPagesInfo().currentPageNumber

  /** Boolean to indicate if there a next page of active participants available */
  val canGoNextPage: Boolean
    get() = participantController.getGridPagesInfo().isNextPagePossible

  /** Boolean to indicate if there a previous page of active participants available */
  val canGoPreviousPage: Boolean
    get() = participantController.getGridPagesInfo().isPreviousPagePossible

  /** Number of total active participant pages */
  val pageCount: Int
    get() = participantController.getGridPagesInfo().pageCount

  fun toMap(): Map<String, Any?> {
    val map = HashMap<String, Any?>()
    map["waitlisted"] = listDyteMeetingParticipantToMap(waitlistController.waitlistedParticipants)
    map["joined"] = listDyteMeetingParticipantToMap(participantController.joinedParticipants)
    map["active"] = listDyteMeetingParticipantToMap(active)
    map["screenshares"] =
      listDyteMeetingParticipantToMap(participantController.screenshareParticipants)
    map["pinned"] = pinned?.toMap()
    map["activeSpeaker"] = activeSpeaker?.toMap()
    map["grid"] = participantController.getGridPagesInfo().toMap()
    return map
  }

  private fun listDyteMeetingParticipantToMap(
    participants: List<DyteMeetingParticipant>
  ): List<Map<String, Any?>> {
    return participants.map { it.toMap() }
  }

  /** Navigates to the given [pageNumber] page */
  @Throws(Exception::class)
  fun setPage(pageNumber: Int) {
    if (pageNumber < 0) {
      throw IllegalArgumentException("page number cant be negative")
    }
    serialScope.launch { participantController.setPage(pageNumber) }
  }

  /** Mutes all [joined] participant's audio in the meeting */
  fun disableAllAudio() {
    // TODO : fix this hardcoding
    serialScope.launch { hostController.muteAllAudio(true) }
  }

  /** Disables all [joined] participant's camera / video in the meeting */
  @Throws(Exception::class)
  fun disableAllVideo() {
    serialScope.launch { hostController.muteAllVideo() }
  }

  fun disableCache() {
    serialScope.launch { participantController.disableCache() }
  }

  fun enableCache() {
    serialScope.launch { participantController.enableCache() }
  }

  /**
   * Accepts all pending requests from [waitlisted] participants waiting to be let in the meeting
   */
  @Throws(Exception::class)
  fun acceptAllWaitingRequests() {
    serialScope.launch { waitlistController.acceptAllWaitingRequests() }
  }

  /** Removes all [joined] participants */
  fun kickAll() {
    serialScope.launch { hostController.kickAll() }
  }

  /** Send a broadcast message to all [joined] participants */
  fun broadcastMessage(type: String, payload: Map<String, *>) {
    serialScope.launch { participantController.broadcastMessage(type, payload) }
  }
}
