package io.dyte.core.host

import io.dyte.core.DyteError
import io.dyte.core.HostControlErrorCodes.Kick_All
import io.dyte.core.HostControlErrorCodes.Kick_Peer
import io.dyte.core.HostControlErrorCodes.Mute_All_Audio
import io.dyte.core.HostControlErrorCodes.Mute_All_Video
import io.dyte.core.HostControlErrorCodes.Mute_Audio
import io.dyte.core.HostControlErrorCodes.Mute_Video
import io.dyte.core.HostControlErrorCodes.Pin_Peer
import io.dyte.core.HostControlErrorCodes.Unpin_Peer
import io.dyte.core.Result
import io.dyte.core.observability.DyteLogger
import io.dyte.core.socket.socketservice.ISockratesSocketService
import io.dyte.core.socket.socketservice.SocketServiceUtils
import io.dyte.webrtc.MediaStreamTrackKind
import media.edge.GlobalPeerPinningRequest
import media.edge.HostMediaControlForAllPeerRequest
import media.edge.HostMediaControlForAllPeerResponse
import media.edge.HostMediaControlForPeerRequest
import media.edge.HostMediaControlForPeerResponse
import media.edge.KickAllPeersResponse
import media.edge.KickPeerRequest
import media.edge.KickPeerResponse

internal class HostSocketServiceController(
  private val socketService: ISockratesSocketService,
) : IHostController {
  override suspend fun kickPeer(peerId: String): Result<Unit, DyteError> {
    return try {
      val req = KickPeerRequest(participant_id = peerId)

      val res =
        socketService.requestResponse(
          event = SocketServiceUtils.MediaEvent.KICK_PEER.id,
          payload = KickPeerRequest.ADAPTER.encode(req)
        )

      val result = !(res == null || KickPeerResponse.ADAPTER.decode(res).status != "success")
      if (result) {
        return Result.Success(Unit)
      } else {
        return Result.Failure(DyteError(Kick_Peer, "Failed to kick peer"))
      }
    } catch (e: Exception) {
      DyteLogger.error("HostSocketService::kick_peer_failed::", e)
      return Result.Failure(
        DyteError(Kick_Peer, "HostSocketService::kick_peer_failed: and raised Exception")
      )
    }
  }

  override suspend fun kickAll(): Result<Unit, DyteError> {
    return try {
      val res =
        socketService.requestResponse(
          event = SocketServiceUtils.MediaEvent.KICK_ALL.id,
          payload = null
        )

      val result = !(res == null || KickAllPeersResponse.ADAPTER.decode(res).status != "success")
      if (result) {
        return Result.Success(Unit)
      } else {
        return Result.Failure(DyteError(Kick_All))
      }
    } catch (e: Exception) {
      DyteLogger.error("HostSocketService::kick_all_failed::", e)
      Result.Failure(DyteError(Kick_All, "HostSocketService::kick_all_failed:: with Exception"))
    }
  }

  override suspend fun muteAllAudio(allowUnmute: Boolean?): Result<Unit, DyteError> {
    if (hostControlForAll(MediaStreamTrackKind.Audio)) {
      return Result.Success(Unit)
    }
    return Result.Failure(DyteError(Mute_All_Audio))
  }

  override suspend fun muteAllVideo(): Result<Unit, DyteError> {
    if (hostControlForAll(MediaStreamTrackKind.Video)) {
      return Result.Success(Unit)
    }
    return Result.Failure(DyteError(Mute_All_Video))
  }

  override suspend fun muteAudio(peerId: String): Result<Unit, DyteError> {
    if (hostControlForPeer(peerId, MediaStreamTrackKind.Audio)) {
      return Result.Success(Unit)
    }
    return Result.Failure(DyteError(Mute_Audio))
  }

  override suspend fun muteVideo(peerId: String): Result<Unit, DyteError> {
    if (hostControlForPeer(peerId, MediaStreamTrackKind.Video)) {
      return Result.Success(Unit)
    }
    return Result.Failure(DyteError(Mute_Video))
  }

  override suspend fun pinPeer(peerId: String): Result<Unit, DyteError> {
    if (pinPeerInternal(peerId)) {
      return Result.Success(Unit)
    }
    return Result.Failure(DyteError(Pin_Peer))
  }

  override suspend fun unpinPeer(): Result<Unit, DyteError> {
    if (pinPeerInternal()) {
      return Result.Success(Unit)
    }
    return Result.Failure(DyteError(Unpin_Peer))
  }

  private suspend fun hostControlForAll(kind: MediaStreamTrackKind): Boolean {
    return try {
      val req =
        HostMediaControlForAllPeerRequest(
          audio = kind == MediaStreamTrackKind.Audio,
          screen_share = false,
          video = kind == MediaStreamTrackKind.Video
        )

      val res =
        socketService.requestResponse(
          event = SocketServiceUtils.MediaEvent.HOST_CONTROL_ALL_PEERS.id,
          payload = HostMediaControlForAllPeerRequest.ADAPTER.encode(req)
        )

      !(res == null || HostMediaControlForAllPeerResponse.ADAPTER.decode(res).status != "success")
    } catch (e: Exception) {
      DyteLogger.error("HostSocketService::host_control_for_all_failed::", e)
      false
    }
  }

  private suspend fun hostControlForPeer(peerId: String, kind: MediaStreamTrackKind): Boolean {
    return try {
      val req =
        HostMediaControlForPeerRequest(
          audio = kind == MediaStreamTrackKind.Audio,
          scree_share = false,
          video = kind == MediaStreamTrackKind.Video,
          participant_id = peerId
        )

      val res =
        socketService.requestResponse(
          event = SocketServiceUtils.MediaEvent.HOST_CONTROL_PEER.id,
          payload = HostMediaControlForPeerRequest.ADAPTER.encode(req)
        )

      !(res == null || HostMediaControlForPeerResponse.ADAPTER.decode(res).status != "success")
    } catch (e: Exception) {
      DyteLogger.error("HostSocketService::host_control_for_peer_failed::", e)
      false
    }
  }

  private suspend fun pinPeerInternal(peerId: String = ""): Boolean {
    return try {
      val req =
        GlobalPeerPinningRequest(
          participant_id = peerId,
        )

      socketService.send(
        event = SocketServiceUtils.MediaEvent.GLOBAL_PIN_PEER.id,
        payload = GlobalPeerPinningRequest.ADAPTER.encode(req)
      )

      true
    } catch (e: Exception) {
      DyteLogger.error("HostSocketService::pin_peer_failed::", e)
      false
    }
  }
}
