package io.dyte.core.feat

import io.dyte.core.controllers.chat.BaseChatController
import io.dyte.core.events.PublicAPIEmitter
import io.dyte.core.listeners.DyteChatEventsListener
import io.dyte.core.platform.Uri
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.launch
import kotlinx.coroutines.newSingleThreadContext
import kotlinx.coroutines.runBlocking

// TODO : add pin/unpin support
// TODO: add pinned message type.
// TODO : getMessagesByUser
// TODO : getMessageByType
class DyteChat internal constructor(private val chatController: BaseChatController) :
  PublicAPIEmitter<DyteChatEventsListener>() {
  protected val serialScope = CoroutineScope(newSingleThreadContext("DyteChat"))

  override val emitterSuperClass = chatController

  val messages: List<DyteChatMessage>
    get() = runBlocking(serialScope.coroutineContext) { chatController.messages }

  fun toMap(): List<Map<String, Any?>> {
    return messages.map { encodeDyteChatMessage(it) }
  }

  /**
   * Sends a string type chat message in the given room
   *
   * @param message
   */
  @Throws(Exception::class)
  fun sendTextMessage(message: String) {
    serialScope.launch { chatController.sendMessage(message) }
  }

  @Deprecated(
    message =
      "Sending a file message no longer needs fileName parameter. This method will be removed in next version of the SDK.",
    replaceWith = ReplaceWith("sendFileMessage(fileUri: Uri)")
  )
  /**
   * Send file message
   *
   * @param filePath Location of file on device storage
   * @param fileName Name of the file, displayed on chat screen
   */
  @Throws(Exception::class)
  fun sendFileMessage(filePath: String, fileName: String) {
    serialScope.launch { chatController.sendFileMessage(filePath) }
  }

  @Deprecated(
    message =
      "Sending an image message no longer needs fileName parameter. This method will be removed in next version of the SDK.",
    replaceWith = ReplaceWith("sendImageMessage(imageUri: Uri)")
  )
  /**
   * Send chat image message
   *
   * @param filePath Location of image file on device storage
   * @param fileName Name of the image file, displayed on chat screen
   */
  @Suppress("UNUSED_PARAMETER")
  @Throws(Exception::class)
  fun sendImageMessage(filePath: String, fileName: String) {
    serialScope.launch { chatController.sendImageMessage(filePath) }
  }

  /**
   * Send chat image message
   *
   * @param imageUri uri of image
   */
  fun sendImageMessage(imageUri: Uri) {
    serialScope.launch { chatController.sendImageMessage(imageUri) }
  }

  /**
   * Send chat image message
   *
   * @param imagePath Location of image file on device storage
   */
  fun sendImageMessage(imagePath: String) {
    serialScope.launch { chatController.sendImageMessage(imagePath) }
  }

  /**
   * Send chat file message
   *
   * @param fileUri uri of file
   */
  fun sendFileMessage(fileUri: Uri) {
    serialScope.launch { chatController.sendFileMessage(fileUri) }
  }

  /**
   * Send chat file message
   *
   * @param filePath Location of file on device storage
   */
  fun sendFileMessage(filePath: String) {
    serialScope.launch { chatController.sendFileMessage(filePath) }
  }
}
