package io.dyte.core.controllers.chat

import io.dyte.core.events.EventEmitter
import io.dyte.core.feat.DyteChatMessage
import io.dyte.core.listeners.DyteChatEventsListener
import io.dyte.core.network.PresignedUrlApiService
import io.dyte.core.observability.DyteLogger
import io.dyte.core.platform.IDytePlatformUtilsProvider
import io.dyte.core.platform.PlatformFile
import io.dyte.core.platform.Uri
import io.dyte.core.plugins.PluginChatSender
import io.ktor.client.HttpClient
import io.ktor.client.plugins.HttpTimeout
import io.ktor.client.plugins.contentnegotiation.ContentNegotiation
import io.ktor.client.request.header
import io.ktor.client.request.put
import io.ktor.client.request.setBody
import io.ktor.client.request.url
import io.ktor.http.HttpHeaders
import io.ktor.http.HttpStatusCode
import io.ktor.serialization.kotlinx.json.json
import kotlinx.serialization.json.Json

/*
 * TODO: Need to setup removeOnMessageListener somewhere.
 * Ideal way would be to listen to `roomJoined` event to set listeners and
 * `roomLeft` event to remove listeners.
 * */
internal abstract class BaseChatController(
  private val platformUtilsProvider: IDytePlatformUtilsProvider,
  private val presignedUrlApiService: PresignedUrlApiService,
  private val canSendChatText: Boolean,
  private val canSendChatFile: Boolean,
) : EventEmitter<DyteChatEventsListener>(), IChatController, PluginChatSender {

  private lateinit var roomUUID: String

  /** TODO: Remove this */
  override fun setRoomUUID(uuid: String) {
    roomUUID = uuid
  }
  /*
   * Note(swapnil): We are using a separate instance of HttpClient in ChatController to upload files.
   * Need to investigate more to see if we can eliminate this extra instance.
   *
   * Reason: We have added a default Authorization header to the shared HttpClient in the ApiClient class. This
   * adds the Authorization header to every request. But the pre-signed upload Urls already have authorization
   * mechanism through some query params and don't allow additional Authorization header. In case, it is
   * added to the request, we get a 400 Bad request response with "Invalid Argument" message.
   * */
  private val httpClient: HttpClient by lazy { createPrivateHttpClient() }

  override val messages = arrayListOf<DyteChatMessage>()

  /**
   * Sets a message listener on the ChatSocketHandler for messages coming from the socket service.
   *
   * TODO: Need to figure out a way to remove listener from ChatSocketHandler. Ideal way would be to
   *   listen to `roomJoined` event to set listeners and `roomLeft` event to remove listeners.
   */
  override suspend fun loadChatMessages() {
    try {
      val newMessages = this.getChatMessages()
      handleMultipleMessages(newMessages)
    } catch (e: Exception) {
      DyteLogger.error("ChatController::loadChatMessages::error", e)
    }
  }

  protected fun handleMultipleMessages(newMessages: List<DyteChatMessage>) {
    messages.clear()
    messages.addAll(newMessages)
    if (newMessages.isNotEmpty()) {
      emitEvent { it.onChatUpdates(messages) }
    }
  }

  protected fun handleMessage(newMessage: DyteChatMessage) {
    messages.add(newMessage)
    emitEvent {
      it.onNewChatMessage(newMessage)
      it.onChatUpdates(messages)
    }
  }

  protected abstract suspend fun getChatMessages(): List<DyteChatMessage>

  override suspend fun sendMessage(message: String) {
    DyteLogger.info("DyteChat.sendTextMessage")
    if (!canSendChatText) {
      DyteLogger.error("sendTextMessage::public_chat_permission_denied")
      throw UnsupportedOperationException("not allowed to send text message")
    }

    if (message.isEmpty()) {
      DyteLogger.error("sendTextMessage::message_can_not_be_empty")
      throw IllegalArgumentException("message can not be empty")
    }
    sendTextMessageSocket(message)
  }

  protected abstract suspend fun sendTextMessageSocket(message: String)

  override suspend fun sendFileMessage(fileUri: Uri) {
    DyteLogger.info("DyteChat.sendFileMessage")
    if (!canSendChatFile) {
      DyteLogger.info("sendFileMessage::chat_permission_denied")
      throw UnsupportedOperationException("not allowed to send file message")
    }

    val file =
      platformUtilsProvider.getPlatformUtils().getPlatformFile(fileUri)
        ?: run {
          DyteLogger.info("sendFileMessage::chat_read_file_failed")
          throw Exception("Requested file doesn't exist or is missing read permission")
        }

    return sendFileMessageInternal(file)
  }

  override suspend fun sendFileMessage(filePath: String) {
    DyteLogger.info("DyteChat.sendFileMessage")
    if (!canSendChatFile) {
      DyteLogger.info("sendFileMessage::chat_permission_denied")
      throw UnsupportedOperationException("not allowed to send file message")
    }

    val file =
      platformUtilsProvider.getPlatformUtils().getPlatformFile(filePath)
        ?: run {
          DyteLogger.info("sendFileMessage::chat_read_file_failed")
          throw Exception("Requested file doesn't exist or is missing read permission")
        }

    return sendFileMessageInternal(file)
  }

  override suspend fun sendImageMessage(imageUri: Uri) {
    DyteLogger.info("DyteChat.sendImageMessage")

    if (!canSendChatFile) {
      DyteLogger.error("sendImageMessage::permission_denied")
      throw UnsupportedOperationException("not allowed to send file message")
    }

    val file =
      platformUtilsProvider.getPlatformUtils().getPlatformFile(imageUri)
        ?: run {
          DyteLogger.error("sendImageMessage::required_argument_image_can_not_be_empty")
          throw Exception("Requested image doesn't exist or is missing read permission")
        }

    return sendImageMessageInternal(file)
  }

  override suspend fun sendImageMessage(imagePath: String) {
    DyteLogger.info("DyteChat.sendImageMessage")
    if (!canSendChatFile) {
      DyteLogger.error("sendImageMessage::permission_denied")
      throw UnsupportedOperationException("not allowed to send image message")
    }

    val file =
      platformUtilsProvider.getPlatformUtils().getPlatformFile(imagePath)
        ?: throw Exception("Requested image doesn't exist or is missing read permission")

    return sendImageMessageInternal(file)
  }

  private suspend fun sendImageMessageInternal(file: PlatformFile) {
    if (file.mimeType !in ACCEPTED_IMAGE_TYPES) {
      DyteLogger.error("sendImageMessage::image_type_not_supported")
      throw UnsupportedOperationException("Unsupported image file found -> ${file.mimeType}")
    }
    val presignedUrl =
      presignedUrlApiService.getPresignedUrl2(filename = file.name, roomUuid = roomUUID)

    val putLocation = presignedUrl?.data?.putLocation
    val getLocation = presignedUrl?.data?.getLocation

    if (putLocation.isNullOrBlank() || getLocation.isNullOrBlank()) {
      DyteLogger.error("sendImageMessage::failure_get_presigned_url")
      throw Exception("Failed to upload image")
    }

    val uploadImageResponse =
      httpClient.put {
        url(putLocation)
        setBody(file.content)
      }

    if (uploadImageResponse.status.value != HttpStatusCode.OK.value) {
      DyteLogger.error("sendImageMessage::failure_upload_image_file")
      throw Exception("Failed to upload image")
    }

    this.sendImageMessageSocket(getLocation)
  }

  protected abstract suspend fun sendImageMessageSocket(path: String)

  private suspend fun sendFileMessageInternal(file: PlatformFile) {
    val presignedUrl = presignedUrlApiService.getPresignedUrl2(file.name, roomUUID)

    val putLocation = presignedUrl?.data?.putLocation
    val getLocation = presignedUrl?.data?.getLocation

    if (putLocation.isNullOrBlank() || getLocation.isNullOrBlank()) {
      DyteLogger.error("sendFileMessage::failure_get_presigned_url")
      throw Exception("Failed to upload file")
    }

    val uploadFileResponse =
      httpClient.put(putLocation) {
        header(HttpHeaders.ContentType, file.mimeType)
        setBody(file.content)
      }

    if (uploadFileResponse.status.value != HttpStatusCode.OK.value) {
      DyteLogger.error("sendFileMessage::failure_upload_file")
      throw Exception("Failed to upload file")
    }

    sendFileMessageSocket(getLocation, file.name, file.size)
  }

  protected abstract suspend fun sendFileMessageSocket(
    fileUrl: String,
    fileName: String,
    fileSize: Long
  )

  override suspend fun sendTextMessageFromPlugin(message: String) {
    sendMessage(message)
  }

  private fun createPrivateHttpClient(): HttpClient {
    return HttpClient {
      install(ContentNegotiation) {
        json(
          Json {
            prettyPrint = true
            isLenient = true
            ignoreUnknownKeys = true
          }
        )
      }
      install(HttpTimeout) {
        socketTimeoutMillis = 30000
        requestTimeoutMillis = 30000
        connectTimeoutMillis = 30000
      }
    }
  }

  protected fun parseTime(t: Long?): String {
    return platformUtilsProvider
      .getPlatformUtils()
      .getUserDisplayableTime(t?.times(1000) ?: error("Message time is null"))
  }

  companion object {
    private val ACCEPTED_IMAGE_TYPES = listOf("image/jpeg", "image/png")
  }
}

interface IChatController {
  val messages: List<DyteChatMessage>

  fun setRoomUUID(uuid: String)

  suspend fun loadChatMessages()

  suspend fun sendMessage(message: String)

  suspend fun sendImageMessage(imageUri: Uri)

  suspend fun sendImageMessage(imagePath: String)

  suspend fun sendFileMessage(fileUri: Uri)

  suspend fun sendFileMessage(filePath: String)
}

enum class MessageType(val type: Int) {
  TEXT(0),
  IMAGE(1),
  FILE(2)
}
