package io.dyte.core.controllers

import io.dyte.core.events.EventEmitter
import io.dyte.core.events.InternalEvents
import io.dyte.core.feat.DyteWaitlistedParticipant
import io.dyte.core.feat.ParticipantFlags
import io.dyte.core.host.IHostController
import io.dyte.core.listeners.DyteWaitlistEventsListener
import io.dyte.core.models.DyteRoomParticipants
import io.dyte.core.models.DyteSelfParticipant
import io.dyte.core.models.WaitListStatus.ACCEPTED
import io.dyte.core.models.WaitListStatus.REJECTED
import io.dyte.core.socket.RoomNodeSocketService
import io.dyte.core.socket.SocketMessageEventListener
import io.dyte.core.socket.events.InboundMeetingEventType
import io.dyte.core.socket.events.OutboundMeetingEventType.ACCEPT_WAITLIST_REQUEST
import io.dyte.core.socket.events.OutboundMeetingEventType.REJECT_WAITLIST_REQUEST
import io.dyte.core.socket.events.payloadmodel.InboundMeetingEvent
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketMeetingPeerUser
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketWaitlistPeerAdded
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketWaitlistPeerClosed
import io.dyte.core.socket.events.payloadmodel.outbound.WebSocketJoinRoomModel
import kotlinx.coroutines.CoroutineScope
import kotlinx.serialization.json.JsonElement
import kotlinx.serialization.json.JsonObject
import kotlinx.serialization.json.JsonPrimitive

internal class WaitlistController(
  private val internalEventsEmitter: EventEmitter<InternalEvents>,
  private val selfParticipant: DyteSelfParticipant,
  private val socketController: RoomNodeSocketService,
  private val scope: CoroutineScope,
  private val participantController: ParticipantController,
  private val hostController: IHostController
) : EventEmitter<DyteWaitlistEventsListener>(), IWaitlistController, InternalEvents {
  override val waitlistedParticipants = arrayListOf<DyteWaitlistedParticipant>()

  private val selfPermissions = selfParticipant.permissions

  init {
    socketController.addMessageEventListener(
      InboundMeetingEventType.WEB_SOCKET_PEER_JOINED,
      object : SocketMessageEventListener {
        override suspend fun onMessageEvent(event: InboundMeetingEvent) {
          val payload = event.payload as WebSocketMeetingPeerUser
          val waitlistedParticipant = waitlistedParticipants.find { it.id == payload.id }
          if (waitlistedParticipant != null) {
            waitlistedParticipants.remove(waitlistedParticipant)
            emitEvent { it.onWaitListParticipantAccepted(waitlistedParticipant) }
          }
        }
      }
    )

    socketController.addMessageEventListener(
      InboundMeetingEventType.WEB_SOCKET_WAITLIST_PEER_ADDED,
      object : SocketMessageEventListener {
        override suspend fun onMessageEvent(event: InboundMeetingEvent) {
          val payload = event.payload as WebSocketWaitlistPeerAdded
          val participant =
            DyteWaitlistedParticipant(
              payload.id,
              payload.id,
              payload.name,
              null,
              false,
              null,
              ParticipantFlags(false, false, false),
              "",
              participantController,
              this@WaitlistController,
              scope
            )
          handleWaitlistedPeer(participant)
        }
      }
    )

    socketController.addMessageEventListener(
      InboundMeetingEventType.WEB_SOCKET_WAITLIST_PEER_CLOSED,
      object : SocketMessageEventListener {
        override suspend fun onMessageEvent(event: InboundMeetingEvent) {
          val payload = event.payload as WebSocketWaitlistPeerClosed
          val closedPeer = waitlistedParticipants.find { it.id == payload.id }
          closedPeer?.let {
            waitlistedParticipants.remove(closedPeer)
            emitEvent { it.onWaitListParticipantClosed(closedPeer) }
          }
        }
      }
    )

    socketController.addMessageEventListener(
      InboundMeetingEventType.WEB_SOCKET_WAITLIST_REJECTED,
      object : SocketMessageEventListener {
        override suspend fun onMessageEvent(event: InboundMeetingEvent) {
          selfParticipant._waitListStatus = REJECTED
          internalEventsEmitter.emitEvent { it.onWaitlistEntryRejected() }
        }
      }
    )

    socketController.addMessageEventListener(
      InboundMeetingEventType.WEB_SOCKET_WAITLIST_ACCEPTED,
      object : SocketMessageEventListener {
        override suspend fun onMessageEvent(event: InboundMeetingEvent) {
          selfParticipant._waitListStatus = ACCEPTED
          internalEventsEmitter.emitEvent { it.onWaitlistEntryAccepted() }
          internalEventsEmitter.emitEvent { it.connectToRoomNode() }
        }
      }
    )
  }

  override fun onRoomJoined(webSocketJoinRoomModel: WebSocketJoinRoomModel) {
    super.onRoomJoined(webSocketJoinRoomModel)
    if (selfPermissions.waitingRoom.canAcceptRequests) {
      webSocketJoinRoomModel.waitlistedPeers.forEach {
        val participant =
          DyteWaitlistedParticipant(
            it.id,
            "",
            it.name ?: "",
            null,
            false,
            null,
            ParticipantFlags(false, false, false),
            "",
            participantController,
            this,
            scope
          )
        handleWaitlistedPeer(participant)
      }
    }
  }

  private fun handleWaitlistedPeer(participant: DyteWaitlistedParticipant) {
    if (waitlistedParticipants.none { it.id == participant.id }) {
      waitlistedParticipants.add(participant)
      emitEvent { it.onWaitListParticipantJoined(participant) }
    }
  }

  override suspend fun acceptAllWaitingRequests() {
    if (selfPermissions.waitingRoom.canAcceptRequests.not()) {
      throw UnsupportedOperationException("not permitted to accept all waitlisted request")
    }
    val waitlisted = ArrayList(waitlistedParticipants)
    waitlisted.forEach { acceptWaitlistedRequest(it) }
  }

  override suspend fun acceptWaitlistedRequest(dyteMeetingParticipant: DyteWaitlistedParticipant) {
    if (selfPermissions.waitingRoom.canAcceptRequests.not()) {
      throw UnsupportedOperationException("not permitted to accept waitlisted request")
    }
    val content = HashMap<String, JsonElement>()
    content["id"] = JsonPrimitive(dyteMeetingParticipant.id)
    socketController.sendMessage(ACCEPT_WAITLIST_REQUEST, JsonObject(content))
    waitlistedParticipants.remove(dyteMeetingParticipant)
    emitEvent { it.onWaitListParticipantAccepted(dyteMeetingParticipant) }
  }

  override suspend fun rejectWaitlistedRequest(dyteMeetingParticipant: DyteWaitlistedParticipant) {
    if (selfPermissions.waitingRoom.canAcceptRequests.not()) {
      throw UnsupportedOperationException("not permitted to reject waitlisted request")
    }
    val content = HashMap<String, JsonElement>()
    content["id"] = JsonPrimitive(dyteMeetingParticipant.id)
    socketController.sendMessage(REJECT_WAITLIST_REQUEST, JsonObject(content))
    waitlistedParticipants.remove(dyteMeetingParticipant)
    emitEvent { it.onWaitListParticipantRejected(dyteMeetingParticipant) }
  }

  override fun emitEvent(event: (listener: DyteWaitlistEventsListener) -> Unit) {
    super.emitEvent(event)
    participantController.emitEvent {
      it.onUpdate(DyteRoomParticipants(participantController, hostController, this))
    }
  }
}

interface IWaitlistController {
  val waitlistedParticipants: List<DyteWaitlistedParticipant>

  suspend fun acceptAllWaitingRequests()

  suspend fun acceptWaitlistedRequest(dyteMeetingParticipant: DyteWaitlistedParticipant)

  suspend fun rejectWaitlistedRequest(dyteMeetingParticipant: DyteWaitlistedParticipant)
}
