package io.dyte.core.controllers

import io.dyte.core.events.EventEmitter
import io.dyte.core.events.InternalEvents
import io.dyte.core.listeners.DyteConnectionEventListener
import io.dyte.core.observability.DyteLogger
import io.dyte.core.platform.IDytePlatformUtilsProvider
import io.dyte.core.socket.IRoomNodeSocketService
import io.dyte.core.socket.SocketConnectionEvent
import io.dyte.core.socket.SocketConnectionEventListener
import io.dyte.core.socket.socketservice.ISockratesSocketService
import io.dyte.core.socket.socketservice.SocketServiceConnectionStateListener
import io.dyte.core.socket.socketservice.SocketServiceUtils
import io.dyte.sockrates.client.WebSocketConnectionState
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.launch
import kotlinx.coroutines.newSingleThreadContext
import kotlinx.datetime.Instant
import socket.room.JoinRoomRequest
import socket.room.Peer
import socket.room.RoomInfoResponse

internal class SocketController(private val controllerContainer: Controller) :
  EventEmitter<DyteConnectionEventListener>() {

  private lateinit var oldSocket: IRoomNodeSocketService
  private lateinit var newSocket: ISockratesSocketService

  private lateinit var internalEmitter: EventEmitter<InternalEvents>
  private var reconnectingToMeetingRoom = false
  private var useHive = false

  private var prevConnected = false
  private lateinit var platformConnectionNotif: PlatformConnectionChangeListener

  val serialScope = CoroutineScope(newSingleThreadContext("DyteScope${hashCode()}"))

  fun init(useHive: Boolean) {
    oldSocket = controllerContainer.roomNodeSocketService
    newSocket = controllerContainer.sockratesSocketService
    internalEmitter = controllerContainer.internalEventEmitter
    this.useHive = useHive
    prevConnected = false
    /**
     * The below listeners are for external event listeners, if it is a hive meeting room node
     * socket service will not be used and sockrates socket service is taken as source of truth in
     * room node meetings, both socket systems are active but room node socket service is used as a
     * source of truth
     */
    if (useHive) {
      newSocket.addConnectionStateListener(
        object : SocketServiceConnectionStateListener {
          override fun onConnectionStateChanged(newState: WebSocketConnectionState) {
            onNewSocketConnectionEvent(newState)
          }
        }
      )
    } else {
      oldSocket.addConnectionListener(
        object : SocketConnectionEventListener {
          override fun onConnectionEvent(event: SocketConnectionEvent) {
            onOldSocketConnectionEvent(event)
          }
        }
      )
    }

    /**
     * The below listeners are for controlling media flow the listeners need to be setup on both, if
     * one of the the socket disconnects we need to disconnect the other one to ensure consistent
     * state on the server
     */
    newSocket.addConnectionStateListener(
      object : SocketServiceConnectionStateListener {
        override fun onConnectionStateChanged(newState: WebSocketConnectionState) {
          onNewSocketConnectionMediaEvent(newState)
        }
      }
    )

    oldSocket.addConnectionListener(
      object : SocketConnectionEventListener {
        override fun onConnectionEvent(event: SocketConnectionEvent) {
          onOldSocketConnectionMediaEvent(event)
        }
      }
    )
    platformConnectionNotif =
      PlatformConnectionChangeListener(controllerContainer.platformUtilsProvider) {
        serialScope.launch { reconnectSocketsWithNewPeerId() }
      }
  }

  /*
  note(swapnil): This method or step can be moved to some other initializer class once
  the roomUUID retrieval is made cleaner.
   */
  suspend fun joinRoomSockratesSocketService() {
    var peerId = controllerContainer.metaController.getPeerId()
    var displayName = controllerContainer.selfController.getSelf().name
    var userId = controllerContainer.selfController.getSelf().userId

    try {
      val roomUuid = if (!useHive) controllerContainer.getRoomUuid() else ""
      /*
       * Note(swapnil): We are storing the roomUuid in the MetaController for using it inside
       * ChatController and PluginsController. Ideal way would be to fetch the roomUuid before
       * initialisation of the feature modules/controllers.
       *
       * WARNING: This is a temporary hack to make the roomUuid available to feature modules.
       * Doing this after connecting to sockets in doInit() was breaking on iOS.
       * TODO: Improve the initialization order.
       * */
      controllerContainer.metaController.setRoomUuid(roomUuid)
      val joinRoomRequest =
        JoinRoomRequest(
          peer = Peer(peer_id = peerId, display_name = displayName, user_id = userId),
          room_uuid = controllerContainer.metaController.getRoomUuid()
        )

      newSocket.requestResponse(
        event = SocketServiceUtils.RoomEvent.JOIN_ROOM.id,
        payload = JoinRoomRequest.ADAPTER.encode(joinRoomRequest)
      )

      // Get meeting started timestamp
      try {
        val roomInfoResponse =
          newSocket.requestResponse(SocketServiceUtils.RoomEvent.GET_ROOM_INFO.id, null)
            ?: throw Exception("GET_ROOM_INFO response is null")
        val parsedRoomInfo = RoomInfoResponse.ADAPTER.decode(roomInfoResponse)

        val roomCreatedAtSeconds =
          parsedRoomInfo.room?.created_at ?: throw Exception("GET_ROOM_INFO room field is null")

        val timeInEpochMillis = roomCreatedAtSeconds * 1000

        controllerContainer.metaController.setMeetingStatedTime(timeInEpochMillis)

        if (useHive) {
          // RoomNodeController sets this for Mediasoup but we don't have the time in
          // PeerJoinResponse
          // for hive, so I'm setting this here.
          val timeInISO8601 =
            Instant.fromEpochMilliseconds(timeInEpochMillis).toString().replace("Z", ".000Z")

          controllerContainer.metaController.setMeetingStartedTimestamp(timeInISO8601.toString())
        }
      } catch (e: Exception) {
        DyteLogger.error("SocketController::failed to get meeting started timestamp", e)
      }
    } catch (e: Exception) {
      DyteLogger.error("Controller::sendJoinRoomSocketService::failed", e)
    }
  }

  private fun onConnected() {
    reconnectingToMeetingRoom = false
    if (!prevConnected) {
      emitEvent { it.onConnectedToMeetingRoom() }
    } else {
      emitEvent { it.onReconnectedToMeetingRoom() }
    }
    prevConnected = true
  }

  private fun onConnecting() {
    if (!prevConnected) {
      emitEvent { it.onConnectingToMeetingRoom() }
    } else {
      emitEvent { it.onReconnectingToMeetingRoom() }
    }
  }

  private fun onConnectFailed() {
    if (!prevConnected) {
      emitEvent { it.onMeetingRoomConnectionFailed() }
    } else {
      emitEvent { it.onMeetingRoomReconnectionFailed() }
    }
  }

  private fun onDisconnected() {
    emitEvent { it.onDisconnectedFromMeetingRoom() }
  }

  private fun onOldSocketConnectionEvent(event: SocketConnectionEvent) {
    DyteLogger.info("SocketController::${event}")
    when (event) {
      is SocketConnectionEvent.Connected -> {
        onConnected()
      }
      is SocketConnectionEvent.ConnectionFailed -> onConnectFailed()
      is SocketConnectionEvent.Disconnected -> onDisconnected()
      else -> {}
    }
  }

  private fun onNewSocketConnectionEvent(state: WebSocketConnectionState) {
    DyteLogger.info("SocketController::${state}")
    when (state) {
      is WebSocketConnectionState.Connected -> {
        onConnected()
      }
      is WebSocketConnectionState.Connecting -> onConnecting()
      is WebSocketConnectionState.ConnectFailed -> onConnectFailed()
      is WebSocketConnectionState.Disconnected -> onDisconnected()
      else -> {}
    }
  }

  private fun onOldSocketConnectionMediaEvent(event: SocketConnectionEvent) {
    DyteLogger.info("SocketController::Old::${event}")
    when (event) {
      is SocketConnectionEvent.Connected -> {
        if (!useHive) {
          internalEmitter.emitEvent { it.connectMedia() }
        }
      }
      is SocketConnectionEvent.Disconnected -> {
        // do not reconnect if manually disconnected
        // https://socket.io/docs/v4/client-socket-instance/#disconnect
        if (event.reason != "io client disconnect") {
          reconnectSocketsWithNewPeerId()
        }
      }
      else -> {}
    }
  }

  private fun onNewSocketConnectionMediaEvent(state: WebSocketConnectionState) {
    DyteLogger.info("SocketController::New::${state}")
    when (state) {
      is WebSocketConnectionState.Connected -> {
        if (controllerContainer.selfController._roomJoined) {
          serialScope.launch { joinRoomSockratesSocketService() }
        }
        if (useHive) {
          internalEmitter.emitEvent { it.connectMedia() }
        }
      }
      is WebSocketConnectionState.Disconnected -> {
        // do not reconnect if manually disconnected
        // code == 1000
        if (state.code != 1000) {
          reconnectSocketsWithNewPeerId()
        }
      }
      else -> {}
    }
  }

  /**
   * Changes socket Urls of both SocketController and SocketService client to use the new peerId and
   * reconnects to both RoomNode socket and SocketService.
   */
  private fun reconnectSocketsWithNewPeerId() {
    DyteLogger.info("SocketController: reconnectSocketsWithNewPeerId $reconnectingToMeetingRoom")
    if (!reconnectingToMeetingRoom) {
      serialScope.launch {
        reconnectingToMeetingRoom = true
        internalEmitter.emitEvent { it.disconnectMedia() }

        if (!useHive) {
          oldSocket.disconnect()
        }
        newSocket.disconnect()

        val peerId = controllerContainer.metaController.refreshPeerId()

        if (!useHive) {
          oldSocket.refreshUrl(peerId)
        }
        newSocket.refreshUrl(peerId)

        if (!useHive) {
          oldSocket.connect()
          emitEvent { it.onReconnectingToMeetingRoom() }
        }
        newSocket.connect()
      }
    }
  }
}

internal expect class PlatformConnectionChangeListener(
  provider: IDytePlatformUtilsProvider,
  disconnectCallback: () -> Unit
) {}
