package io.dyte.core.controllers

import io.dyte.core.models.ProduceData
import io.dyte.core.network.BaseApiService
import io.dyte.core.observability.DyteLogger
import io.dyte.core.socket.events.OutboundMeetingEventType
import io.dyte.core.socket.events.OutboundMeetingEventType.MUTE_VIDEO
import io.dyte.core.socket.events.OutboundMeetingEventType.PAUSE_PRODUCER
import io.dyte.core.socket.events.OutboundMeetingEventType.PRODUCE
import io.dyte.core.socket.events.OutboundMeetingEventType.RESUME_PRODUCER
import io.dyte.core.socket.events.payloadmodel.inbound.ConsumerAppData
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketConsumerModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketProducerConnectModel
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import kotlinx.serialization.json.JsonElement
import kotlinx.serialization.json.JsonObject
import kotlinx.serialization.json.JsonPrimitive

internal class MediaSoupController(controllerContainer: IControllerContainer) :
  IMediaSoupController, BaseController(controllerContainer) {
  private lateinit var _produceDataAudio: ProduceData
  private lateinit var _produceDataVideo: ProduceData

  //    override val produceData: ProduceData
  //        get() = _produceData

  private var _videoProduceId: String? = null
  override val videoProduceId: String?
    get() = _videoProduceId

  private var _audioProduceId: String? = null
  override val audioProduceId: String?
    get() = _audioProduceId

  private val consumers = arrayListOf<WebSocketConsumerModel>()

  override suspend fun onReceiveTransportConnected(transportId: String, dtlsParameters: String) {
    withContext(serialScope.coroutineContext) {
      DyteLogger.info("onReceiveTransportConnected")
      val content = HashMap<String, JsonElement>()
      content["transportId"] = JsonPrimitive(transportId)
      content["dtlsParameters"] = BaseApiService.json.parseToJsonElement(dtlsParameters)
      val payload = JsonObject(content)
      val resp =
        controllerContainer.roomNodeSocketService.sendMessage(
          OutboundMeetingEventType.CONNECT_WEB_RTC_TRANSPORT,
          payload
        )
    }
  }

  override suspend fun onSendTransportConnected(transportId: String, dtlsParameters: String) {
    withContext(serialScope.coroutineContext) {
      DyteLogger.info("onSendTransportConnected")
      val content = HashMap<String, JsonElement>()
      content["transportId"] = JsonPrimitive(transportId)
      content["dtlsParameters"] = BaseApiService.json.parseToJsonElement(dtlsParameters)
      val payload = JsonObject(content)
      val resp =
        controllerContainer.roomNodeSocketService.sendMessage(
          OutboundMeetingEventType.CONNECT_WEB_RTC_TRANSPORT,
          payload
        )
    }
  }

  override suspend fun onProduce(
    transportId: String,
    kind: String,
    rtpParameters: String,
    appData: String?
  ): String {
    return withContext(serialScope.coroutineContext) {
      if (kind == "video") {
        _produceDataVideo = ProduceData(transportId, kind, rtpParameters, appData)

        if (controllerContainer.selfController.getSelf()._videoEnabled) {
          _videoProduceId = sendUnmuteMessage(_produceDataVideo)
        }

        return@withContext _videoProduceId ?: ""
      } else {
        _produceDataAudio = ProduceData(transportId, kind, rtpParameters, appData)
        _audioProduceId = sendUnmuteMessage(_produceDataAudio)

        return@withContext _audioProduceId ?: ""
      }
    }
  }

  override suspend fun produceVideoIfNeeded() {
    withContext(serialScope.coroutineContext) {
      if (_videoProduceId == null) {
        _videoProduceId = sendUnmuteMessage(_produceDataVideo)
      }
    }
  }

  override fun destroyCurrentVideo() {
    serialScope.launch { _videoProduceId = null }
  }

  override suspend fun sendDestroyVideoMessage(): Boolean {
    return withContext(serialScope.coroutineContext) {
      val producerId = videoProduceId
      producerId?.let {
        val content = HashMap<String, JsonElement>()
        content["producerId"] = JsonPrimitive(producerId)
        controllerContainer.roomNodeSocketService.sendMessage(MUTE_VIDEO, JsonObject(content))
        return@withContext true
      }
      return@withContext false
    }
  }

  private suspend fun sendUnmuteMessage(produceData: ProduceData): String {
    val appData = requireNotNull(produceData.appData) { "appData must be non-null" }

    val content = HashMap<String, JsonElement>()
    content["transportId"] = JsonPrimitive(produceData.transportId)
    content["rtpParameters"] = BaseApiService.json.parseToJsonElement(produceData.rtpParameters)
    content["kind"] = JsonPrimitive(produceData.kind)
    content["appData"] = BaseApiService.json.parseToJsonElement(appData)
    try {
      val resp = controllerContainer.roomNodeSocketService.sendMessage(PRODUCE, JsonObject(content))
      val parsedResp =
        controllerContainer.socketMessageResponseParser.parseResponse(resp).payload
          as WebSocketProducerConnectModel
      return parsedResp.id
    } catch (e: Exception) {
      // TODO : handle this, this gives error
      DyteLogger.error("MediasoupController::unmute_error", e)
    }
    return ""
  }

  override fun onNewConsumer(webSocketConsumerModel: WebSocketConsumerModel) {
    serialScope.launch { consumers.add(webSocketConsumerModel) }
  }

  override fun getAppDataFromConsumerId(consumerId: String): ConsumerAppData? {
    return consumers.find { it.id == consumerId }?.appData
  }

  override fun getConsumerType(consumerId: String): String {
    val consumer = consumers.find { it.id == consumerId }
    return consumer?.kind ?: throw IllegalArgumentException("consumer with $consumerId not found")
  }

  override fun onProducerPause(producerId: String) {
    val content = mapOf("producerId" to JsonPrimitive(producerId))
    serialScope.launch {
      controllerContainer.roomNodeSocketService.sendMessage(PAUSE_PRODUCER, JsonObject(content))
    }
  }

  override fun onProducerResume(producerId: String) {
    val content = mapOf("producerId" to JsonPrimitive(producerId))
    serialScope.launch {
      controllerContainer.roomNodeSocketService.sendMessage(RESUME_PRODUCER, JsonObject(content))
    }
  }

  override fun onProducerClose(producerId: String) {
    val content = HashMap<String, JsonElement>()
    content["producerId"] = JsonPrimitive(producerId)
    serialScope.launch {
      controllerContainer.roomNodeSocketService.sendMessage(MUTE_VIDEO, JsonObject(content))
    }
  }
}

internal interface IMediaSoupController {
  //    val produceData: ProduceData
  val videoProduceId: String?
  val audioProduceId: String?

  suspend fun onProduce(
    transportId: String,
    kind: String,
    rtpParameters: String,
    appData: String?
  ): String

  suspend fun onReceiveTransportConnected(transportId: String, dtlsParameters: String)

  suspend fun onSendTransportConnected(transportId: String, dtlsParameters: String)

  suspend fun produceVideoIfNeeded()

  suspend fun sendDestroyVideoMessage(): Boolean

  // Below method will actually setting current video producer id to null, so that we can generate
  // new videoProducer Id every time when unmute
  fun destroyCurrentVideo()

  fun onNewConsumer(webSocketConsumerModel: WebSocketConsumerModel)

  fun getAppDataFromConsumerId(consumerId: String): ConsumerAppData?

  fun getConsumerType(consumerId: String): String

  fun onProducerPause(producerId: String)

  fun onProducerResume(producerId: String)

  fun onProducerClose(producerId: String)
}
