package io.dyte.core

import io.dyte.core.controllers.Controller
import io.dyte.core.feat.DyteChat
import io.dyte.core.feat.DyteLiveStream
import io.dyte.core.feat.DyteMeta
import io.dyte.core.feat.DytePlugins
import io.dyte.core.feat.DytePoll
import io.dyte.core.feat.DyteRecording
import io.dyte.core.feat.DyteStage
import io.dyte.core.listeners.DyteChatEventsListener
import io.dyte.core.listeners.DyteDataUpdateListener
import io.dyte.core.listeners.DyteLiveStreamEventsListener
import io.dyte.core.listeners.DyteMeetingRoomEventsListener
import io.dyte.core.listeners.DyteParticipantEventsListener
import io.dyte.core.listeners.DytePluginEventsListener
import io.dyte.core.listeners.DytePollEventsListener
import io.dyte.core.listeners.DyteRecordingEventsListener
import io.dyte.core.listeners.DyteSelfEventsListener
import io.dyte.core.listeners.DyteStageEventListener
import io.dyte.core.listeners.DyteWaitlistEventsListener
import io.dyte.core.models.DyteMeetingInfo
import io.dyte.core.models.DyteMeetingInfoV2
import io.dyte.core.models.DyteParticipants
import io.dyte.core.models.DyteSelfParticipant
import io.dyte.core.observability.DyteLogger
import io.dyte.core.platform.IDytePlatformUtilsProvider
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.newSingleThreadContext
import kotlinx.coroutines.withContext

/**
 * Dyte mobile client
 *
 * todo : rename -> DyteClient
 *
 * @constructor Create empty Dyte mobile client
 */
open class DyteMobileClient
internal constructor(dytePlatformUtilsProvider: IDytePlatformUtilsProvider) : IDyteClient {
  private var controller: Controller = Controller(dytePlatformUtilsProvider)
  private var serialScope = CoroutineScope(newSingleThreadContext("DyteMobileClient"))

  override val localUser: DyteSelfParticipant
    get() = controller.getSelf()

  override val participants: DyteParticipants
    get() = controller.participants

  override val chat: DyteChat
    get() = controller.getChat()

  override val recording: DyteRecording
    get() = controller.getRecording()

  override val polls: DytePoll
    get() = controller.getPolls()

  override val meta: DyteMeta
    get() = controller.getMeta()

  override val plugins: DytePlugins
    get() = controller.getPlugins()

  override val liveStream: DyteLiveStream
    get() = controller.getLiveStream()

  override val stage: DyteStage
    get() = controller.stage

  override fun init(dyteMeetingInfo: DyteMeetingInfo) {
    serialScope.launch { controller.init(dyteMeetingInfo) }
  }

  override fun init(dyteMeetingInfo: DyteMeetingInfo, onSuccess: () -> Unit, onFailed: () -> Unit) {
    serialScope.launch {
      try {
        controller.init(dyteMeetingInfo)
        onSuccess.invoke()
      } catch (e: Exception) {
        onFailed.invoke()
      }
    }
  }

  override fun init(dyteMeetingInfo: DyteMeetingInfoV2) {
    serialScope.launch { controller.init(dyteMeetingInfo) }
  }

  override fun init(
    dyteMeetingInfo: DyteMeetingInfoV2,
    onInitCompleted: () -> Unit,
    onInitFailed: () -> Unit
  ) {
    serialScope.launch {
      try {
        controller.init(dyteMeetingInfo)
        onInitCompleted.invoke()
      } catch (e: Exception) {
        onInitFailed.invoke()
      }
    }
  }

  override fun joinRoom() {
    serialScope.launch { controller.joinRoom() }
  }

  override fun joinRoom(onSuccess: () -> Unit, onFailed: () -> Unit) {
    serialScope.launch {
      try {
        controller.joinRoom()
        withContext(Dispatchers.Main) { onSuccess.invoke() }
      } catch (e: Exception) {
        withContext(Dispatchers.Main) { onFailed.invoke() }
        e.printStackTrace()
        DyteLogger.error("DyteMeeting::join_room_error ${e.message}", e)
      }
    }
  }

  override fun leaveRoom() {
    serialScope.launch { controller.leaveRoom() }
  }

  override fun leaveRoom(onSuccess: () -> Unit, onFailed: () -> Unit) {
    serialScope.launch {
      try {
        controller.leaveRoom()
        withContext(Dispatchers.Main) { onSuccess.invoke() }
      } catch (e: Exception) {
        withContext(Dispatchers.Main) { onFailed.invoke() }
        DyteLogger.error("DyteMeeting::leave_room_error", e)
      }
    }
  }

  override fun release() {
    serialScope.launch { controller.release() }
  }

  override fun addMeetingRoomEventsListener(roomEventListener: DyteMeetingRoomEventsListener) {
    controller.socketController.addMainThreadListener(roomEventListener)
    /*
      Setup listeners on both for backward compatibility. Connection related events
      is a meta responsibility, therefore init and join related events
      should not be combined with connection events in one listener interface
    */
    controller.eventController.addRoomEventListener(roomEventListener)
  }

  override fun removeMeetingRoomEventsListener(roomEventListener: DyteMeetingRoomEventsListener) {
    controller.socketController.removeMainThreadListener(roomEventListener)
    controller.eventController.removeRoomEventListener(roomEventListener)
  }

  override fun addSelfEventsListener(selfEventsListener: DyteSelfEventsListener) {
    controller.selfEventListeners.addMainThreadListener(selfEventsListener)
  }

  override fun removeSelfEventsListener(selfEventsListener: DyteSelfEventsListener) {
    controller.selfEventListeners.removeMainThreadListener(selfEventsListener)
  }

  override fun addParticipantEventsListener(
    participantEventsListener: DyteParticipantEventsListener
  ) {
    controller.participantEventListener.addMainThreadListener(participantEventsListener)
  }

  override fun removeParticipantEventsListener(
    participantEventsListener: DyteParticipantEventsListener
  ) {
    controller.participantEventListener.removeMainThreadListener(participantEventsListener)
  }

  override fun addPluginEventsListener(pluginEventsListener: DytePluginEventsListener) {
    controller.pluginsEventsListener.addMainThreadListener(pluginEventsListener)
  }

  override fun removePluginEventsListener(pluginEventsListener: DytePluginEventsListener) {
    controller.pluginsEventsListener.removeMainThreadListener(pluginEventsListener)
  }

  override fun addLiveStreamEventsListener(liveStreamEventsListener: DyteLiveStreamEventsListener) {
    controller.eventController.addLiveStreamEventListener(liveStreamEventsListener)
  }

  override fun removeLiveStreamEventsListener(
    liveStreamEventsListener: DyteLiveStreamEventsListener
  ) {
    controller.eventController.removeLiveStreamEventListener(liveStreamEventsListener)
  }

  override fun addWaitlistEventsListener(waitlistEventsListener: DyteWaitlistEventsListener) {
    controller.eventController.addWaitlistEventListener(waitlistEventsListener)
  }

  override fun removeWaitlistEventsListener(waitlistEventsListener: DyteWaitlistEventsListener) {
    controller.eventController.removeWaitlistEventListener(waitlistEventsListener)
  }

  override fun addStageEventsListener(stageEventsListener: DyteStageEventListener) {
    controller.stageEventListeners.addMainThreadListener(stageEventsListener)
  }

  override fun removeStageEventsListener(stageEventsListener: DyteStageEventListener) {
    controller.stageEventListeners.removeMainThreadListener(stageEventsListener)
  }

  override fun addChatEventsListener(chatEventsListener: DyteChatEventsListener) {
    controller.chatEventListeners.addMainThreadListener(chatEventsListener)
  }

  override fun removeChatEventsListener(chatEventsListener: DyteChatEventsListener) {
    controller.chatEventListeners.removeMainThreadListener(chatEventsListener)
  }

  override fun addRecordingEventsListener(recordingEventsListener: DyteRecordingEventsListener) {
    controller.recordingEventListeners.addMainThreadListener(recordingEventsListener)
  }

  override fun removeRecordingEventsListener(recordingEventsListener: DyteRecordingEventsListener) {
    controller.recordingEventListeners.removeMainThreadListener(recordingEventsListener)
  }

  override fun addPollEventsListener(pollEventsListener: DytePollEventsListener) {
    controller.pollsEventListener.addMainThreadListener(pollEventsListener)
  }

  override fun removePollEventsListener(pollEventsListener: DytePollEventsListener) {
    controller.pollsEventListener.removeMainThreadListener(pollEventsListener)
  }

  override fun addDataUpdateListener(dataUpdateListener: DyteDataUpdateListener) {
    controller.eventController.addDataUpdateListener(dataUpdateListener)
  }

  override fun removeDataUpdateListener(dataUpdateListener: DyteDataUpdateListener) {
    controller.eventController.removeDataUpdateListener(dataUpdateListener)
  }
}

/**
 * Dyte client
 *
 * @constructor Create empty Dyte client
 */
internal interface IDyteClient {
  /**
   * Initialises the meeting with the given [dyteMeetingInfo], sets up everything that is required
   * to join the meeting
   */
  fun init(dyteMeetingInfo: DyteMeetingInfo)

  /**
   * Initialises the meeting with the given [dyteMeetingInfo], sets up everything that is required
   * to join the meeting. Once init is complete, you get a callback on [onInitCompleted] or
   * [onInitFailed] if there was a failure
   */
  fun init(dyteMeetingInfo: DyteMeetingInfo, onInitCompleted: () -> Unit, onInitFailed: () -> Unit)

  /**
   * Initialises the meeting with the given [dyteMeetingInfo], sets up everything that is required
   * to join the meeting.
   */
  fun init(dyteMeetingInfo: DyteMeetingInfoV2)

  /**
   * Initialises the meeting with the given [dyteMeetingInfo], sets up everything that is required
   * to join the meeting. Once init is complete, you get a callback on [onInitCompleted] or
   * [onInitFailed] if there was a failure
   */
  fun init(
    dyteMeetingInfo: DyteMeetingInfoV2,
    onInitCompleted: () -> Unit,
    onInitFailed: () -> Unit
  )

  /** Join the meeting */
  fun joinRoom()

  /**
   * Join the meeting. Once meeting has been join you get a callback on [onRoomJoined] or
   * [onRoomJoinFailed] in case of failure
   */
  fun joinRoom(onRoomJoined: () -> Unit, onRoomJoinFailed: () -> Unit)

  /** Leave the meeting */
  fun leaveRoom()

  /**
   * Leave the meeting. Once meeting has been left you get a callback on [onRoomLeft] or
   * [onRoomLeaveFailed] in case of failure
   */
  fun leaveRoom(onRoomLeft: () -> Unit, onRoomLeaveFailed: () -> Unit)

  fun release()

  /**
   * The localUser object stores information about the local user, their current media state and
   * exposes methods to enable and disable media etc
   */
  val localUser: DyteSelfParticipant

  /**
   * The [participants] object consists of 4 lists of participants described below and methods that
   * work on multiple participants
   * - **waitlisted** - List of participant objects that are waiting to be let in
   * - **joined** - List of participant objects that are in the meeting right now
   * - **active** - List of participant objects that should be rendered on the screen right now
   * - **pinned** - List of participant objects that are marked pinned
   *
   * @return [DyteParticipants]
   */
  val participants: DyteParticipants

  /**
   * The [chat] object stores the chat messages that were sent in the meeting. This includes text
   * messages, images, and files.
   */
  val chat: DyteChat

  /**
   * The [recording] object stores the current state of meeting recording and also exposed methods
   * to start and stop recording
   */
  val recording: DyteRecording

  /**
   * The [polls] object stores the polls created in the meeting and also exposes methods to create
   * and vote on polls.
   */
  val polls: DytePoll

  /** The [meta] object stores meeting metadata information like title, startedAt time etc */
  val meta: DyteMeta

  /**
   * The [plugins] object consists of 2 lists of plugins described below
   * - **active** - List of plugins that are active in the meeting right now
   * - **all** - List of all available plugins
   */
  val plugins: DytePlugins

  val stage: DyteStage

  val liveStream: DyteLiveStream

  fun addMeetingRoomEventsListener(meetingRoomEventsListener: DyteMeetingRoomEventsListener)

  fun removeMeetingRoomEventsListener(meetingRoomEventsListener: DyteMeetingRoomEventsListener)

  fun addSelfEventsListener(selfEventsListener: DyteSelfEventsListener)

  fun removeSelfEventsListener(selfEventsListener: DyteSelfEventsListener)

  fun addParticipantEventsListener(participantEventsListener: DyteParticipantEventsListener)

  fun removeParticipantEventsListener(participantEventsListener: DyteParticipantEventsListener)

  fun addPluginEventsListener(pluginEventsListener: DytePluginEventsListener)

  fun removePluginEventsListener(pluginEventsListener: DytePluginEventsListener)

  fun addLiveStreamEventsListener(liveStreamEventsListener: DyteLiveStreamEventsListener)

  fun removeLiveStreamEventsListener(liveStreamEventsListener: DyteLiveStreamEventsListener)

  fun addWaitlistEventsListener(waitlistEventsListener: DyteWaitlistEventsListener)

  fun removeWaitlistEventsListener(waitlistEventsListener: DyteWaitlistEventsListener)

  fun addStageEventsListener(stageEventsListener: DyteStageEventListener)

  fun removeStageEventsListener(stageEventsListener: DyteStageEventListener)

  fun addChatEventsListener(chatEventsListener: DyteChatEventsListener)

  fun removeChatEventsListener(chatEventsListener: DyteChatEventsListener)

  fun addRecordingEventsListener(recordingEventsListener: DyteRecordingEventsListener)

  fun removeRecordingEventsListener(recordingEventsListener: DyteRecordingEventsListener)

  fun addPollEventsListener(pollEventsListener: DytePollEventsListener)

  fun removePollEventsListener(pollEventsListener: DytePollEventsListener)

  fun addDataUpdateListener(dataUpdateListener: DyteDataUpdateListener)

  fun removeDataUpdateListener(dataUpdateListener: DyteDataUpdateListener)
}
