package io.dyte.core.plugins.roomnode

import io.dyte.core.Result
import io.dyte.core.feat.DytePlugin
import io.dyte.core.network.info.PluginPermissions
import io.dyte.core.plugins.PluginMessage
import io.dyte.core.plugins.PluginsController
import io.dyte.core.plugins.PluginsInternalRoomJoinListenerStore
import io.dyte.core.socket.IRoomNodeSocketService
import io.dyte.core.socket.ISocketMessageResponseParser
import io.dyte.core.socket.SocketMessageEventListener
import io.dyte.core.socket.events.InboundMeetingEventType
import io.dyte.core.socket.events.OutboundMeetingEventType
import io.dyte.core.socket.events.OutboundMeetingEventType.GET_ROOM_STATE
import io.dyte.core.socket.events.payloadmodel.InboundMeetingEvent
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketPluginDisabled
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketPluginEnabled
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketPluginEvent
import io.dyte.core.socket.events.payloadmodel.outbound.WebSocketRoomStateModel
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import kotlinx.serialization.Serializable
import kotlinx.serialization.json.Json
import kotlinx.serialization.json.JsonObject
import kotlinx.serialization.json.buildJsonObject
import kotlinx.serialization.json.encodeToJsonElement
import kotlinx.serialization.json.jsonObject
import kotlinx.serialization.json.jsonPrimitive
import kotlinx.serialization.json.put
import kotlinx.serialization.json.putJsonObject

internal class RoomNodePluginsController(
  private val json: Json,
  private val roomNodeSocket: IRoomNodeSocketService,
  private val socketIoMessageResponseParser: ISocketMessageResponseParser,
  private val selfParticipantId: String,
  scope: CoroutineScope,
  selfPluginsPermissions: PluginPermissions,
  pluginsInternalRoomJoinListenerStore: PluginsInternalRoomJoinListenerStore,
  plugins: List<DytePlugin>
) :
  PluginsController(scope, selfPluginsPermissions, pluginsInternalRoomJoinListenerStore, plugins) {
  private val pluginSocketMessageEventListener =
    object : SocketMessageEventListener {
      override suspend fun onMessageEvent(event: InboundMeetingEvent) {
        scope.launch { onPluginSocketEvent(event) }
      }
    }

  init {
    addPluginSocketMessageListeners()
    pluginsInternalRoomJoinListenerStore.addListener(this)
  }

  private fun addPluginSocketMessageListeners() {
    roomNodeSocket.addMessageEventListener(
      InboundMeetingEventType.WEB_SOCKET_PLUGIN_ENABLED,
      pluginSocketMessageEventListener
    )
    roomNodeSocket.addMessageEventListener(
      InboundMeetingEventType.WEB_SOCKET_PLUGIN_DISABLED,
      pluginSocketMessageEventListener
    )
    roomNodeSocket.addMessageEventListener(
      InboundMeetingEventType.WEB_SOCKET_PLUGIN_DATA,
      pluginSocketMessageEventListener
    )
    roomNodeSocket.addMessageEventListener(
      InboundMeetingEventType.WEB_SOCKET_PLUGIN_EVENT,
      pluginSocketMessageEventListener
    )
  }

  override suspend fun onPluginMessage(pluginId: String, pluginMessage: PluginMessage) {
    withContext(scope.coroutineContext) {
      val messageJson = pluginMessage.message
      val type = messageJson["type"]?.jsonPrimitive?.content
      when (type) {
        "pluginEvent",
        "storePluginData",
        "getPluginData",
        "enablePluginForAll",
        "enablePluginForUser",
        "disablePluginForUser",
        "chatMessage" -> {
          val response = roomNodeSocket.sendMessageSyncGeneric(messageJson)
          val jsonResponse = json.parseToJsonElement(response)
          submitEventToPlugin(pluginId, jsonResponse.jsonObject)
        }
        "getJoinedPeers" -> {
          val roomStateResponse = roomNodeSocket.sendMessage(GET_ROOM_STATE, null)
          val roomState =
            (socketIoMessageResponseParser.parseResponse(roomStateResponse).payload
                as WebSocketRoomStateModel)
              .roomState!!
          val peers = roomState.peers ?: emptyList()
          val pluginEvent = buildJsonObject {
            put("type", "websocket/get-joined-peers")
            putJsonObject("payload") {
              messageJson["payload"]?.let { payloadElement ->
                payloadElement.jsonObject.entries.forEach { put(it.key, it.value) }
              }
              put("peers", json.encodeToJsonElement(peers))
            }
          }

          submitEventToPlugin(pluginId, pluginEvent)
        }
        "getPeerInfo" -> {
          val peerId =
            messageJson["payload"]?.jsonObject?.get("peerId")?.jsonPrimitive?.content
              ?: selfParticipantId
          val socketPayload = buildJsonObject {
            put("type", type)
            putJsonObject("payload") {
              messageJson["payload"]?.let { payloadElement ->
                payloadElement.jsonObject.entries.forEach { put(it.key, it.value) }
              }
              put("peerId", peerId)
            }
          }
          val response = roomNodeSocket.sendMessageSyncGeneric(socketPayload)
          val jsonResponse = json.parseToJsonElement(response)
          submitEventToPlugin(pluginId, jsonResponse.jsonObject)
        }
        "getRoomState" -> {
          val roomStateResponse = roomNodeSocket.sendMessage(GET_ROOM_STATE, null)
          (socketIoMessageResponseParser.parseResponse(roomStateResponse).payload
              as WebSocketRoomStateModel)
            .roomState
            ?.let {
              val roomState = it
              val peers = roomState.peers ?: emptyList()
              roomState.plugins
                ?.find { it.id == pluginId }
                ?.enabledBy
                ?.let {
                  val enabledBy = it
                  peers
                    .find { it.userId == enabledBy }
                    ?.id
                    ?.let {
                      val enablerPeerId = it
                      val payload = buildJsonObject {
                        put("type", "websocket/get-room-state")
                        putJsonObject("payload") {
                          messageJson["payload"]?.let { payloadElement ->
                            payloadElement.jsonObject.entries.forEach { put(it.key, it.value) }
                          }
                          put("roomName", roomState.roomName)
                          put("pluginInitiatorPeerId", enablerPeerId)
                          put("pluginInitiatorUserId", enabledBy)
                          put("peers", json.encodeToJsonElement(peers))
                          put("displayTitle", roomState.displayTitle)
                          put("roomUUID", roomState.roomUUID)
                          put("currentPeer", selfParticipantId)
                        }
                      }
                      submitEventToPlugin(pluginId, payload)
                    }
                }
            }
        }
        "getPluginInitiatorId" -> {
          val roomStateResponse = roomNodeSocket.sendMessage(GET_ROOM_STATE, null)
          (socketIoMessageResponseParser.parseResponse(roomStateResponse).payload
              as WebSocketRoomStateModel)
            .roomState
            ?.let {
              val roomState = it
              val peers = roomState.peers ?: emptyList()
              roomState.plugins
                ?.find { it.id == pluginId }
                ?.enabledBy
                ?.let {
                  val enabledBy = it
                  peers
                    .find { it.userId == enabledBy }
                    ?.id
                    ?.let {
                      val enablerPeerId = it
                      val payload = buildJsonObject {
                        put("type", "websocket/get-plugin-initiator-peer-id")
                        putJsonObject("payload") {
                          messageJson["payload"]?.let { payloadElement ->
                            payloadElement.jsonObject.entries.forEach { put(it.key, it.value) }
                          }
                          put("pluginInitiatorPeerId", enablerPeerId)
                        }
                      }
                      submitEventToPlugin(pluginId, payload)
                    }
                }
            }
        }
        "getPluginInitiatorUserId" -> {
          val roomStateResponse = roomNodeSocket.sendMessage(GET_ROOM_STATE, null)
          (socketIoMessageResponseParser.parseResponse(roomStateResponse).payload
              as WebSocketRoomStateModel)
            .roomState
            ?.let {
              val roomState = it
              roomState.plugins
                ?.find { it.id == pluginId }
                ?.enabledBy
                ?.let {
                  val enabledBy = it
                  val payload = buildJsonObject {
                    put("type", "websocket/get-plugin-initiator-user-id")
                    putJsonObject("payload") {
                      messageJson["payload"]?.let { payloadElement ->
                        payloadElement.jsonObject.entries.forEach { put(it.key, it.value) }
                      }
                      put("pluginInitiatorUserId", enabledBy)
                    }
                  }
                  submitEventToPlugin(pluginId, payload)
                }
            }
        }
        "pluginEventToParent" -> {}
        else -> {
          // no-op
        }
      }
    }
  }

  override suspend fun loadActivePlugins() {
    val roomStateResponse = roomNodeSocket.sendMessage(GET_ROOM_STATE, null)
    val roomPlugins =
      (socketIoMessageResponseParser.parseResponse(roomStateResponse).payload
          as WebSocketRoomStateModel)
        .roomState
        ?.plugins
        ?: return

    // remove the deactivated local plugins, if any
    val roomActivePluginIds = roomPlugins.map { it.id }.toSet()
    for (localActivePluginId in _activePlugins.keys) {
      if (!roomActivePluginIds.contains(localActivePluginId)) {
        disablePluginLocallyAndNotify(localActivePluginId)
      }
    }

    // populate the active plugins
    roomPlugins.forEach { roomPlugin ->
      enablePluginLocallyAndNotify(roomPlugin.id, roomPlugin.enabledBy)
    }
  }

  override suspend fun addPlugin(pluginId: String, staggered: Boolean): Result<Unit, Exception> {
    val activatePluginPayload = buildJsonObject {
      put("id", pluginId)
      put("staggered", staggered)
    }

    val responseString =
      try {
        roomNodeSocket.sendMessage(OutboundMeetingEventType.ADD_ROOM_PLUGIN, activatePluginPayload)
      } catch (e: Exception) {
        logger.error("PluginsController::addPlugin::socket error", e)
        return Result.Failure(e)
      }

    val response =
      try {
        json.decodeFromString<RoomNodeSocketResponse>(responseString)
      } catch (e: Exception) {
        /*
         * note(swapnil): catching parsing exception separately so that it will be convenient to convert
         * it into invalid response related DyteError later.
         * */
        logger.error("PluginsController::addPlugin::response parse error", e)
        return Result.Failure(e)
      }

    return if (response.isSuccess()) {
      logger.info("PluginsController::addPlugin::success")
      Result.Success(Unit)
    } else {
      logger.error("PluginsController::addPlugin::err response")
      Result.Success(Unit) // Expected: Result.Failure(Exception("Failed to activate plugin"))
    }
  }

  override suspend fun removePlugin(pluginId: String): Result<Unit, Exception> {
    val deactivatePluginPayload = buildJsonObject { put("id", pluginId) }

    val responseString =
      try {
        roomNodeSocket.sendMessage(
          OutboundMeetingEventType.REMOVE_ROOM_PLUGIN,
          deactivatePluginPayload
        )
      } catch (e: Exception) {
        logger.error("PluginsController::removePlugin::socket error", e)
        return Result.Failure(e)
      }

    val response =
      try {
        json.decodeFromString<RoomNodeSocketResponse>(responseString)
      } catch (e: Exception) {
        /*
         * note(swapnil): catching parsing exception separately so that it will be convenient to convert
         * it into invalid response related DyteError later.
         * */
        logger.error("PluginsController::removePlugin::response parse error", e)
        return Result.Failure(e)
      }

    return if (response.isSuccess()) {
      logger.info("PluginsController::removePlugin::success")
      Result.Success(Unit)
    } else {
      logger.error("PluginsController::removePlugin::err response")
      Result.Success(Unit) // Expected: Result.Failure(Exception("Failed to close plugin"))
    }
  }

  override suspend fun onCleared() {
    pluginsInternalRoomJoinListenerStore.removeListener()
    super.onCleared()
  }

  private suspend fun onPluginSocketEvent(event: InboundMeetingEvent) {
    when (event.eventType) {
      InboundMeetingEventType.WEB_SOCKET_PLUGIN_ENABLED -> {
        onEnablePlugin(event.payload as WebSocketPluginEnabled)
      }
      InboundMeetingEventType.WEB_SOCKET_PLUGIN_DISABLED -> {
        onDisablePlugin(event.payload as WebSocketPluginDisabled)
      }
      InboundMeetingEventType.WEB_SOCKET_PLUGIN_DATA,
      InboundMeetingEventType.WEB_SOCKET_PLUGIN_EVENT -> {
        onPluginSocketMessage(event.eventType.type, event.payload as WebSocketPluginEvent)
      }
      else -> {
        // TODO: log invalid plugin socket event
      }
    }
  }

  private suspend fun onEnablePlugin(payload: WebSocketPluginEnabled) {
    if (payload.id != null && payload.enabledBy != null) {
      // Return if plugin is already active
      var pluginActive = false
      _activePlugins[payload.id]?.let { plugin ->
        if (plugin.isActive) {
          pluginActive = true
        }
      }

      if (!pluginActive) {
        enablePluginLocallyAndNotify(payload.id!!, payload.enabledBy!!)
      }
    }
  }

  private suspend fun onDisablePlugin(payload: WebSocketPluginDisabled) {
    val pluginId = payload.id ?: return
    disablePluginLocallyAndNotify(pluginId)
  }

  private fun onPluginSocketMessage(type: String, payload: WebSocketPluginEvent) {
    payload.payload["id"]?.jsonPrimitive?.content?.let {
      val pluginPayload = buildJsonObject {
        put("type", type)
        put("payload", payload.payload)
      }
      submitEventToPlugin(it, pluginPayload)
    }
  }
}

@Serializable
internal data class RoomNodeSocketResponse(val type: String, val payload: JsonObject) {
  fun isSuccess(): Boolean {
    return (type == TYPE_NORMAL && !payload.containsKey(PAYLOAD_KEY_ERR))
  }

  companion object {
    internal const val TYPE_NORMAL = "websocket/resolve"
    internal const val PAYLOAD_KEY_ERR = "err"
  }
}
