package io.dyte.core.network

import io.dyte.core.network.info.MeetingSessionInfo
import io.dyte.core.network.info.ParticipantInfo
import io.dyte.core.network.info.RecordingInfo
import io.dyte.core.network.models.ActiveLiveStreamResponse
import io.dyte.core.network.models.GraphQlRequest
import io.dyte.core.network.models.GraphQlRequestVariables
import io.dyte.core.network.models.IceServersWrapper
import io.dyte.core.network.models.IpDetailsResponsePayload
import io.dyte.core.network.models.MeetingSessionDataResponseV1
import io.dyte.core.network.models.MeetingSessionDataWrapper
import io.dyte.core.network.models.MultiplePluginResponse
import io.dyte.core.network.models.PluginAuthResponse
import io.dyte.core.network.models.PluginAuthorizationModel
import io.dyte.core.network.models.PluginConfigResponse
import io.dyte.core.network.models.PluginResponse
import io.dyte.core.network.models.RecordingResponseWrapper
import io.dyte.core.network.models.RecordingResponseWrapperV2
import io.dyte.core.network.models.StartLiveStreamResponse
import io.dyte.core.network.models.StartRecordingModel
import io.dyte.core.network.models.StopLiveStreamResponse
import io.dyte.core.network.models.StopRecordingModel
import io.dyte.core.network.models.StopRecordingModelV2
import io.dyte.core.network.models.UserDataWrapper
import io.dyte.core.network.models.UserDataWrapperV1
import io.dyte.core.network.models.UserPresetDataWrapper
import io.dyte.core.network.models.UserPresetRequestModel
import io.dyte.core.observability.PostDyteLogs
import io.ktor.client.HttpClient
import io.ktor.client.call.body
import io.ktor.client.plugins.HttpTimeout
import io.ktor.client.plugins.contentnegotiation.ContentNegotiation
import io.ktor.client.plugins.defaultRequest
import io.ktor.client.request.get
import io.ktor.client.request.header
import io.ktor.client.request.post
import io.ktor.client.request.put
import io.ktor.client.request.setBody
import io.ktor.client.statement.bodyAsText
import io.ktor.http.ContentType
import io.ktor.http.contentType
import io.ktor.serialization.kotlinx.json.json
import kotlinx.serialization.json.Json
import kotlinx.serialization.json.buildJsonObject
import kotlinx.serialization.json.encodeToJsonElement

// TODO : get rid of auth headers from all requests
private const val VERSION = "0.5.0"
private const val CLIENT_TYPE = "CLIENT_APP"
private const val GRAPHQL_REQUEST_QUERY: String =
  "\nquery Session(" +
    "$" +
    "roomName: String!, " +
    "$" +
    "password: String) {\n\tsession(roomName: " +
    "$" +
    "roomName, password: " +
    "$" +
    "password) {\n\t  title,\n\t  roomNodeLink,\n\t  roomName,\n\t  password\n\t}\n  }\n"

internal class ApiClient(
  baseDomain: String,
  private val authToken: String,
  private val peerId: String,
  private val roomName: String,
  private val meetingId: String,
  private val orgId: String,
  private val isV2AuthToken: Boolean
) : IApiClient {
  private val baseUrl: String = "https://api.${baseDomain}"

  internal val client: HttpClient = HttpClient {
    install(ContentNegotiation) {
      json(
        Json {
          prettyPrint = true
          isLenient = true
          ignoreUnknownKeys = true
        }
      )
    }
    install(HttpTimeout) {
      socketTimeoutMillis = 30000
      requestTimeoutMillis = 30000
      connectTimeoutMillis = 30000
    }

    defaultRequest {
      header("Authorization", "Bearer $authToken")
      header("dyte-tracing-id", peerId)
      contentType(ContentType.Application.Json)
    }
  }

  val json = Json { ignoreUnknownKeys = true }

  @Throws(Exception::class)
  override suspend fun getRoomNodeData(): MeetingSessionInfo {
    val ipInformation: IpDetailsResponsePayload? =
      try {
        getIpDetails(peerId)
      } catch (e: Exception) {
        null
      }
    var ipInformationPayload = buildJsonObject {}
    ipInformation?.let {
      ipInformationPayload = buildJsonObject {
        put("ip_information", BaseApiService.json.encodeToJsonElement(ipInformation))
      }
    }
    return if (isV2AuthToken) {
      val data =
        client
          .post("${baseUrl}/v2/internals/rooms") {
            setBody(ipInformationPayload)
            contentType(ContentType.Application.Json)
          }
          .body<MeetingSessionDataWrapper>()
      MeetingSessionInfo(requireNotNull(data.data.title), data.data.roomNodeLink, data.data.useHive)
    } else {
      val graphQlRequestVariables =
        GraphQlRequestVariables(
          roomName,
          "",
          BaseApiService.json.encodeToString(
            IpDetailsResponsePayload.serializer(),
            ipInformation ?: IpDetailsResponsePayload()
          )
        )
      // val requestData = GraphQlRequest(GRAPHQL_REQUEST_QUERY, graphQlRequestVariables)
      val requestData = GraphQlRequest(GRAPHQL_ROOM_NODE_DATA_QUERY, graphQlRequestVariables)
      val response =
        client
          .post("${baseUrl}/graphql") {
            setBody(requestData)
            contentType(ContentType.Application.Json)
          }
          .body<MeetingSessionDataResponseV1>()
      MeetingSessionInfo(
        requireNotNull(response.data.session.title),
        requireNotNull(response.data.session.roomNodeLink)
      )
    }
  }

  // TODO : fix this
  @Throws(Exception::class)
  override suspend fun getICEServers(): IceServersWrapper {
    val response = client.get("${baseUrl}/iceservers")
    return response.body()
  }

  @Throws(Exception::class)
  override suspend fun getParticipantInfo(): ParticipantInfo {
    return if (isV2AuthToken) {
      val userData =
        client.get("${baseUrl}/v2/internals/participant-details").body<UserDataWrapper>()
      ParticipantInfo.getParticipantInfoFromV2(userData)
    } else {
      val userDataWrapper =
        client
          .get("${baseUrl}/auth/basicUserDetails?authToken=${authToken}")
          .body<UserDataWrapperV1>()
      val userPresetDataWrapper = getUserPreset()
      ParticipantInfo.getParticipantInfoFromV1(userDataWrapper, userPresetDataWrapper)
    }
  }

  @Throws(Exception::class)
  override suspend fun getUserPreset(): UserPresetDataWrapper {
    val roomName = roomName
    val response =
      client.post("${baseUrl}/v1/userpreset") {
        setBody(UserPresetRequestModel(authToken, CLIENT_TYPE, roomName, VERSION))
        contentType(ContentType.Application.Json)
      }
    return response.body()
  }

  @Throws(Exception::class)
  override suspend fun startRecording(): RecordingInfo {
    val recordingInfo: RecordingInfo
    if (isV2AuthToken) {
      val response =
        client
          .post("${baseUrl}/v2/recordings") {
            setBody(StartRecordingModel(meetingId))
            contentType(ContentType.Application.Json)
          }
          .body<RecordingResponseWrapperV2>()
      recordingInfo = RecordingInfo(response.data.id)
    } else {
      val response =
        client
          .post("${baseUrl}/v1/organizations/${orgId}/rooms/${roomName}/recording") {
            contentType(ContentType.Application.Json)
          }
          .body<RecordingResponseWrapper>()
      recordingInfo = RecordingInfo(response.data.recording.id)
    }
    return recordingInfo
  }

  @Throws(Exception::class)
  override suspend fun stopRecording(recordingInfo: RecordingInfo) {
    if (isV2AuthToken) {
      client.put("${baseUrl}/v2/recordings/${recordingInfo.id}") {
        setBody(StopRecordingModelV2("stop"))
        contentType(ContentType.Application.Json)
      }
    } else {
      client.put(
        "${baseUrl}/v1/organizations/${orgId}/rooms/${roomName}/recordings/${recordingInfo.id}"
      ) {
        setBody(StopRecordingModel("stop"))
        contentType(ContentType.Application.Json)
      }
    }
  }

  @Throws(Exception::class)
  override suspend fun getActiveRecording(): RecordingInfo {
    return if (isV2AuthToken) {
      val response =
        client
          .get("${baseUrl}/v2/recordings/active-recording/${meetingId}") {
            contentType(ContentType.Application.Json)
          }
          .body<RecordingResponseWrapperV2>()
      RecordingInfo(response.data.id)
    } else {
      val response =
        client
          .get("${baseUrl}/v1/organizations/${orgId}/rooms/${roomName}/active-recording/") {
            contentType(ContentType.Application.Json)
          }
          .body<RecordingResponseWrapper>()
      RecordingInfo(response.data.recording.id)
    }
  }

  @Throws(Exception::class)
  override suspend fun getPlugins(): MultiplePluginResponse {
    val pluginsUrl = "${baseUrl}/v2/plugins/user"
    val response = client.get(pluginsUrl) { contentType(ContentType.Application.Json) }
    return response.body()
  }

  // TODO : fix this endpoint
  @Throws(Exception::class)
  override suspend fun getPluginDetails(pluginId: String): PluginResponse {
    val pluginDetailsUrl = "${baseUrl}/v2/plugins/view/$pluginId"
    val response = client.get(pluginDetailsUrl) { contentType(ContentType.Application.Json) }
    return response.body()
  }

  // TODO : fix this endpoint
  @Throws(Exception::class)
  override suspend fun getPluginConfig(pluginBaseUrl: String): PluginConfigResponse {
    val response =
      client.get("${pluginBaseUrl}/dyte-config.json") { contentType(ContentType.Application.Json) }
    return response.body()
  }

  // TODO : fix this endpoint
  @Throws(Exception::class)
  override suspend fun authorizePlugin(pluginId: String): PluginAuthResponse {
    val authorizePluginUrl = "${baseUrl}/v2/plugins/authorize/$pluginId"
    val response =
      client.post(authorizePluginUrl) {
        contentType(ContentType.Application.Json)
        setBody(PluginAuthorizationModel(roomName, peerId))
      }
    return response.body()
  }

  @Throws(Exception::class)
  override suspend fun uploadLogs(logs: PostDyteLogs) {
    client.post("https://api-silos.dyte.io/otel/logs") {
      contentType(ContentType.Application.Json)
      setBody(logs)
    }
  }

  @Throws(Exception::class)
  override suspend fun getLiveStreamUrl(): String {
    val response =
      client
        .get("${baseUrl}/v2/meetings/${meetingId}/active-livestream")
        .body<ActiveLiveStreamResponse>()
    return response.data.url
  }

  @Throws(Exception::class)
  override suspend fun startLiveStream() {
    client.post("${baseUrl}/v2/meetings/${meetingId}/livestreams").body<StartLiveStreamResponse>()
  }

  @Throws(Exception::class)
  override suspend fun stopLiveStream() {
    client
      .post("${baseUrl}/v2/meetings/${meetingId}/active-livestream/stop")
      .body<StopLiveStreamResponse>()
  }

  override fun getClient(): HttpClient {
    return client
  }

  @Throws(Exception::class)
  override suspend fun getIpDetails(peerId: String): IpDetailsResponsePayload {
    val response = client.get("https://media-location.dyte.io")
    if (response.status.value in 200..299) {
      // Workaround for deserializing as the response doesn't have [application/json; charset=utf-8]
      // content-type set
      val ipDetailsResponsePayload =
        BaseApiService.json.decodeFromString(
          deserializer = IpDetailsResponsePayload.serializer(),
          string = response.bodyAsText()
        )

      // Is this check really needed? And shouldn't we check if location has both longitude as well
      // as latitude?
      if ((ipDetailsResponsePayload.location?.length ?: 6) > 5) {
        return ipDetailsResponsePayload
      }
      throw Exception("Insufficient data")
    }

    throw Exception("Failed to get IP Details")
  }

  override fun getApiBaseUrl(): String {
    return baseUrl
  }

  companion object {
    private const val GRAPHQL_ROOM_NODE_DATA_QUERY =
      """
      query Session(${'$'}roomName: String!, ${'$'}password: String, ${'$'}ipInformation: String) {
        session(roomName: ${'$'}roomName, password: ${'$'}password) {
          title,
          roomNodeLink(ipInformation: ${'$'}ipInformation),
          useHiveMedia,
          roomName,
          password
        }
      }
    """
  }
}

interface IApiClient {
  @Throws(Exception::class) suspend fun getRoomNodeData(): MeetingSessionInfo

  @Throws(Exception::class) suspend fun getParticipantInfo(): ParticipantInfo

  @Throws(Exception::class) suspend fun getUserPreset(): UserPresetDataWrapper

  @Throws(Exception::class) suspend fun getICEServers(): IceServersWrapper

  @Throws(Exception::class) suspend fun startRecording(): RecordingInfo

  @Throws(Exception::class) suspend fun stopRecording(recordingInfo: RecordingInfo)

  @Throws(Exception::class) suspend fun getActiveRecording(): RecordingInfo

  @Throws(Exception::class) suspend fun getPlugins(): MultiplePluginResponse

  @Throws(Exception::class) suspend fun getPluginDetails(pluginId: String): PluginResponse

  @Throws(Exception::class) suspend fun getPluginConfig(pluginBaseUrl: String): PluginConfigResponse

  @Throws(Exception::class) suspend fun authorizePlugin(pluginId: String): PluginAuthResponse

  @Throws(Exception::class) suspend fun uploadLogs(logs: PostDyteLogs)

  @Throws(Exception::class) suspend fun getLiveStreamUrl(): String

  @Throws(Exception::class) suspend fun startLiveStream()

  @Throws(Exception::class) suspend fun stopLiveStream()

  fun getClient(): HttpClient

  @Throws(Exception::class) suspend fun getIpDetails(peerId: String): IpDetailsResponsePayload

  fun getApiBaseUrl(): String
}
