package io.dyte.core.plugins

import io.dyte.core.Result
import io.dyte.core.events.EventEmitter
import io.dyte.core.feat.DytePlugin
import io.dyte.core.listeners.DytePluginEventsListener
import io.dyte.core.network.info.PluginPermissions
import io.dyte.core.observability.DyteLogger
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import kotlinx.serialization.json.JsonObject

internal abstract class PluginsController(
  protected val scope: CoroutineScope,
  private val selfPluginsPermissions: PluginPermissions,
  protected val pluginsInternalRoomJoinListenerStore: PluginsInternalRoomJoinListenerStore,
  plugins: List<DytePlugin>
) :
  EventEmitter<DytePluginEventsListener>(),
  PluginHostMeeting,
  PluginFileProvider,
  PluginsInternalRoomJoinListener {
  protected val _allPlugins = mutableMapOf<String, DytePlugin>()
  protected val _activePlugins = mutableMapOf<String, DytePlugin>()

  protected val logger: DyteLogger = DyteLogger

  val allPlugins: List<DytePlugin>
    get() = _allPlugins.values.toList()

  val activePlugins: List<DytePlugin>
    get() = _activePlugins.values.toList()

  init {
    // populate _allPlugins map
    plugins.forEach { plugin ->
      _allPlugins[plugin.id] = plugin
      plugin.setHostMeeting(this)
      plugin.setFileProvider(this)
    }
  }

  override fun onRoomJoined() {
    scope.launch {
      try {
        loadActivePlugins()
      } catch (e: Exception) {
        logger.error("PluginsController::loadActivePlugins::failure", e)
      }
    }
  }

  override suspend fun activatePlugin(
    pluginId: String,
    staggered: Boolean
  ): Result<Unit, Exception> {
    return withContext(scope.coroutineContext) {
      if (!selfPluginsPermissions.canLaunch) {
        logger.error("PluginHostMeeting::activatePlugin::failure permission denied")
        return@withContext Result.Failure(
          UnsupportedOperationException("Not permitted to activate plugin")
        )
      }

      return@withContext addPlugin(pluginId, staggered)
    }
  }

  override suspend fun deactivatePlugin(pluginId: String): Result<Unit, Exception> {
    return withContext(scope.coroutineContext) {
      if (!selfPluginsPermissions.canClose) {
        logger.error("PluginHostMeeting::deactivatePlugin::failure permission denied")
        return@withContext Result.Failure(
          UnsupportedOperationException("Not permitted to deactivate plugin")
        )
      }

      return@withContext removePlugin(pluginId)
    }
  }

  override suspend fun onFileRequest(plugin: DytePlugin) {
    withContext(scope.coroutineContext) {
      if (_activePlugins.containsKey(plugin.id)) {
        emitEvent { it.onPluginFileRequest(plugin) }
      }
    }
  }

  @Throws(Exception::class) protected abstract suspend fun loadActivePlugins()

  protected abstract suspend fun addPlugin(
    pluginId: String,
    staggered: Boolean
  ): Result<Unit, Exception>

  protected abstract suspend fun removePlugin(pluginId: String): Result<Unit, Exception>

  protected fun submitEventToPlugin(pluginId: String, payload: JsonObject) {
    _activePlugins[pluginId]?.submitMessage(PluginMessage(payload))
  }

  protected suspend fun enablePluginLocallyAndNotify(pluginId: String, enabledBy: String) {
    _allPlugins[pluginId]?.let { plugin ->
      val result = plugin.enableLocal(enabledBy)
      if (result is Result.Success) {
        _activePlugins[plugin.id] = plugin
        emitEvent { it.onPluginActivated(plugin) }
      }
    }
  }

  protected suspend fun disablePluginLocallyAndNotify(pluginId: String) {
    _activePlugins[pluginId]?.let { plugin ->
      val result = plugin.disableLocal()
      if (result is Result.Success) {
        _activePlugins.remove(plugin.id)
        emitEvent { it.onPluginDeactivated(plugin) }
      }
    }
  }

  /** Called when this PluginsController is no longer used and will be destroyed. */
  internal open suspend fun onCleared() {
    withContext(scope.coroutineContext) { _allPlugins.values.forEach { plugin -> plugin.clear() } }
  }
}

/*
 * note(swapnil): Used by the RoomNodeController to notify Plugins feature after joinRoom in complete without
 * exposing the other Plugin APIs.
 * Ideally there should be a separate class to manage these Room event or internal events to which all
 * interested classes can subscribe and get notified. That class can be used by the RoomNodeController
 * to notify the interested components in a decoupled way.
 * */
internal interface PluginsRoomJoinListener {
  suspend fun onRoomJoined()
}
