package io.dyte.core.controllers

import io.dyte.core.models.DyteMeetingType
import io.dyte.core.network.info.ParticipantInfo

/**
 * Meta controller
 *
 * Contains meta details about the meeting, example: [DyteMeetingInfoV1]
 *
 * @constructor Create empty Meta controller
 */
internal class MetaController(
  val meetingInfo: MeetingInfo,
  val controllerContainer: IControllerContainer
) : IMetaController {

  private lateinit var participantInfo: ParticipantInfo
  private var meetingUserPeerId: String =
    controllerContainer.platformUtilsProvider.getPlatformUtils().getUuid()
  private lateinit var roomUuid: String
  private var useHive: Boolean? = null

  open fun init() {
    meetingInfo.meetingId =
      controllerContainer.platformUtilsProvider
        .getPlatformUtils()
        .decodeAuthToken(meetingInfo.authToken)
    meetingInfo.isV2Meeting = meetingInfo.meetingId != null
  }

  override fun getDisplayName(): String {
    return participantInfo.name
  }

  override fun getRoomName(): String {
    return meetingInfo.getRoomName()
  }

  override fun getAuthToken(): String {
    return meetingInfo.authToken
  }

  override fun getBaseUrl(): String {
    return meetingInfo.baseUrl
  }

  override fun getBaseUrlWithoutVersion(): String {
    return meetingInfo.baseUrl.substring(0, meetingInfo.baseUrl.length - 3)
  }

  override fun getPeerId(): String {
    return meetingUserPeerId
  }

  override fun getOrgId(): String {
    return if (::participantInfo.isInitialized) participantInfo.organizationId else ""
  }

  override fun getMeetingTitle(): String {
    return meetingInfo.meetingTitle ?: ""
  }

  override fun getMeetingStatedTimestamp(): String {
    return meetingInfo.meetingStartedAt ?: ""
  }

  override fun getMeetingStatedTime(): Long {
    return meetingInfo.meetingStartedAtEpox ?: -1
  }

  override fun setMeetingStatedTime(time: Long) {
    meetingInfo.meetingStartedAtEpox = time
  }

  override fun setMeetingTitle(title: String) {
    meetingInfo.meetingTitle = title
  }

  override fun setMeetingStartedTimestamp(timestamp: String) {
    meetingInfo.meetingStartedAt = timestamp
  }

  override fun getMeetingId(): String {
    return meetingInfo.getRoomName()
  }

  override fun isV2AuthToken(): Boolean {
    return meetingInfo.isV2Meeting
  }

  override fun setParticipantInfo(participantInfo: ParticipantInfo) {
    this.participantInfo = participantInfo
  }

  override fun isWebinar(): Boolean {
    return participantInfo.presetInfo.viewType == "WEBINAR"
  }

  override fun isGroupCall(): Boolean {
    return participantInfo.presetInfo.viewType == "GROUP_CALL"
  }

  override fun getMeetingConfig(): MeetingConfig {
    return this.meetingInfo.config
  }

  override fun setRoomUuid(roomUuid: String) {
    this.roomUuid = roomUuid
  }

  override fun getRoomUuid(): String {
    return roomUuid
  }

  override fun getIsHive(): Boolean {
    return useHive!!
  }

  override fun setIsHive(isHive: Boolean) {
    useHive = isHive
  }

  override fun getMeetingType(): DyteMeetingType {
    return DyteMeetingType.fromViewTypeString(participantInfo.presetInfo.viewType)
  }

  override fun refreshPeerId(): String {
    val newPeerId = controllerContainer.platformUtilsProvider.getPlatformUtils().getUuid()
    meetingUserPeerId = newPeerId
    return newPeerId
  }
}

internal data class MeetingInfo(
  val authToken: String,
  val config: MeetingConfig,
  val baseUrl: String
) {
  internal var isV2Meeting: Boolean = false
  internal var meetingId: String? = null
  internal var roomName: String? = null
  internal var viewType: String = "GROUP_CALL"
  internal var meetingTitle: String? = null
  internal var meetingStartedAt: String? = null
  internal var meetingStartedAtEpox: Long? = null

  fun getRoomName(): String {
    val roomName = if (isV2Meeting) meetingId else roomName
    return if (roomName != null) roomName else ""
  }
}

data class MeetingConfig(
  val enableAudio: Boolean,
  val enableVideo: Boolean,
)

interface IMetaController {
  fun getPeerId(): String

  fun getBaseUrl(): String

  fun getBaseUrlWithoutVersion(): String

  fun getDisplayName(): String

  fun getRoomName(): String

  fun getAuthToken(): String

  fun getOrgId(): String

  fun setMeetingTitle(title: String)

  fun setMeetingStartedTimestamp(timestamp: String)

  fun getMeetingTitle(): String

  fun getMeetingStatedTimestamp(): String

  fun getMeetingStatedTime(): Long

  fun setMeetingStatedTime(time: Long)

  fun getMeetingId(): String

  fun getMeetingConfig(): MeetingConfig

  fun isWebinar(): Boolean

  fun isGroupCall(): Boolean

  fun setParticipantInfo(participantInfo: ParticipantInfo)

  fun isV2AuthToken(): Boolean

  fun setRoomUuid(roomUuid: String)

  fun getRoomUuid(): String

  fun getIsHive(): Boolean

  fun setIsHive(isHive: Boolean)

  fun getMeetingType(): DyteMeetingType

  /**
   * Generates a new peerId for local user.
   *
   * **NOTE**: This is needed as a part of a workaround that we have done to make the socket-service
   * work properly after the reconnection.
   *
   * @return new peerId
   */
  fun refreshPeerId(): String
}
