package io.dyte.core.socket.socketservice

import io.dyte.core.controllers.DyteEventType
import io.dyte.core.controllers.IControllerContainer
import io.dyte.core.controllers.StageStatus
import io.dyte.core.models.AudioDeviceType
import io.dyte.core.models.DyteJoinedMeetingParticipant
import io.dyte.core.models.DyteMediaPermission
import io.dyte.core.models.DyteMeetingType
import io.dyte.core.observability.DyteLogger
import io.dyte.core.socket.events.OutboundMeetingEventType
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketPeerLeftModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketWebinarStagePeer
import io.dyte.core.socket.events.payloadmodel.outbound.WebSocketJoinRoomModel
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.launch
import kotlinx.coroutines.newSingleThreadContext
import kotlinx.serialization.json.JsonPrimitive
import kotlinx.serialization.json.addJsonObject
import kotlinx.serialization.json.buildJsonArray
import kotlinx.serialization.json.buildJsonObject
import socket.livestreaming.DenyStageAccessRequest
import socket.livestreaming.GrantStageAccessRequest
import socket.livestreaming.LeaveStageRequest

internal class StageSocketHandler(
  private val socketService: SocketService,
  private val controllerContainer: IControllerContainer
) : IStageSocketHandler {
  val serialScope = CoroutineScope(newSingleThreadContext("StageSocketHandler${hashCode()}"))
  override val stageStatus: StageStatus
    get() = _stageStatus

  private var _stageStatus: StageStatus = StageStatus.OFF_STAGE

  private val _requestsToPresent = arrayListOf<DyteJoinedMeetingParticipant>()
  override val requestedParticipants: List<DyteJoinedMeetingParticipant>
    get() = _requestsToPresent

  /*private val _stageParticipants = arrayListOf<DyteJoinedMeetingParticipant>()
  override val stageParticipants: List<DyteJoinedMeetingParticipant>
    get() = _stageParticipants*/

  private val meetingType = controllerContainer.metaController.getMeetingType()

  /*private val socketEventListener =
  object : SocketServiceEventListener {
    override fun onEvent(event: Int, eventId: String?, payload: ByteArray?) {
      when (event) {
        // when new peer joins stage
        SocketServiceUtils.RoomEvent.GET_STAGE_PEERS.id -> {
          payload?.let {
            val response = GetStagePeersResponse.ADAPTER.decode(payload)
            _stageParticipants.clear()
            response.stage_peers.forEach { peerId ->
              val peersToRemove = _requestsToPresent.filter { it.id == peerId }
              _requestsToPresent.removeAll(peersToRemove.toSet())
              peersToRemove.forEach { peerOnStage ->
                _stageParticipants.add(DyteStageParticipant(peerOnStage.name, peerOnStage.id))
              }
            }
          }
        }
        SocketServiceUtils.RoomEvent.GRANT_STAGE_REQUEST.id -> {
          setStageStatus(StageStatus.ACCEPTED_TO_JOIN_STAGE)
        }

        // when someone requests to be on stage
        SocketServiceUtils.RoomEvent.GET_STAGE_REQUESTS.id -> {
          payload?.let {
            val response = GetStageRequestsResponse.ADAPTER.decode(payload)
            _requestsToPresent.clear()
            response.stage_requests.forEach { requestPeer ->
              val requestedPeer =
                DyteStageParticipant(requestPeer.display_name, requestPeer.user_id)
              _requestsToPresent.add(requestedPeer)
            }
          }
            ?: run {
              // empty response means no more peers requesting to be on stage
              _requestsToPresent.clear()
            }

          controllerContainer.eventController.triggerEvent(
            DyteEventType.OnStageRequestsUpdated(_requestsToPresent)
          )
        }
        else -> {}
      }
    }
  }*/

  override suspend fun requestAccess() {
    if (
      controllerContainer.presetController.permissions.media.video.permission ==
        DyteMediaPermission.CAN_REQUEST ||
        controllerContainer.presetController.permissions.media.audioPermission ==
          DyteMediaPermission.CAN_REQUEST
    ) {
      controllerContainer.socketController.sendMessage(
        OutboundMeetingEventType.REQUEST_TO_JOIN_STAGE,
        JsonPrimitive("REQUEST_TO_PRESENT")
      )
      setStageStatus(StageStatus.REQUESTED_TO_JOIN_STAGE)
    }
  }

  override suspend fun denyAccess(ids: List<String>) {
    if (meetingType == DyteMeetingType.LIVESTREAM) {
      socketService.send(
        SocketServiceUtils.RoomEvent.DENY_STAGE_REQUEST.id,
        DenyStageAccessRequest(ids).encode()
      )
    } else {
      if (controllerContainer.presetController.permissions.host.canAcceptRequests.not()) {
        DyteLogger.error("DyteWebinar::unauthorized_rejectRequest")
        throw UnsupportedOperationException("not allowed to reject webinar requests")
      }
      ids.forEach { idToReject ->
        val payload = buildJsonObject { put("id", JsonPrimitive(idToReject)) }
        controllerContainer.socketController.sendMessage(
          OutboundMeetingEventType.REJECT_PRESENTING_REQUEST,
          payload
        )
      }
    }
  }

  override suspend fun grantAccess(ids: List<String>) {
    if (meetingType == DyteMeetingType.LIVESTREAM) {
      socketService.send(
        SocketServiceUtils.RoomEvent.GRANT_STAGE_REQUEST.id,
        GrantStageAccessRequest(ids).encode()
      )
    } else {
      if (controllerContainer.presetController.permissions.host.canAcceptRequests.not()) {
        DyteLogger.error("DyteWebinar::unauthorized_acceptRequest")
        throw UnsupportedOperationException("not allowed to accept webinar requests")
      }
      val idToAccept = buildJsonArray {
        ids.forEach { addJsonObject { put("id", JsonPrimitive(it)) } }
      }
      controllerContainer.socketController.sendMessage(
        OutboundMeetingEventType.ACCEPT_PRESENTING_REQUEST,
        idToAccept
      )
    }
  }

  override suspend fun join() {
    if (meetingType == DyteMeetingType.LIVESTREAM) {
      if (stageStatus == StageStatus.OFF_STAGE) {
        setStageStatus(StageStatus.REQUESTED_TO_JOIN_STAGE)
        socketService.send(SocketServiceUtils.RoomEvent.REQUEST_STAGE_ACCESS.id, null)
      } else if (stageStatus == StageStatus.REQUESTED_TO_JOIN_STAGE) {
        if (controllerContainer.selfController.roomJoined.not()) {
          controllerContainer.socketController.connect()
        }
        setStageStatus(StageStatus.ON_STAGE)
        socketService.send(SocketServiceUtils.RoomEvent.JOIN_STAGE.id, null)
        controllerContainer.roomNodeController.joinRoom()
      }
    } else {
      if (
        controllerContainer.presetController.permissions.media.video.permission ==
          DyteMediaPermission.ALLOWED ||
          controllerContainer.presetController.permissions.media.audioPermission ==
            DyteMediaPermission.ALLOWED ||
          stageStatus == StageStatus.ACCEPTED_TO_JOIN_STAGE
      ) {
        DyteLogger.info(
          "Stage :: Join Stage audio ? ${controllerContainer.selfController.getSelf().audioEnabled} video ? ${controllerContainer.selfController.getSelf().videoEnabled}"
        )
        controllerContainer.socketController.sendMessage(
          OutboundMeetingEventType.START_PRESENTING,
          null
        )
        setStageStatus(StageStatus.ON_STAGE)
        if (controllerContainer.selfController.getSelf().audioEnabled) {
          if (!controllerContainer.metaController.getIsHive())
            controllerContainer.socketController.sendMessage(
              OutboundMeetingEventType.UN_MUTE_SELF_AUDIO,
              null
            )
        }
      } else {
        DyteLogger.error("DyteWebinar::unauthorized_joinStage")
        throw UnsupportedOperationException("not allowed to join stage")
      }
    }
  }

  override suspend fun leave() {
    if (meetingType == DyteMeetingType.LIVESTREAM) {
      setStageStatus(StageStatus.OFF_STAGE)
      val request =
        LeaveStageRequest(listOf(controllerContainer.selfController.getSelf().userId)).encode()
      socketService.send(SocketServiceUtils.RoomEvent.LEAVE_STAGE.id, request)
      controllerContainer.roomNodeController.leaveRoom()
      controllerContainer.platformUtilsProvider.getVideoUtils().destroyAll()
      controllerContainer.sfuUtils.leaveCall()
      controllerContainer.socketController.disconnect()
    } else {
      controllerContainer.socketController.sendMessage(
        OutboundMeetingEventType.STOP_PRESENTING,
        null
      )
      setStageStatus(StageStatus.OFF_STAGE)
    }

    controllerContainer.selfController.disableAudio()
    controllerContainer.selfController.disableVideo()
  }

  override suspend fun kick(id: String) {
    if (meetingType == DyteMeetingType.LIVESTREAM) {
      // TODO : implement this
    } else {
      val payload = buildJsonObject {
        put("id", JsonPrimitive(id))
        put("type", JsonPrimitive("REQUESTED_BY_MODERATOR"))
      }
      controllerContainer.socketController.sendMessage(
        OutboundMeetingEventType.REMOVE_PEER_FROM_STAGE,
        payload
      )
    }
  }

  override suspend fun cancelRequestAccess() {
    if (meetingType == DyteMeetingType.LIVESTREAM) {
      socketService.send(SocketServiceUtils.RoomEvent.CANCEL_STAGE_REQUEST.id, null)
    } else {
      controllerContainer.socketController.sendMessage(
        OutboundMeetingEventType.WITHDRAW_REQUEST_TO_JOIN_STAGE,
        null
      )
    }
    setStageStatus(StageStatus.OFF_STAGE)
  }

  override fun setStageStatus(stageStatus: StageStatus) {
    if (_stageStatus == stageStatus) {
      return
    }
    _stageStatus = stageStatus
    controllerContainer.selfController.getSelf()._stageStatus = stageStatus
    controllerContainer.eventController.triggerEvent(
      DyteEventType.OnSelfStageStatusUpdate(_stageStatus)
    )
  }

  override fun onRequestedToPresent() {
    setStageStatus(StageStatus.ACCEPTED_TO_JOIN_STAGE)
    controllerContainer.eventController.triggerEvent(DyteEventType.OnWebinarPresentRequest)
  }

  override fun onPeerAddedToStage(peerId: String) {
    val addedPeer = _requestsToPresent.filter { it.id == peerId }
    if (addedPeer.isNotEmpty()) {
      addedPeer.first()._stageStatus = StageStatus.ACCEPTED_TO_JOIN_STAGE
      _requestsToPresent.removeAll(addedPeer.toSet())
      controllerContainer.eventController.triggerEvent(
        DyteEventType.OnWebinarPresentRequestAccepted(addedPeer.first())
      )
      controllerContainer.eventController.triggerEvent(
        DyteEventType.OnStageRequestsUpdated(requestedParticipants)
      )
    } else {
      val joinedParticipants =
        controllerContainer.participantController.meetingRoomParticipants.joined
      val joinedPeer = joinedParticipants.find { it.id == peerId }
      joinedPeer?.let {
        joinedPeer._stageStatus = StageStatus.ACCEPTED_TO_JOIN_STAGE
        controllerContainer.eventController.triggerEvent(
          DyteEventType.OnParticipantsUpdate(
            controllerContainer.participantController.meetingRoomParticipants
          )
        )
      }
    }
  }

  override fun onPeerStartedPresenting(peerId: String) {
    val addedPeer = _requestsToPresent.filter { it.id == peerId }
    if (addedPeer.isNotEmpty()) {
      addedPeer.first()._stageStatus = StageStatus.ON_STAGE
      _requestsToPresent.removeAll(addedPeer.toSet())
      controllerContainer.eventController.triggerEvent(
        DyteEventType.OnWebinarPresentRequestAccepted(addedPeer.first())
      )
      controllerContainer.eventController.triggerEvent(
        DyteEventType.OnStageRequestsUpdated(requestedParticipants)
      )
    } else {
      val joinedParticipants =
        controllerContainer.participantController.meetingRoomParticipants.joined
      val joinedPeer = joinedParticipants.find { it.id == peerId }
      joinedPeer?.let {
        joinedPeer._stageStatus = StageStatus.ON_STAGE
        controllerContainer.eventController.triggerEvent(
          DyteEventType.OnParticipantsUpdate(
            controllerContainer.participantController.meetingRoomParticipants
          )
        )
      }
    }
  }

  override fun onPeerStoppedPresenting(peerId: String) {
    val addedPeer = _requestsToPresent.filter { it.id == peerId }
    if (addedPeer.isNotEmpty()) {
      addedPeer.first()._stageStatus = StageStatus.OFF_STAGE
      _requestsToPresent.removeAll(addedPeer.toSet())
      controllerContainer.eventController.triggerEvent(
        DyteEventType.OnParticipantRemovedFromStage(addedPeer.first())
      )
      controllerContainer.eventController.triggerEvent(
        DyteEventType.OnStageRequestsUpdated(requestedParticipants)
      )
    } else {
      val joinedParticipants =
        controllerContainer.participantController.meetingRoomParticipants.joined
      val joinedPeer = joinedParticipants.find { it.id == peerId }
      joinedPeer?.let {
        joinedPeer._stageStatus = StageStatus.OFF_STAGE
        controllerContainer.eventController.triggerEvent(
          DyteEventType.OnParticipantsUpdate(
            controllerContainer.participantController.meetingRoomParticipants
          )
        )
        controllerContainer.eventController.triggerEvent(
          DyteEventType.OnParticipantRemovedFromStage(joinedPeer)
        )
      }
    }
    serialScope.launch { controllerContainer.participantController.setPage(0) }
  }

  override fun onPeerRejectedToStage(peerId: String) {
    val removedPeers = _requestsToPresent.filter { it.id == peerId }
    if (removedPeers.isNotEmpty()) {
      removedPeers.first()._stageStatus = StageStatus.REJECTED_TO_JOIN_STAGE
      _requestsToPresent.removeAll(removedPeers.toSet())
      controllerContainer.eventController.triggerEvent(
        DyteEventType.OnWebinarPresentRequestRejected(removedPeers.first())
      )
      controllerContainer.eventController.triggerEvent(
        DyteEventType.OnStageRequestsUpdated(requestedParticipants)
      )
    }
  }

  override fun onStoppedPresenting() {
    setStageStatus(StageStatus.OFF_STAGE)
    controllerContainer.eventController.triggerEvent(DyteEventType.OnWebinarStoppedPresenting)
  }

  override fun onStartedPresenting() {
    setStageStatus(StageStatus.ON_STAGE)
    controllerContainer.selfController.onEnteredInRoom()
    controllerContainer.eventController.triggerEvent(DyteEventType.OnWebinarStartedPresenting)
    serialScope.launch {
      controllerContainer.roomNodeController.connectToMediaProduction()
      controllerContainer.sfuUtils.produceMedia()
      controllerContainer.participantController.setPage(0)
    }
    val deviceType = controllerContainer.selfController.getSelectedAudioDevice()?.type
    if (
      deviceType == null ||
        deviceType == AudioDeviceType.SPEAKER ||
        deviceType == AudioDeviceType.UNKNOWN
    ) {
      controllerContainer.platformUtilsProvider.getMediaUtils().routeAudioFromSpeakerphone()
    }
  }

  override fun onRequestToPresentPeerAdded(id: String) {
    val stagePeer =
      controllerContainer.participantController.meetingRoomParticipants.joined.find { it.id == id }
    stagePeer?.let {
      stagePeer._stageStatus = StageStatus.REQUESTED_TO_JOIN_STAGE
      _requestsToPresent.add(stagePeer)
      controllerContainer.eventController.triggerEvent(
        DyteEventType.OnWebinarPresentRequestAdded(stagePeer)
      )
      controllerContainer.eventController.triggerEvent(
        DyteEventType.OnStageRequestsUpdated(requestedParticipants)
      )
    }
  }

  override fun onRequestToPresentPeerClosed(requestToPresentParticipant: WebSocketPeerLeftModel) {
    val addedPeer = _requestsToPresent.filter { it.id == requestToPresentParticipant.id }
    if (addedPeer.isNotEmpty()) {
      addedPeer.first()._stageStatus = StageStatus.ON_STAGE
      _requestsToPresent.removeAll(addedPeer.toSet())
      controllerContainer.eventController.triggerEvent(
        DyteEventType.OnWebinarPresentRequestClosed(addedPeer.first())
      )
      controllerContainer.eventController.triggerEvent(
        DyteEventType.OnStageRequestsUpdated(requestedParticipants)
      )
    }
  }

  override fun onRequestToPresentPeerWithdrawn(
    requestToPresentParticipant: WebSocketWebinarStagePeer
  ) {
    val withdrawnPeer = _requestsToPresent.filter { it.id == requestToPresentParticipant.id }
    if (withdrawnPeer.isNotEmpty()) {
      withdrawnPeer.first()._stageStatus = StageStatus.OFF_STAGE
      _requestsToPresent.removeAll(withdrawnPeer.toSet())
      controllerContainer.eventController.triggerEvent(
        DyteEventType.OnWebinarPresentRequestWithdrawn(withdrawnPeer.first())
      )
      controllerContainer.eventController.triggerEvent(
        DyteEventType.OnStageRequestsUpdated(requestedParticipants)
      )
    }
  }

  override fun handleRoomJoined(roomState: WebSocketJoinRoomModel) {
    _requestsToPresent.clear()
    val waitingPeers = arrayListOf<DyteJoinedMeetingParticipant>()
    roomState.requestToJoinPeersList?.forEach { waitingPeer ->
      val joinedPeer =
        controllerContainer.participantController.meetingRoomParticipants.joined.find {
          it.id == waitingPeer.id
        }
      joinedPeer?.let {
        it._stageStatus = StageStatus.REQUESTED_TO_JOIN_STAGE
        waitingPeers.add(joinedPeer)
      }
    }
    _requestsToPresent.addAll(waitingPeers)
    controllerContainer.eventController.triggerEvent(
      DyteEventType.OnStageRequestsUpdated(requestedParticipants)
    )
  }

  override fun handleRemovedFromStage() {
    setStageStatus(StageStatus.OFF_STAGE)
    controllerContainer.participantController.onRemovedFromStage()
    controllerContainer.eventController.triggerEvent(DyteEventType.OnWebinarRemovedFromStage)
  }

  override fun onStageRequestRejected() {
    setStageStatus(StageStatus.REJECTED_TO_JOIN_STAGE)
  }

  override fun onPeerRemovedFromStage(peerId: String) {
    val removedPeer = _requestsToPresent.filter { it.id == peerId }
    if (removedPeer.isNotEmpty()) {
      removedPeer.first()._stageStatus = StageStatus.OFF_STAGE
      _requestsToPresent.removeAll(removedPeer.toSet())
      controllerContainer.eventController.triggerEvent(
        DyteEventType.OnParticipantRemovedFromStage(removedPeer.first())
      )
      controllerContainer.eventController.triggerEvent(
        DyteEventType.OnStageRequestsUpdated(requestedParticipants)
      )
    } else {
      val joinedParticipants =
        controllerContainer.participantController.meetingRoomParticipants.joined
      val joinedPeer = joinedParticipants.find { it.id == peerId }
      joinedPeer?.let {
        joinedPeer._stageStatus = StageStatus.OFF_STAGE
        controllerContainer.eventController.triggerEvent(
          DyteEventType.OnParticipantRemovedFromStage(joinedPeer)
        )
        controllerContainer.eventController.triggerEvent(
          DyteEventType.OnParticipantsUpdate(
            controllerContainer.participantController.meetingRoomParticipants
          )
        )
      }
    }
  }
}

internal interface IStageSocketHandler {
  val stageStatus: StageStatus
  val requestedParticipants: List<DyteJoinedMeetingParticipant>

  suspend fun requestAccess()

  suspend fun cancelRequestAccess()

  suspend fun join()

  suspend fun leave()

  suspend fun kick(id: String)

  suspend fun grantAccess(ids: List<String>)

  suspend fun denyAccess(ids: List<String>)

  fun onStageRequestRejected()

  fun setStageStatus(stageStatus: StageStatus)

  fun onRequestedToPresent()

  fun onRequestToPresentPeerAdded(id: String)

  fun onRequestToPresentPeerClosed(requestToPresentParticipant: WebSocketPeerLeftModel)

  fun onRequestToPresentPeerWithdrawn(requestToPresentParticipant: WebSocketWebinarStagePeer)

  fun onPeerAddedToStage(peerId: String)

  fun onPeerStartedPresenting(peerId: String)

  fun onPeerStoppedPresenting(peerId: String)

  fun onPeerRemovedFromStage(peerId: String)

  fun onPeerRejectedToStage(peerId: String)

  fun onStoppedPresenting()

  fun onStartedPresenting()

  fun handleRoomJoined(roomState: WebSocketJoinRoomModel)

  fun handleRemovedFromStage()
}
