package io.dyte.core.models

import io.dyte.core.controllers.IControllerContainer
import io.dyte.core.controllers.PermissionType.CAMERA
import io.dyte.core.controllers.PermissionType.MICROPHONE
import io.dyte.core.controllers.StageStatus
import io.dyte.core.feat.ParticipantFlags
import io.dyte.core.media.DyteVideoMiddleware
import io.dyte.core.media.DyteVideoMiddlewares
import io.dyte.core.models.WaitListStatus.NONE
import io.dyte.core.network.info.DyteDesignToken
import io.dyte.core.platform.VideoView
import kotlinx.coroutines.*

// TODO : add enabledPreview, disablePreview
// TODO : getAudioDevices, getVideoDevices, getSpeakerDevices, getDeviceBydId, getAllDevices,
// setDevice
// TODO : setIsPinned, pin, unpin,
// TODO : add roomJoined, remove userId
// TODO : wrap audioTrack and videoTrack inside preview
// TODO : we can get rid of audioTrack, its not been used anywhere.
class DyteSelfParticipant
internal constructor(
  override val id: String,
  override val userId: String,
  override var name: String,
  override val picture: String?,
  override val isHost: Boolean,
  override val clientSpecificId: String?,
  override val flags: ParticipantFlags,
  override val presetName: String,
  controllerContainer: IControllerContainer
) :
  DyteJoinedMeetingParticipant(
    id,
    userId,
    name,
    picture,
    isHost,
    clientSpecificId,
    flags,
    presetName,
    controllerContainer
  ) {
  val permissions = controllerContainer.presetController.permissions

  internal var _waitListStatus: WaitListStatus = NONE
  val waitListStatus: WaitListStatus
    get() = _waitListStatus

  internal var _cameraPermissionGranted: Boolean =
    controllerContainer.permissionController.isPermissionGrated(CAMERA)
  val isCameraPermissionGranted: Boolean
    get() = _cameraPermissionGranted

  internal var _microphonePermissionGranted: Boolean =
    controllerContainer.permissionController.isPermissionGrated(MICROPHONE)
  val isMicrophonePermissionGranted: Boolean
    get() = _microphonePermissionGranted

  val roomJoined: Boolean
    get() = controllerContainer.selfController.roomJoined

  val designToken: DyteDesignToken
    get() = controllerContainer.presetController.designToken

  /** This method is used to unmute the local participant's audio. */
  override fun fetchVideoEnabled(): Boolean {
    if (!controllerContainer.metaController.isVideoEnabled()) {
      return false
    }
    // Means user is already have videoEnable == true or enableVideo is called by user once.
    return super.fetchVideoEnabled()
  }

  override fun fetchAudioEnabled(): Boolean {
    if (!controllerContainer.metaController.isAudioEnabled()) {
      return false
    }
    // Means user is already have videoEnable == true or enableVideo is called by user once.
    return super.fetchAudioEnabled()
  }

  fun enableAudio() {
    serialScope.launch { controllerContainer.selfController.enableAudio() }
  }

  /** This method is used to mute the local participant's audio. */
  override fun disableAudio() {
    serialScope.launch { controllerContainer.selfController.disableAudio() }
  }

  /** This method is used to start streaming the local participant's video to the meeting. */
  fun enableVideo() {
    serialScope.launch { controllerContainer.selfController.enableVideo() }
  }

  /** This participant is used to disable the local participant's video. */
  override fun disableVideo() {
    serialScope.launch { controllerContainer.selfController.disableVideo() }
  }

  fun getSelfPreview(): VideoView {
    return controllerContainer.platformUtilsProvider.getVideoUtils().getSelfPreview()
  }

  fun setDisplayName(name: String) {
    serialScope.launch { this@DyteSelfParticipant.name = name }
  }

  fun getAudioDevices(): List<DyteAudioDevice> {
    return runBlocking { controllerContainer.selfController.getAudioDevices() }
  }

  fun getVideoDevices(): List<DyteVideoDevice> {
    return runBlocking { controllerContainer.selfController.getVideoDevices() }
  }

  fun setAudioDevice(dyteAndroidDevice: DyteAudioDevice) {
    controllerContainer.selfController.setDevice(dyteAndroidDevice)
  }

  fun setVideoDevice(dyteVideoDevice: DyteVideoDevice) {
    serialScope.launch { controllerContainer.selfController.setDevice(dyteVideoDevice) }
  }

  fun switchCamera() {
    val devices = getVideoDevices()
    val device = devices.firstOrNull { it.type != getSelectedVideoDevice()?.type }
    device?.let { setVideoDevice(device) }
  }

  fun getSelectedVideoDevice(): DyteVideoDevice? {
    return controllerContainer.selfController.getSelectedVideoDevice()
  }

  fun getSelectedAudioDevice(): DyteAudioDevice? {
    return controllerContainer.selfController.getSelectedAudioDevice()
  }

  fun canDoParticipantHostControls(): Boolean {
    return runBlocking {
      withContext(serialScope.coroutineContext) {
        controllerContainer.presetController.canDoParticipantHostControls()
      }
    }
  }

  fun shouldShowSetupScreen(): Boolean {
    return runBlocking {
      withContext(serialScope.coroutineContext) {
        permissions.media.canPublishAudio || permissions.media.canPublishVideo
      }
    }
  }

  fun shouldJoinMediaRoom(): Boolean {
    return runBlocking {
      withContext(serialScope.coroutineContext) {
        permissions.media.canPublishAudio ||
          permissions.media.canPublishVideo ||
          controllerContainer.stageController.stageStatus == StageStatus.ON_STAGE
      }
    }
  }

  fun addVideoMiddleware(middleware: DyteVideoMiddleware) =
    DyteVideoMiddlewares.addVideoMiddleware(middleware)

  fun removeVideoMiddleware(middleware: DyteVideoMiddleware) =
    DyteVideoMiddlewares.removeVideoMiddleware(middleware)

  fun getVideoMiddlewares(): List<DyteVideoMiddleware> = DyteVideoMiddlewares.getVideoMiddlewares()

  override fun toMap(): Map<String, Any?> {
    val map = HashMap(super.toMap())
    map["selfPermissions"] = permissions.toMap()
    map["systemPermissions"] =
      hashMapOf(
        "isCameraPermissionGranted" to isCameraPermissionGranted,
        "isMicrophonePermissionGranted" to isMicrophonePermissionGranted
      )

    return map
  }
}
