package io.dyte.core.featureflag

import io.dyte.core.featureflag.FeatureFlagService.UserAttributeKeys
import io.dyte.core.observability.DyteLogger
import io.ktor.client.HttpClient
import io.ktor.client.call.body
import io.ktor.client.plugins.timeout
import io.ktor.client.request.header
import io.ktor.client.request.post
import io.ktor.client.request.setBody
import io.ktor.http.ContentType
import io.ktor.http.contentType
import io.ktor.http.takeFrom
import kotlinx.serialization.json.JsonElement
import kotlinx.serialization.json.JsonObject
import kotlinx.serialization.json.addJsonObject
import kotlinx.serialization.json.boolean
import kotlinx.serialization.json.buildJsonObject
import kotlinx.serialization.json.contentOrNull
import kotlinx.serialization.json.jsonArray
import kotlinx.serialization.json.jsonObject
import kotlinx.serialization.json.jsonPrimitive
import kotlinx.serialization.json.put
import kotlinx.serialization.json.putJsonArray

internal class FlagsmithService(
  private val environmentKey: String,
  private val httpClient: HttpClient,
  private var timeout: Long = 5000L
) : FeatureFlagService {
  private val featureFlags: MutableMap<String, FeatureFlag> = HashMap()

  override suspend fun initializeWithUserContext(
    identifier: String,
    userAttributes: Map<String, Any>
  ) {
    // val meetingHash = generateMeetingHash(roomName)

    val payload = createIdentifyPayload(identifier, userAttributes)

    // val flagsmithIdentifier = finalizeFlagsmithIdentifier(forceEvaluate, identifier)

    try {
      val flagsmithResponse =
        httpClient.post {
          url.takeFrom(FLAGSMITH_URL_IDENTIFY)
          timeout { requestTimeoutMillis = this@FlagsmithService.timeout }
          contentType(ContentType.Application.Json)
          header(HEADER_ENVIRONMENT_KEY, environmentKey)
          setBody(payload)
        }
      if (flagsmithResponse.status.value == HTTP_STATUS_CODE_OK) {
        parseAndStoreFlags(flagStore = featureFlags, flagsmithJson = flagsmithResponse.body())
      }
      DyteLogger.info("flagsmith::allFlags::$featureFlags")
    } catch (e: Exception) {
      DyteLogger.error("flagsmith::init::error", e)
    }
  }

  private fun parseAndStoreFlags(
    flagStore: MutableMap<String, FeatureFlag>,
    flagsmithJson: JsonElement
  ) {
    var name: String
    var enabled: Boolean
    var value: String?
    try {
      val flagsArray =
        flagsmithJson.jsonObject[RESPONSE_KEY_FLAGS]?.jsonArray
          ?: error("flags array must be present")
      for (flag in flagsArray) {
        try {
          name =
            flag.jsonObject["feature"]
              ?.jsonObject
              ?.get(RESPONSE_KEY_FLAG_NAME)
              ?.jsonPrimitive
              ?.content
              ?: error("flag name must not be null")
          enabled =
            flag.jsonObject[RESPONSE_KEY_FLAG_ENABLED]?.jsonPrimitive?.boolean
              ?: error("flag enable status must be present")
          value = flag.jsonObject[RESPONSE_KEY_FLAG_VALUE]?.jsonPrimitive?.contentOrNull
          flagStore[name] = FeatureFlag(name, enabled, value)
        } catch (e: Exception) {
          DyteLogger.error("FlagsmithRepository: failed to parse the flagsmith flag", e)
        }
      }
    } catch (e: Exception) {
      DyteLogger.error("FlagsmithRepository: failed to parse the flagsmithJson", e)
    }
  }

  override fun isFeatureEnabled(featureName: String): Boolean {
    return featureFlags[featureName]?.enabled ?: false
  }

  override fun getConfigValue(configName: String): String {
    return featureFlags[configName]?.value ?: ""
  }

  private data class FeatureFlag(val name: String, val enabled: Boolean, val value: String? = null)

  /*
   * Note(swapnil): will be implemented and used after we sync or merge the mobile-core flags with web-core.
   * TODO: Implement same as it is in web-core
   * */
  /*private fun generateMeetingHash(roomName: String): Int {
    return -110
  }*/

  private fun createIdentifyPayload(
    identifier: String,
    userAttributes: Map<String, Any>
  ): JsonObject {
    return buildJsonObject {
      put(IDENTIFY_PAYLOAD_KEY_IDENTIFIER, identifier)
      putJsonArray(IDENTIFY_PAYLOAD_KEY_TRAITS) {
        // Entity
        addJsonObject {
          put(IDENTIFY_PAYLOAD_KEY_TRAIT_KEY, UserAttributeKeys.ENTITY)
          put(
            IDENTIFY_PAYLOAD_KEY_TRAIT_VALUE,
            userAttributes.getValue(UserAttributeKeys.ENTITY) as String
          )
        }

        // Client ID
        val clientId = userAttributes[UserAttributeKeys.CLIENT_ID] as String?
        addJsonObject {
          put(IDENTIFY_PAYLOAD_KEY_TRAIT_KEY, UserAttributeKeys.CLIENT_ID)
          put(IDENTIFY_PAYLOAD_KEY_TRAIT_VALUE, clientId)
        }

        // Room name
        addJsonObject {
          put(IDENTIFY_PAYLOAD_KEY_TRAIT_KEY, UserAttributeKeys.ROOM_NAME)
          put(
            IDENTIFY_PAYLOAD_KEY_TRAIT_VALUE,
            userAttributes.getValue(UserAttributeKeys.ROOM_NAME) as String
          )
        }

        // Anonymous user
        addJsonObject {
          put(IDENTIFY_PAYLOAD_KEY_TRAIT_KEY, UserAttributeKeys.ANONYMOUS_USER)
          put(
            IDENTIFY_PAYLOAD_KEY_TRAIT_VALUE,
            userAttributes.getOrElse(UserAttributeKeys.ANONYMOUS_USER) { false } as Boolean
          )
        }

        // SDK Version
        addJsonObject {
          put(IDENTIFY_PAYLOAD_KEY_TRAIT_KEY, UserAttributeKeys.SDK_VERSION)
          put(
            IDENTIFY_PAYLOAD_KEY_TRAIT_VALUE,
            userAttributes.getValue(UserAttributeKeys.SDK_VERSION) as String
          )
        }

        // Is Mobile?
        addJsonObject {
          put(IDENTIFY_PAYLOAD_KEY_TRAIT_KEY, UserAttributeKeys.IS_MOBILE)
          put(
            IDENTIFY_PAYLOAD_KEY_TRAIT_VALUE,
            userAttributes.getOrElse(UserAttributeKeys.IS_MOBILE) { true } as Boolean
          )
        }

        // OS name
        addJsonObject {
          put(IDENTIFY_PAYLOAD_KEY_TRAIT_KEY, UserAttributeKeys.OS_NAME)
          put(
            IDENTIFY_PAYLOAD_KEY_TRAIT_VALUE,
            userAttributes.getValue(UserAttributeKeys.OS_NAME) as String
          )
        }

        // OS Version name
        addJsonObject {
          put(IDENTIFY_PAYLOAD_KEY_TRAIT_KEY, UserAttributeKeys.OS_VERSION_NAME)
          put(
            IDENTIFY_PAYLOAD_KEY_TRAIT_VALUE,
            userAttributes.getValue(UserAttributeKeys.OS_VERSION_NAME) as String
          )
        }
      }
    }
  }

  /*
   * TODO(swapnil): Understand the force-evaluate usage from the web-team and implement this method.
   * */
  /*private fun finalizeFlagsmithIdentifier(forceEvaluate: Boolean, identifier: String): String {
    if (forceEvaluate) {
      return identifier + generateRandomSuffix()
    }

    return identifier
  }*/

  /*
   * TODO(swapnil): Understand the force-evaluate usage from the web-team and implement this method.
   * */
  /*private fun generateRandomSuffix(): String {
    return "_${generateRandomString(10)}"
  }*/

  companion object {
    private const val FLAGSMITH_URL_FLAGS = "https://edge.api.flagsmith.com/api/v1/flags/"
    private const val FLAGSMITH_URL_IDENTIFY = "https://edge.api.flagsmith.com/api/v1/identities/"
    private const val HEADER_ENVIRONMENT_KEY = "X-Environment-Key"
    private const val HTTP_STATUS_CODE_OK = 200
    private const val RESPONSE_KEY_FLAG_NAME = "name"
    private const val RESPONSE_KEY_FLAG_ENABLED = "enabled"
    private const val RESPONSE_KEY_FLAG_VALUE = "feature_state_value"
    private const val RESPONSE_KEY_FLAGS = "flags"

    private const val IDENTIFY_PAYLOAD_KEY_IDENTIFIER = "identifier"
    private const val IDENTIFY_PAYLOAD_KEY_TRAITS = "traits"
    private const val IDENTIFY_PAYLOAD_KEY_TRAIT_KEY = "trait_key"
    private const val IDENTIFY_PAYLOAD_KEY_TRAIT_VALUE = "trait_value"

    /*
     * TODO(swapnil): Understand the force-evaluate usage from the web-team and implement this method.
     *
     * Note(swapnil): This string generation method would be either moved outside as a common utility
     * method or replaced when we implement a proper Base36 encoding method.
     * */
    /*private val characterSet = charArrayOf(
      '0', '1', '2', '3', '4', '5',
      '6', '7', '8', '9', 'a', 'b',
      'c', 'd', 'e', 'f', 'g', 'h',
      'i', 'j', 'k', 'l', 'm', 'n',
      'o', 'p', 'q', 'r', 's', 't',
      'u', 'v', 'w', 'x', 'y', 'z'
    )*/

    /*private fun generateRandomString(length: Int = 5): String {
      val stringBuilder: StringBuilder = StringBuilder()
      var randomIndex = 0
      for (i in 0 until length) {
        randomIndex = (Random.nextInt(until = characterSet.size))
        stringBuilder.append(characterSet[randomIndex])
      }

      return stringBuilder.toString()
    }*/
  }
}
