package io.dyte.core.feat

import io.dyte.core.controllers.IControllerContainer
import io.dyte.core.controllers.StageStatus
import io.dyte.core.listeners.DyteParticipantUpdateListener
import io.dyte.webrtc.VideoStreamTrack
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.launch
import kotlinx.coroutines.newSingleThreadContext

/**
 * Dyte meeting participant
 *
 * @property id The ID of the participant, this ID is temporary changes every time the participant
 *   rejoins the meeting
 * @property userId The userId of the participant, this ID is tied to [clientSpecificId] and does
 *   not change for a user in a meeting
 * @property name The participant's name
 * @property picture The participant's picture (if any).
 * @property clientSpecificId Identifier provided while adding the participant.
 * @property audioEnabled A boolean value indicating if the audio currently enabled.
 * @property videoEnabled A boolean value indicating if the video currently enabled.
 */
abstract class DyteMeetingParticipant
internal constructor(
  open val id: String,
  open val userId: String,
  open val name: String,
  open val picture: String?,
  open val isHost: Boolean,
  open val clientSpecificId: String?,
  open val flags: ParticipantFlags,
  open val presetName: String,
  internal val controllerContainer: IControllerContainer
) {
  /** The video track for the local user. */
  internal var _videoTrack: VideoStreamTrack? = null
  val videoTrack: VideoStreamTrack?
    get() = _videoTrack

  internal var _stageStatus: StageStatus = StageStatus.OFF_STAGE
  val stageStatus: StageStatus
    get() = _stageStatus

  /** The audio track for the local user. */
  // TODO : add audio track here
  // var audioTrack: Any? = null

  internal var _audioEnabled: Boolean = false
  val audioEnabled: Boolean
    get() = fetchAudioEnabled()

  internal var _videoEnabled: Boolean = false
  val videoEnabled: Boolean
    get() = fetchVideoEnabled()

  protected val serialScope = CoroutineScope(newSingleThreadContext("DyteMeetingParticipant"))

  protected open fun fetchVideoEnabled(): Boolean {
    return _videoEnabled
  }

  protected open fun fetchAudioEnabled(): Boolean {
    return _audioEnabled
  }

  open fun toMap(): Map<String, Any?> {
    val map = HashMap<String, Any?>()
    map["id"] = id
    map["userId"] = userId
    map["name"] = name
    map["picture"] = picture
    map["isHost"] = isHost
    map["clientSpecificId"] = clientSpecificId
    map["flags"] = flags.toMap()
    map["audioEnabled"] = audioEnabled
    map["videoEnabled"] = videoEnabled
    return map
  }

  fun addParticipantUpdateListener(participantUpdateListener: DyteParticipantUpdateListener) {
    val self = this
    serialScope.launch {
      controllerContainer.eventController.addParticipantUpdateListener(
        self,
        participantUpdateListener
      )
    }
  }

  fun removeParticipantUpdateListener(participantUpdateListener: DyteParticipantUpdateListener) {
    serialScope.launch {
      controllerContainer.eventController.removeParticipantUpdateListener(
        this@DyteMeetingParticipant,
        participantUpdateListener
      )
    }
  }

  fun removeParticipantUpdateListeners() {
    controllerContainer.eventController.removeParticipantUpdateListeners(this)
  }
}

data class ParticipantFlags(
  val recorder: Boolean,
  val hiddenParticipant: Boolean,
  val webinarHiddenParticipant: Boolean
) {
  fun toMap(): Map<String, Any?> {
    val map = HashMap<String, Any?>()
    map["recorder"] = recorder
    map["hiddenParticipant"] = hiddenParticipant
    return map
  }
}
