package io.dyte.core.plugins

import io.dyte.core.controllers.DyteEventType
import io.dyte.core.controllers.IControllerContainer
import io.dyte.core.feat.DytePlugin
import io.ktor.http.URLBuilder
import io.ktor.http.set
import kotlinx.serialization.json.JsonObject

internal abstract class PluginsController(private val controllerContainer: IControllerContainer) {
  protected val _allPlugins = mutableMapOf<String, DytePlugin>()
  protected val _activePlugins = mutableMapOf<String, DytePlugin>()

  val allPlugins: List<DytePlugin>
    get() = _allPlugins.values.toList()

  val activePlugins: List<DytePlugin>
    get() = _activePlugins.values.toList()

  suspend fun activatePlugin(pluginId: String, staggered: Boolean) {
    if (controllerContainer.presetController.permissions.plugins.canLaunch.not()) {
      throw UnsupportedOperationException("can not start plugin")
    } else {
      launchPlugin(pluginId, staggered)
    }
  }

  suspend fun deactivatePlugin(pluginId: String) {
    if (controllerContainer.presetController.permissions.plugins.canClose.not()) {
      throw UnsupportedOperationException("can not close plugin")
    } else {
      closePlugin(pluginId)
    }
  }

  abstract suspend fun init()

  protected abstract suspend fun launchPlugin(pluginId: String, staggered: Boolean)

  protected abstract suspend fun closePlugin(pluginId: String)

  abstract suspend fun handlePluginMessage(pluginId: String, message: JsonObject)

  fun handlePluginFileRequest(plugin: DytePlugin) {
    controllerContainer.eventController.triggerEvent(DyteEventType.OnPluginFileRequest(plugin))
  }

  protected fun submitEventToPlugin(pluginId: String, payload: JsonObject) {
    _activePlugins[pluginId]?.submitEvent(payload)
  }

  protected suspend fun enablePluginForLocalUser(pluginId: String, enabledBy: String) {
    // for safe-guard
    if (_activePlugins.containsKey(pluginId)) return

    val plugin = _allPlugins[pluginId] ?: return

    // just for safe-guard
    if (plugin.isActive) return

    try {
      val pluginAuthToken = controllerContainer.apiClient.authorizePlugin(plugin.id).data.token
      val pluginUrl = URLBuilder(plugin.baseURL)
      pluginUrl.set {
        parameters["auth"] = pluginAuthToken
        parameters["pluginId"] = plugin.id
        parameters["backend"] = controllerContainer.metaController.getBaseUrl()
        encodedParameters["parent"] = "*"
      }

      plugin.enableLocal(pluginAuthToken, pluginUrl.buildString(), enabledBy)
      _activePlugins[plugin.id] = plugin
    } catch (e: Exception) {
      // TODO: log the enable plugin failure
    }
  }
}

/*
 * note(swapnil): Used by the RoomNodeController to notify Plugins feature after joinRoom in complete without
 * exposing the other Plugin APIs.
 * Ideally there should be a separate class to manage these Room event or internal events to which all
 * interested classes can subscribe and get notified. That class can be used by the RoomNodeController
 * to notify the interested components in a decoupled way.
 * */
internal interface PluginsRoomJoinListener {
  suspend fun onRoomJoined()
}
