package io.dyte.core.models

import io.dyte.core.controllers.IControllerContainer
import io.dyte.core.feat.DyteMeetingParticipant
import io.dyte.core.feat.DyteWaitlistedParticipant
import kotlinx.coroutines.*
import kotlinx.serialization.json.JsonObject

/**
 * Dyte meeting participants
 *
 * @property waitlisted List of participant objects that are waiting to be let in
 * @property joined List of participant objects that are in the meeting right now
 * @property active List of participant objects that should be rendered on the screen right now
 * @property pinned List of participant objects that are marked pinned
 */
data class DyteRoomParticipants
internal constructor(
  val waitlisted: List<DyteWaitlistedParticipant>,
  val joined: List<DyteJoinedMeetingParticipant>,
  private val _active: List<DyteJoinedMeetingParticipant>,
  val screenshares: List<DyteScreenShareMeetingParticipant>,
  private val controllerContainer: IControllerContainer,
) {

  val serialScope = CoroutineScope(newSingleThreadContext("DyteRoomParticipants"))

  val pinned: DyteJoinedMeetingParticipant?
    get() = runBlocking {
      withContext(serialScope.coroutineContext) {
        controllerContainer.participantController.pinnedParticipant
      }
    }

  val active: List<DyteJoinedMeetingParticipant>
    get() = _active.distinctBy { it.id }

  val activeSpeaker: DyteJoinedMeetingParticipant?
    get() = runBlocking {
      withContext(serialScope.coroutineContext) {
        controllerContainer.participantController.activeSpeaker
      }
    }

  val currentPageNumber: Int
    get() = controllerContainer.participantController.getGridPagesInfo().currentPageNumber

  /** Boolean to indicate if there a next page of active participants available */
  val canGoNextPage: Boolean
    get() = controllerContainer.participantController.getGridPagesInfo().isNextPagePossible

  /** Boolean to indicate if there a previous page of active participants available */
  val canGoPreviousPage: Boolean
    get() = controllerContainer.participantController.getGridPagesInfo().isPreviousPagePossible

  /** Number of total active participant pages */
  val pageCount: Int
    get() = controllerContainer.participantController.getGridPagesInfo().pageCount

  fun toMap(): Map<String, Any?> {
    val map = HashMap<String, Any?>()
    map["waitlisted"] = listDyteMeetingParticipantToMap(waitlisted)
    map["joined"] = listDyteMeetingParticipantToMap(joined)
    map["active"] = listDyteMeetingParticipantToMap(active)
    map["screenshares"] = listDyteMeetingParticipantToMap(screenshares)
    map["pinned"] = pinned?.toMap()
    map["activeSpeaker"] = activeSpeaker?.toMap()
    map["grid"] = controllerContainer.participantController.getGridPagesInfo().toMap()
    return map
  }

  private fun listDyteMeetingParticipantToMap(
    participants: List<DyteMeetingParticipant>
  ): List<Map<String, Any?>> {
    return participants.map { it.toMap() }
  }

  /** Navigates to the given [pageNumber] page */
  @Throws(Exception::class)
  fun setPage(pageNumber: Int) {
    if (pageNumber < 0) {
      throw IllegalArgumentException("page number cant be negative")
    }
    serialScope.launch { controllerContainer.participantController.setPage(pageNumber) }
  }

  /** Mutes all [joined] participant's audio in the meeting */
  fun disableAllAudio() {
    // TODO : fix this hardcoding
    serialScope.launch { controllerContainer.hostController.muteAllAudio(true) }
  }

  /** Disables all [joined] participant's camera / video in the meeting */
  @Throws(Exception::class)
  fun disableAllVideo() {
    serialScope.launch { controllerContainer.hostController.muteAllVideo() }
  }

  /**
   * Accepts all pending requests from [waitlisted] participants waiting to be let in the meeting
   */
  @Throws(Exception::class)
  fun acceptAllWaitingRequests() {
    serialScope.launch { controllerContainer.waitlistController.acceptAllWaitingRequests() }
  }

  /** Removes all [joined] participants */
  fun kickAll() {
    serialScope.launch { controllerContainer.hostController.kickAll() }
  }

  /** Send a broadcast message to all [joined] participants */
  fun broadcastMessage(payload: JsonObject) {
    serialScope.launch { controllerContainer.participantController.broadcastMessage(payload) }
  }
}
