package io.dyte.core.media

import io.dyte.core.observability.DyteLogger
import io.dyte.webrtc.*
import kotlinx.coroutines.runBlocking

open class DyteCommonMediaUtils : IDyteCommonMediaUtils {
  private var platform: String = "android"

  init {
    try {
      // WebRtc.initialize()
    } catch (e: Exception) {
      e.printStackTrace()
    }
  }

  private var audioTrack: AudioStreamTrack? = null
  private var videoTrack: VideoStreamTrack? = null

  private var videoMediaStream: MediaStream? = null
  private var audioMediaStream: MediaStream? = null

  private var cameraFacingUser = true

  override fun getVideoStream(): MediaStream? {
    if (
      videoMediaStream != null &&
        videoMediaStream?.videoTracks?.first()?.readyState is MediaStreamTrackState.Live
    ) {
      DyteLogger.info("DyteCommonMediaUtils::getVideoStream::getUserMedia::stream_exists")
      return videoMediaStream!!
    }

    DyteLogger.info("DyteCommonMediaUtils::getVideoStream::getUserMedia::success")

    val camExists = runBlocking { MediaDevices.enumerateDevices().isNotEmpty() }

    if (camExists) {
      runBlocking {
        try {
          if (platform == "iOS") {
            videoMediaStream =
              MediaDevices.getUserMedia {
                video {
                  facingMode(if (cameraFacingUser) FacingMode.User else FacingMode.Environment)
                }
              }
          } else {
            videoMediaStream =
              MediaDevices.getUserMedia {
                video {
                  facingMode(if (cameraFacingUser) FacingMode.User else FacingMode.Environment)
                  height(640)
                  width(480)
                  frameRate(30.0)
                }
              }
          }
          DyteLogger.info("DyteCommonMediaUtils::getVideoStream::getUserMedia::success")
        } catch (e: Exception) {
          DyteLogger.error("DyteCommonMediaUtils::getVideoStream::getUserMedia::error", e)
        }
      }
    }

    return videoMediaStream
  }

  override fun getAudioStream(): MediaStream? {
    if (
      audioMediaStream != null &&
        audioMediaStream?.audioTracks?.first()?.readyState is MediaStreamTrackState.Live
    ) {
      DyteLogger.info("DyteCommonMediaUtils::getAudioStream::stream_exists")
      return audioMediaStream!!
    }

    DyteLogger.info("DyteCommonMediaUtils::getAudioStream::getUserMedia")
    runBlocking {
      try {
        audioMediaStream = MediaDevices.getUserMedia { audio(true) }
        DyteLogger.info("DyteCommonMediaUtils::getAudioStream::getUserMedia::success")
      } catch (e: Exception) {
        DyteLogger.error("DyteCommonMediaUtils::getAudioStream::getUserMedia::error", e)
      }
    }

    return audioMediaStream
  }

  override fun createAudioTrack(): AudioStreamTrack? {
    audioTrack = getAudioStream()?.audioTracks?.first()
    return audioTrack
  }

  override fun createVideoTrack(): VideoStreamTrack? {
    videoTrack = getVideoStream()?.videoTracks?.first()
    return videoTrack
  }

  override fun stopVideo() {
    videoTrack?.enabled = false
    DyteLogger.info("DyteCommonMediaUtils | video stopped")
  }

  override fun resumeVideo() {
    videoTrack?.enabled = true
    DyteLogger.info("DyteCommonMediaUtils | video resumed")
  }

  override fun switchCamera() {
    runBlocking {
      if (videoTrack?.enabled == true) {
        DyteLogger.info("DyteCommonMediaUtils | switchCamera | switching camera")
        videoTrack?.switchCamera()
        cameraFacingUser = !cameraFacingUser
      } else {
        DyteLogger.info(
          "DyteCommonMediaUtils | switchCamera | video track is not enabled, not switching camera"
        )
      }
    }
  }

  override fun dispose() {
    // WebRtc.dispose()
    try {
      videoMediaStream?.release()
      audioMediaStream?.release()
    } catch (e: Exception) {
      DyteLogger.error("DyteCommonMediaUtils | dispose mediaStream release failed", e)
    }

    videoMediaStream = null
    audioMediaStream = null
    DyteLogger.info("DyteCommonMediaUtils | audio/video streams disposed")
  }

  override fun onCameraClosed() {
    videoTrack = null
  }

  override fun getVideoTrack(): VideoStreamTrack? {
    return if (videoTrack == null || videoTrack?.readyState is MediaStreamTrackState.Ended)
      createVideoTrack()
    else videoTrack
  }

  override fun setPlatform(platform: String) {
    this.platform = platform
  }
}

interface IDyteCommonMediaUtils {

  fun getVideoStream(): MediaStream?

  fun getAudioStream(): MediaStream?
  /**
   * Switch camera
   *
   * toggles between front camera and back camera
   */
  fun switchCamera()

  /**
   * Create audio track
   *
   * @return instance of AudioTrack from webrtc
   */
  fun createAudioTrack(): AudioStreamTrack?

  /**
   * Create video track
   *
   * @return instance of videoTrack from webrtc
   */
  fun createVideoTrack(): VideoStreamTrack?

  fun stopVideo()

  fun resumeVideo()

  fun dispose()

  fun onCameraClosed()

  // EglContext for Android can be retrieved by doing: WebRtc.rootEglBase.eglBaseContext

  fun getVideoTrack(): VideoStreamTrack?

  fun setPlatform(platform: String)
}
