package io.dyte.core.hive

import io.dyte.core.observability.DyteLogger
import io.dyte.core.socket.socketservice.SocketService
import io.dyte.core.socket.socketservice.SocketServiceEventListener
import io.dyte.core.socket.socketservice.SocketServiceUtils
import media.edge.PeerJoinCompleteRequest
import media.edge.PeerJoinCompleteResponse
import media.edge.PeerJoinRequest
import socket.room.BroadcastMessage

internal interface HiveNodeSocketHandler {
  fun subscribeToBroadcasts(
    onPeerJoinedBroadcast: SocketServiceEventListener,
    onPeerProducerCreateBroadcast: SocketServiceEventListener,
    onPeerProducerToggleBroadcast: SocketServiceEventListener,
    onSelectedPeer: SocketServiceEventListener,
    onSelectedPeerDiff: SocketServiceEventListener,
    onPeerLeaveBroadcast: SocketServiceEventListener,
    onPeerProducerCloseBroadcast: SocketServiceEventListener,
    onGlobalPeerPinBroadcast: SocketServiceEventListener,
    onRecordingStartedBroadcast: SocketServiceEventListener,
    onRecordingStoppedBroadcast: SocketServiceEventListener,
    onBroadcastMessage: SocketServiceEventListener
  )

  suspend fun joinRoom(peerJoinRequest: PeerJoinRequest)

  suspend fun completeJoinRoom(): PeerJoinCompleteResponse?

  suspend fun broadcastMessage(broadcastMessage: BroadcastMessage)
}

internal class DefaultHiveNodeSocketHandler(private val socketService: SocketService) :
  HiveNodeSocketHandler {
  override fun subscribeToBroadcasts(
    onPeerJoinedBroadcast: SocketServiceEventListener,
    onPeerProducerCreateBroadcast: SocketServiceEventListener,
    onPeerProducerToggleBroadcast: SocketServiceEventListener,
    onSelectedPeer: SocketServiceEventListener,
    onSelectedPeerDiff: SocketServiceEventListener,
    onPeerLeaveBroadcast: SocketServiceEventListener,
    onPeerProducerCloseBroadcast: SocketServiceEventListener,
    onGlobalPeerPinBroadcast: SocketServiceEventListener,
    onRecordingStartedBroadcast: SocketServiceEventListener,
    onRecordingStoppedBroadcast: SocketServiceEventListener,
    onBroadcastMessage: SocketServiceEventListener
  ) {
    socketService.subscribe(
      SocketServiceUtils.MediaEvent.PEER_JOINED_BROADCAST.id,
      onPeerJoinedBroadcast
    )

    socketService.subscribe(
      SocketServiceUtils.MediaEvent.PEER_PRODUCER_CREATE_BROADCAST.id,
      onPeerProducerCreateBroadcast
    )

    socketService.subscribe(
      SocketServiceUtils.MediaEvent.PEER_PRODUCER_TOGGLE_BROADCAST.id,
      onPeerProducerToggleBroadcast
    )

    socketService.subscribe(SocketServiceUtils.MediaEvent.SELECTED_PEER.id, onSelectedPeer)

    socketService.subscribe(SocketServiceUtils.MediaEvent.SELECTED_PEER_DIFF.id, onSelectedPeerDiff)

    socketService.subscribe(
      SocketServiceUtils.MediaEvent.PEER_LEAVE_BROADCAST.id,
      onPeerLeaveBroadcast
    )

    socketService.subscribe(
      SocketServiceUtils.MediaEvent.PEER_PRODUCER_CLOSE_BROADCAST.id,
      onPeerProducerCloseBroadcast
    )

    socketService.subscribe(
      SocketServiceUtils.MediaEvent.GLOBAL_PEER_PIN_BROADCAST.id,
      onGlobalPeerPinBroadcast
    )

    socketService.subscribe(
      SocketServiceUtils.MediaEvent.RECORDING_STARTED_BROADCAST.id,
      onRecordingStartedBroadcast
    )

    socketService.subscribe(
      SocketServiceUtils.MediaEvent.RECORDING_STOPPED_BROADCAST.id,
      onRecordingStoppedBroadcast
    )

    socketService.subscribe(SocketServiceUtils.RoomEvent.BROADCAST_MESSAGE.id, onBroadcastMessage)
  }

  override suspend fun joinRoom(peerJoinRequest: PeerJoinRequest) {
    try {
      socketService.send(
        event = SocketServiceUtils.MediaEvent.JOIN_ROOM.id,
        payload = PeerJoinRequest.ADAPTER.encode(peerJoinRequest)
      )
    } catch (e: Exception) {
      DyteLogger.error("HiveSocketHandler: Error on join room: ${e.message}")
    }
  }

  override suspend fun completeJoinRoom(): PeerJoinCompleteResponse? {
    return try {
      val req = PeerJoinCompleteRequest()

      PeerJoinCompleteResponse.ADAPTER.decode(
        socketService.requestResponse(
          event = SocketServiceUtils.MediaEvent.SELF_JOIN_COMPLETE.id,
          payload = PeerJoinCompleteRequest.ADAPTER.encode(req)
        )!!
      )
    } catch (e: Exception) {
      DyteLogger.error("HiveSocketHandler: Error on complete join room: ${e.message}")
      null
    }
  }

  override suspend fun broadcastMessage(broadcastMessage: BroadcastMessage) {
    try {
      socketService.send(
        event = SocketServiceUtils.RoomEvent.BROADCAST_MESSAGE.id,
        payload = BroadcastMessage.ADAPTER.encode(broadcastMessage)
      )
    } catch (e: Exception) {
      DyteLogger.error("HiveSocketHandler: Error on broadcast message: ${e.message}")
    }
  }
}
