package io.dyte.core.feat

import io.dyte.core.controllers.IControllerContainer
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.launch
import kotlinx.coroutines.newSingleThreadContext

/**
 * Dyte poll
 *
 * @constructor Create empty Dyte poll
 * @property controllerContainer
 */
class DytePoll internal constructor(private val controller: IControllerContainer) {

  private val serialScope = CoroutineScope(newSingleThreadContext("DytePoll"))
  val polls: List<DytePollMessage>
    get() = controller.pollsController.polls

  fun toMap(): List<Map<String, Any?>> {
    return polls.map { it.toMap() }
  }

  companion object {
    @Suppress("UNCHECKED_CAST")
    fun fromMap(pollMessage: Map<String, Any>): DytePollMessage {
      val options = pollMessage["options"] as List<Map<String, Any>>
      val pollOptions: List<DytePollOption> = options.map { DytePollOption.fromMap(it) }
      val id = pollMessage["id"] as String
      val question = pollMessage["question"] as String
      val anonymous = pollMessage["anonymous"] as Boolean
      val hideVotes = pollMessage["hideVotes"] as Boolean
      val createdBy = pollMessage["createdBy"] as String
      return DytePollMessage(id, question, anonymous, hideVotes, createdBy, pollOptions)
    }
  }

  /**
   * Vote on poll
   *
   * @param pollMessage
   * @param pollOption
   */
  fun vote(pollMessage: DytePollMessage, pollOption: DytePollOption) {
    serialScope.launch { controller.pollsController.vote(pollMessage, pollOption) }
  }

  /**
   * Create poll
   *
   * @param question
   * @param options
   * @param anonymous
   * @param hideVotes
   */
  fun create(question: String, options: List<String>, anonymous: Boolean, hideVotes: Boolean) {
    serialScope.launch {
      controller.pollsController.create(question, options, anonymous, hideVotes)
    }
  }
}
