package io.dyte.core.controllers

import io.dyte.callstats.CallStats
import io.dyte.core.controllers.PermissionType.CAMERA
import io.dyte.core.controllers.PermissionType.MICROPHONE
import io.dyte.core.events.JoinRoomEventPublisher
import io.dyte.core.models.DyteMeetingType.LIVESTREAM
import io.dyte.core.models.DyteMeetingType.WEBINAR
import io.dyte.core.models.WaitListStatus
import io.dyte.core.models.WaitListStatus.WAITING
import io.dyte.core.network.BaseApiService
import io.dyte.core.network.models.IceServerData
import io.dyte.core.observability.DyteLogger
import io.dyte.core.platform.IDyteMediaSoupUtils
import io.dyte.core.socket.SocketConnectionEvent
import io.dyte.core.socket.events.OutboundMeetingEventType
import io.dyte.core.socket.events.OutboundMeetingEventType.CREATE_WEB_RTC_TRANSPORT
import io.dyte.core.socket.events.OutboundMeetingEventType.GET_ROOM_STATE
import io.dyte.core.socket.events.OutboundMeetingEventType.GET_ROUTER_RTP_CAPABILITIES
import io.dyte.core.socket.events.payloadmodel.inbound.Device
import io.dyte.core.socket.events.payloadmodel.outbound.CreateWebRTCTransportPayloadRequestModel
import io.dyte.core.socket.events.payloadmodel.outbound.JoinRoomPayloadRequestModel
import io.dyte.core.socket.events.payloadmodel.outbound.RouterCapabilitiesModel
import io.dyte.core.socket.events.payloadmodel.outbound.WebRtcCreateTransportModel
import io.dyte.core.socket.events.payloadmodel.outbound.WebSocketJoinRoomModel
import io.dyte.core.socket.events.payloadmodel.outbound.WebSocketRoomStateModel
import io.dyte.core.socket.socketservice.SocketServiceUtils
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import kotlinx.serialization.json.Json
import kotlinx.serialization.json.encodeToJsonElement
import socket.room.JoinRoomRequest
import socket.room.Peer

internal class RoomNodeController(
  controllerContainer: IControllerContainer,
  private val callStatsClient: CallStats,
  private val joinRoomEventPublisher: JoinRoomEventPublisher? = null
) : IRoomNodeController, BaseController(controllerContainer) {

  private lateinit var iceServers: List<IceServerData>
  private lateinit var jsonParser: Json

  private lateinit var routerCapabilities: RouterCapabilitiesModel
  private lateinit var roomState: WebSocketRoomStateModel
  private lateinit var webRtcCreateTransportModelConsumer: WebRtcCreateTransportModel
  private lateinit var webRtcCreateTransportModelProducer: WebRtcCreateTransportModel
  private lateinit var joinRoom: WebSocketJoinRoomModel

  private var reconnectingToMeetingRoom = false

  private val socketConnectionEventListener =
    object : SocketConnectionEventListener {
      override fun onConnectionEvent(event: SocketConnectionEvent) {
        onSocketConnectionEvent(event)
      }
    }

  override fun init() {
    jsonParser = BaseApiService.json
    controllerContainer.socketController.addConnectionListener(socketConnectionEventListener)
    serialScope.launch {
      try {
        iceServers = getICEServers()
      } catch (e: Exception) {
        DyteLogger.error("RoomNodeController::init::ice_server_fetch_failed", e)
      }
    }
  }

  override suspend fun joinRoom() {
    withContext(serialScope.coroutineContext) {
      DyteLogger.info("SelfController.joinRoom")

      if (controllerContainer.selfController.getSelf().shouldJoinMediaRoom()) {
        try {
          getRoomState()
          _joinRoom()
          if (controllerContainer.selfController.getSelf().waitListStatus == WAITING) {
            return@withContext
          }
          connectToMediaConsumption()
          connectToMediaProduction()

          // Pick out active plugins, if any
          // controllerContainer.pluginsController.handleRoomState(roomState)
          // Notify other components who are listening to join-room event
          joinRoomEventPublisher?.publish()

          DyteLogger.info("room state processed")

          DyteLogger.info(
            "can produce audio ? ${controllerContainer.presetController.canPublishAudio()} } video ? ${controllerContainer.presetController.canPublishVideo()}"
          )

          val selfTracks =
            (controllerContainer.sfuUtils as? IDyteMediaSoupUtils)?.getSelfTrack()
              ?: Pair(null, null)
          val selfParticipant = controllerContainer.selfController.getSelf()
          // selfParticipant.audioTrack = selfTracks.first
          selfParticipant._videoTrack = selfTracks.second
          controllerContainer.selfController.onEnteredInRoom()

          if (controllerContainer.metaController.getMeetingType() == LIVESTREAM) {
            controllerContainer.liveStreamController.loadData()
          }

          controllerContainer.eventController.triggerEvent(
            DyteEventType.OnMeetingRoomJoined(controllerContainer.selfController.getSelf())
          )

          controllerContainer.eventController.triggerEvent(DyteEventType.OnMetaUpdate)
          DyteLogger.info("joined!")

          if (controllerContainer.metaController.getMeetingType() == WEBINAR) {
            controllerContainer.stageController.stage.join()
          }

          if (controllerContainer.permissionController.isPermissionGrated(CAMERA).not()) {
            DyteLogger.info("joined without camera permission")
            controllerContainer.eventController.triggerEvent(
              DyteEventType.OnMeetingRoomJoinedWithoutCameraPermission
            )
          }
          if (controllerContainer.permissionController.isPermissionGrated(MICROPHONE).not()) {
            DyteLogger.info("joined without microphone permission")
            controllerContainer.eventController.triggerEvent(
              DyteEventType.OnMeetingRoomJoinedWithoutMicPermission
            )
          }
          controllerContainer.pollsController.loadPolls()
          controllerContainer.chatController.loadChatMessages()
          controllerContainer.platformUtilsProvider.getMediaUtils().routeAudioFromSpeakerphone()
          controllerContainer.participantController.setPage(0)
        } catch (e: Exception) {
          DyteLogger.error("RoomNodeController::join_room_error", e)
          controllerContainer.eventController.triggerEvent(DyteEventType.OnMeetingRoomJoinFailed(e))
        }
      } else {
        if (controllerContainer.metaController.getMeetingType() != LIVESTREAM) {
          getRoomState()
          _joinRoom()
          if (controllerContainer.selfController.getSelf().waitListStatus == WAITING) {
            return@withContext
          }
          controllerContainer.selfController.onEnteredInRoom()
          connectToMediaConsumption()
          controllerContainer.participantController.handleRoomJoined(joinRoom)
          controllerContainer.participantController.setPage(0)
        }
        if (controllerContainer.selfController.getSelf().waitListStatus == WAITING) {
          return@withContext
        }
        controllerContainer.pollsController.loadPolls()
        controllerContainer.chatController.loadChatMessages()
        controllerContainer.eventController.triggerEvent(
          DyteEventType.OnMeetingRoomJoined(controllerContainer.selfController.getSelf())
        )
        controllerContainer.eventController.triggerEvent(DyteEventType.OnMetaUpdate)
      }
    }
  }

  override fun leaveRoom() {
    controllerContainer.selfController.onLeftFromRoom()
  }

  private suspend fun getICEServers(): List<IceServerData> {
    return requireNotNull(controllerContainer.apiClient.getICEServers().iceServers)
  }

  private fun onSocketConnectionEvent(event: SocketConnectionEvent) {
    when (event) {
      SocketConnectionEvent.Connected -> {
        serialScope.launch {
          DyteLogger.info("RoomNodeController::SOCKET_CONNECTED")
          /*
           * We emit OnConnectedToMeetingRoom event only when we connect to RoomNode socket for 1st time.
           * i.e. during meeting.init()
           * */
          if (!reconnectingToMeetingRoom) {
            controllerContainer.eventController.triggerEvent(DyteEventType.OnConnectedToMeetingRoom)
          }
        }
      }
      SocketConnectionEvent.Connecting -> {
        DyteLogger.info("RoomNodeController::SOCKET_CONNECTING")
        controllerContainer.eventController.triggerEvent(DyteEventType.OnConnectingToMeetingRoom)
      }
      SocketConnectionEvent.ConnectionFailed -> {
        DyteLogger.info("RoomNodeController::SOCKET_CONNECTION_ERROR")
        controllerContainer.eventController.triggerEvent(
          DyteEventType.OnMeetingRoomConnectionFailed
        )
      }
      is SocketConnectionEvent.Disconnected -> {
        serialScope.launch {
          DyteLogger.info("RoomNodeController::SOCKET_DISCONNECTED")
          /*
           * We emit OnDisconnectedFromMeetingRoom event only when we leave the meeting room.
           * */
          if (!reconnectingToMeetingRoom) {
            controllerContainer.eventController.triggerEvent(
              DyteEventType.OnDisconnectedFromMeetingRoom
            )
          }
        }
      }
      SocketConnectionEvent.Reconnected -> {
        serialScope.launch {
          DyteLogger.info("RoomNodeController::SOCKET_CONNECTED")

          /*
           * note(swapnil): On socketIo reconnection, we are connecting to both RoomNode socket &
           * SocketService using a new peerId as a workaround to make the socket-service connection
           * work properly after reconnection.
           * */
          controllerContainer.socketController.disconnect()
          val newPeerId = controllerContainer.metaController.refreshPeerId()
          reconnectSocketsWithNewPeerId(newPeerId)

          // We shouldn't call joinRoom if the local user was outside the meeting room before the
          // network drop
          if (controllerContainer.selfController.roomJoined) {
            (controllerContainer.sfuUtils as? IDyteMediaSoupUtils)?.handlePreReconnection()
            sendJoinRoomMessageToSocketService()
            joinRoom()
            DyteLogger.info("RoomNodeController | Rejoined meeting room on network reconnection")
          } else {
            DyteLogger.info(
              "RoomNodeController | not Rejoined meeting room on network reconnection"
            )
          }
          reconnectingToMeetingRoom = false
          controllerContainer.eventController.triggerEvent(DyteEventType.OnReconnectedToMeetingRoom)
        }
      }
      SocketConnectionEvent.Reconnecting -> {
        serialScope.launch {
          reconnectingToMeetingRoom = true
          DyteLogger.info(
            "RoomNodeController | Detected network issue. Reconnecting to the meeting room"
          )

          /*
           * Manually disconnect from SocketService as we want to keep its connection
           * in-sync with SocketIO, so that when reconnection happens for a MediaSoup meeting,
           * we are connected to SocketService before calling joinRoom().
           * */
          controllerContainer.socketService.disconnect()

          // We shouldn't leave call if the local user was outside the meeting room before the
          // network drop
          if (controllerContainer.selfController.roomJoined) {
            controllerContainer.stageController.onRemovedFromStage()
            controllerContainer.selfController.getSelf()._audioEnabled = false
            controllerContainer.selfController.getSelf()._videoEnabled = false
            controllerContainer.platformUtilsProvider.getVideoUtils().destroyAll()
            controllerContainer.sfuUtils.leaveCall()
          }
          controllerContainer.eventController.triggerEvent(
            DyteEventType.OnReconnectingToMeetingRoom
          )
        }
      }
      is SocketConnectionEvent.ReconnectionAttempt -> {
        DyteLogger.info(
          "RoomNodeController | Attempt ${event.attemptNumber} to reconnect to meeting room"
        )
      }
      is SocketConnectionEvent.ReconnectionAttemptFailed -> {
        DyteLogger.info(
          "RoomNodeController | Attempt ${event.attemptNumber} failed to reconnect to meeting room"
        )
      }
      SocketConnectionEvent.ReconnectionFailed -> {
        DyteLogger.info("RoomNodeController | Failed to reconnect to the meeting room")
        controllerContainer.eventController.triggerEvent(
          DyteEventType.OnMeetingRoomReconnectionFailed
        )
      }
    }
  }

  override suspend fun connectToMediaProduction() {
    val producerResponse =
      controllerContainer.socketController.sendMessage(
        CREATE_WEB_RTC_TRANSPORT,
        jsonParser.encodeToJsonElement(
          CreateWebRTCTransportPayloadRequestModel(
            forceTcp = false,
            producing = true,
            consuming = false
          )
        )
      )
    webRtcCreateTransportModelProducer =
      controllerContainer.socketMessageResponseParser.parseResponse(producerResponse).payload
        as WebRtcCreateTransportModel
    (controllerContainer.sfuUtils as? IDyteMediaSoupUtils)?.createWebRtcTransportProd(
      webRtcCreateTransportModelProducer,
      iceServers
    )
    DyteLogger.info("created produce transport")
  }

  private suspend fun connectToMediaConsumption() {
    (controllerContainer.sfuUtils as? IDyteMediaSoupUtils)?.loadRouterRtpCapabilities(
      routerCapabilities
    )

    val consumerResponse =
      controllerContainer.socketController.sendMessage(
        CREATE_WEB_RTC_TRANSPORT,
        jsonParser.encodeToJsonElement(
          CreateWebRTCTransportPayloadRequestModel(
            forceTcp = false,
            producing = false,
            consuming = true
          )
        )
      )

    webRtcCreateTransportModelConsumer =
      (controllerContainer.socketMessageResponseParser.parseResponse(consumerResponse).payload
        as WebRtcCreateTransportModel)
    (controllerContainer.sfuUtils as? IDyteMediaSoupUtils)?.createWebRtcTransportRecv(
      webRtcCreateTransportModelConsumer,
      iceServers
    )

    DyteLogger.info("created receive transport")
  }

  private suspend fun getRoomState() {
    val roomStateResponse = controllerContainer.socketController.sendMessage(GET_ROOM_STATE, null)
    roomState =
      requireNotNull(
        controllerContainer.socketMessageResponseParser.parseResponse(roomStateResponse).payload
          as WebSocketRoomStateModel
      )
    DyteLogger.info("room state received")

    // check for remote peers joined before local user
    controllerContainer.participantController.handleRoomState(roomState)
  }

  private suspend fun _joinRoom() {
    val routerCapabilitiesResponse =
      controllerContainer.socketController.sendMessage(GET_ROUTER_RTP_CAPABILITIES, null)
    routerCapabilities =
      (controllerContainer.socketMessageResponseParser
        .parseResponse(routerCapabilitiesResponse)
        .payload as RouterCapabilitiesModel)
    DyteLogger.info("received and processed router capabilities")

    val joinRoomPayload = JoinRoomPayloadRequestModel()
    joinRoomPayload.device =
      Device(
        osName = controllerContainer.platformUtilsProvider.getPlatformUtils().getOsName(),
        osVersionName = controllerContainer.platformUtilsProvider.getPlatformUtils().getOsVersion()
      )
    joinRoomPayload.displayName = controllerContainer.selfController.getSelf().name
    joinRoomPayload.audioMuted = !controllerContainer.selfController.getSelf().audioEnabled
    joinRoomPayload.rtpCapabilities = routerCapabilities
    val payload = jsonParser.encodeToJsonElement(joinRoomPayload)
    val joinRoomResponse =
      controllerContainer.socketController.sendMessage(OutboundMeetingEventType.JOIN_ROOM, payload)

    joinRoom =
      controllerContainer.socketMessageResponseParser.parseResponse(joinRoomResponse).payload
        as WebSocketJoinRoomModel

    DyteLogger.info("join room done!")

    if (
      joinRoom.waitlisted == true &&
        controllerContainer.selfController.getSelf()._waitListStatus != WaitListStatus.ACCEPTED
    ) {
      DyteLogger.info("aww snap, I'm in waiting room!")
      controllerContainer.selfController.getSelf()._waitListStatus = WAITING
      controllerContainer.eventController.triggerEvent(
        DyteEventType.OnSelfWaitListStatusUpdate(WAITING)
      )
      return
    }

    controllerContainer.participantController.handleRoomJoined(joinRoom)
    controllerContainer.waitlistController.handleRoomJoined(joinRoom)
    controllerContainer.stageController.handleRoomJoined(joinRoom)
    // TODO : fix this
    //    controllerContainer.stageController.setStageStatus(StageStatus.ON_STAGE)
    controllerContainer.metaController.setMeetingStartedTimestamp(
      requireNotNull(joinRoom.startedAt)
    )
  }

  private suspend fun sendJoinRoomMessageToSocketService() {
    val joinRoomRequest =
      JoinRoomRequest(
        peer =
          Peer(
            peer_id = controllerContainer.metaController.getPeerId(),
            display_name = controllerContainer.selfController.getSelf().name,
            user_id = controllerContainer.selfController.getSelf().userId
          ),
        room_uuid = controllerContainer.metaController.getRoomUuid()
      )

    try {
      val response =
        controllerContainer.socketService.requestResponse(
          event = SocketServiceUtils.RoomEvent.JOIN_ROOM.id,
          payload = JoinRoomRequest.ADAPTER.encode(joinRoomRequest)
        )
    } catch (e: Exception) {
      DyteLogger.error("RoomNodeController::sendJoinRoomMessageToSocketService::failed", e)
    }
  }

  /**
   * Changes socket Urls of both SocketController and SocketService client to use the new peerId and
   * reconnects to both RoomNode socket and SocketService.
   */
  private suspend fun reconnectSocketsWithNewPeerId(peerId: String) {
    controllerContainer.socketController.refreshUrl(peerId)
    controllerContainer.socketService.refreshUrl(peerId)

    // Connect again to RoomNodeSocket after changing the Url
    controllerContainer.socketController.connect()

    /*
     * note(swapnil): Manually connect to SocketService on SocketIO reconnection rather than
     * depending on Sockrates' auto-reconnection.
     * We want to make sure that we are connected to SocketService before calling joinRoom() while
     * reconnecting to a MediaSoup meeting.
     *
     * If we rely on Sockrates' auto-reconnection then there could be a scenario where SocketIO
     * auto-reconnects before Sockrates and the RoomNodeController starts rejoining the room.
     * Since mobile-core is not connected to Sockrates yet, it won't be able to send JOIN_ROOM message
     * or load Chat, Active Plugins, Polls from SocketService (if-flag-enabled).
     * */
    controllerContainer.socketService.connect()
  }
}
