package io.dyte.core.chat

import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketChatMessage
import kotlinx.serialization.json.Json
import kotlinx.serialization.json.jsonObject
import kotlinx.serialization.json.jsonPrimitive
import kotlinx.serialization.json.long
import socket.chat.ChatMessage

internal sealed class SocketChatMessage(
  val type: MessageType,
  val userId: String,
  val displayName: String,
  val time: Long,
  val id: String,
  val edited: Boolean,
  val read: Boolean,
  val pluginId: String?,
  val pinned: Boolean,
  val targetUserIds: List<String>?
) {
  enum class MessageType(val type: Int) {
    TEXT(0),
    IMAGE(1),
    FILE(2)
  }
}

internal class TextSocketChatMessage(
  val message: String,
  userId: String,
  displayName: String,
  time: Long,
  id: String,
  edited: Boolean = false,
  read: Boolean = false,
  pluginId: String? = null,
  pinned: Boolean = false,
  targetUserIds: List<String>? = null
) :
  SocketChatMessage(
    type = MessageType.TEXT,
    userId,
    displayName,
    time,
    id,
    edited,
    read,
    pluginId,
    pinned,
    targetUserIds
  )

internal class ImageSocketChatMessage(
  val link: String,
  userId: String,
  displayName: String,
  time: Long,
  id: String,
  edited: Boolean = false,
  read: Boolean = false,
  pluginId: String? = null,
  pinned: Boolean = false,
  targetUserIds: List<String>? = null
) :
  SocketChatMessage(
    type = MessageType.IMAGE,
    userId,
    displayName,
    time,
    id,
    edited,
    read,
    pluginId,
    pinned,
    targetUserIds
  )

internal class FileSocketChatMessage(
  val name: String,
  val size: Long,
  val link: String,
  userId: String,
  displayName: String,
  time: Long,
  id: String,
  edited: Boolean = false,
  read: Boolean = false,
  pluginId: String? = null,
  pinned: Boolean = false,
  targetUserIds: List<String>? = null
) :
  SocketChatMessage(
    type = MessageType.FILE,
    userId,
    displayName,
    time,
    id,
    edited,
    read,
    pluginId,
    pinned,
    targetUserIds
  )

internal object SocketChatMessageMapper {
  fun fromSocketIoChatMessage(socketIoChatMessage: WebSocketChatMessage): SocketChatMessage {
    when (socketIoChatMessage.type) {
      0 -> {
        return TextSocketChatMessage(
          message = socketIoChatMessage.message ?: error("Text message is null"),
          userId = socketIoChatMessage.userId,
          displayName = socketIoChatMessage.displayName,
          time = socketIoChatMessage.time ?: error("Message time is null"),
          id = ""
        )
      }
      1 -> {
        return ImageSocketChatMessage(
          link = socketIoChatMessage.link ?: "",
          userId = socketIoChatMessage.userId,
          displayName = socketIoChatMessage.displayName,
          time = socketIoChatMessage.time ?: error("Message time is null"),
          id = ""
        )
      }
      2 -> {
        return FileSocketChatMessage(
          name = socketIoChatMessage.name ?: "",
          size = socketIoChatMessage.size ?: 0L,
          link = socketIoChatMessage.link ?: "",
          userId = socketIoChatMessage.userId,
          displayName = socketIoChatMessage.displayName,
          time = socketIoChatMessage.time ?: error("Message time is null"),
          id = ""
        )
      }
      else -> {
        throw UnsupportedOperationException(
          "Message type ${socketIoChatMessage.type} not supported"
        )
      }
    }
  }

  fun fromSocketServiceChatMessage(socketServerChatMessage: ChatMessage): SocketChatMessage {
    when (socketServerChatMessage.payload_type) {
      0 -> {
        return TextSocketChatMessage(
          message = socketServerChatMessage.payload,
          userId = socketServerChatMessage.user_id,
          displayName = socketServerChatMessage.display_name,
          time = socketServerChatMessage.created_at,
          id = socketServerChatMessage.chat_id
        )
      }
      1 -> {
        return ImageSocketChatMessage(
          link = socketServerChatMessage.payload,
          userId = socketServerChatMessage.user_id,
          displayName = socketServerChatMessage.display_name,
          time = socketServerChatMessage.created_at,
          id = socketServerChatMessage.chat_id
        )
      }
      2 -> {
        val fileMessagePayload = Json.parseToJsonElement(socketServerChatMessage.payload).jsonObject

        return FileSocketChatMessage(
          name = fileMessagePayload.getValue("name").jsonPrimitive.content,
          size = fileMessagePayload.getValue("size").jsonPrimitive.long,
          link = fileMessagePayload.getValue("link").jsonPrimitive.content,
          userId = socketServerChatMessage.user_id,
          displayName = socketServerChatMessage.display_name,
          time = socketServerChatMessage.created_at,
          id = socketServerChatMessage.chat_id
        )
      }
      else -> {
        throw UnsupportedOperationException(
          "Message type ${socketServerChatMessage.payload_type} not supported"
        )
      }
    }
  }
}
