package io.dyte.core.feat

import io.dyte.core.controllers.IControllerContainer
import io.dyte.core.controllers.StageStatus
import io.dyte.core.models.DyteJoinedMeetingParticipant
import io.dyte.core.socket.socketservice.IStageSocketHandler
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.launch
import kotlinx.coroutines.newSingleThreadContext

class DyteStage
internal constructor(
  private val controllerContainer: IControllerContainer,
  private val stageSocketHandler: IStageSocketHandler
) {
  private val serialScope = CoroutineScope(newSingleThreadContext("DyteStage"))

  val accessRequests: List<DyteJoinedMeetingParticipant>
    get() = stageSocketHandler.requestedParticipants

  val viewers: List<DyteJoinedMeetingParticipant>
    get() = _getViewers()

  val onStage: List<DyteJoinedMeetingParticipant>
    get() = controllerContainer.participantController.meetingRoomParticipants.active

  val stageStatus: StageStatus
    get() = stageSocketHandler.stageStatus

  private fun _getViewers(): List<DyteJoinedMeetingParticipant> {
    val active =
      controllerContainer.participantController.meetingRoomParticipants.active.map { it.id }
    val joined = controllerContainer.participantController.meetingRoomParticipants.joined
    val viewers = ArrayList(joined)
    viewers.removeAll { active.contains(it.id) }
    return viewers
  }

  fun requestAccess() {
    serialScope.launch { stageSocketHandler.requestAccess() }
  }

  fun cancelRequestAccess() {
    serialScope.launch { stageSocketHandler.cancelRequestAccess() }
  }

  fun join() {
    serialScope.launch {
      try {
        stageSocketHandler.join()
      } catch (e: Exception) {
        e.printStackTrace()
      }
    }
  }

  fun leave() {
    serialScope.launch { stageSocketHandler.leave() }
  }

  fun kick(id: String) {
    serialScope.launch { stageSocketHandler.kick(id) }
  }

  fun grantAccess(id: String) {
    serialScope.launch { stageSocketHandler.grantAccess(listOf(id)) }
  }

  fun grantAccessAll() {
    serialScope.launch {
      val requestIds = accessRequests.map { it.id }
      stageSocketHandler.grantAccess(requestIds)
    }
  }

  fun denyAccess(id: String) {
    serialScope.launch { stageSocketHandler.denyAccess(listOf(id)) }
  }

  fun denyAccessAll() {
    serialScope.launch {
      val requestIds = accessRequests.map { it.id }
      stageSocketHandler.denyAccess(requestIds)
    }
  }
}
