package io.dyte.core.socket.socketservice

import io.dyte.sockrates.client.*
import io.dyte.sockrates.client.SockratesResult.Failure
import io.dyte.sockrates.client.SockratesResult.Success
import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext
import message.v1.SocketMessage
import kotlin.coroutines.cancellation.CancellationException
import kotlin.random.Random

interface SocketService {
  suspend fun connect()

  suspend fun disconnect()

  suspend fun send(event: Int, payload: ByteArray?, messageId: String? = null)

  @Throws(ResponseTimeoutException::class, CancellationException::class)
  suspend fun requestResponse(event: Int, payload: ByteArray?, messageId: String? = null): ByteArray?

  fun subscribe(event: Int, listener: SocketServiceEventListener)

  fun unsubscribe(event: Int, listener: SocketServiceEventListener)

  fun addConnectionStateListener(listener: SocketServiceConnectionStateListener)

  fun removeConnectionStateListener(listener: SocketServiceConnectionStateListener)
}

interface SocketServiceEventListener {
  fun onEvent(event: Int, eventId: String?, payload: ByteArray?)
}

interface SocketServiceConnectionStateListener {
  fun onConnectionStateChanged(newState: SocketServiceConnectionState)
}

class ResponseTimeoutException(override val message: String) : RuntimeException(message)

internal class DefaultSocketService(
  baseUrl: String,
  private val peerId: String,
  roomName: String,
  authToken: String,
  useHive: Boolean,
  private val workContext: CoroutineDispatcher = Dispatchers.Default,
) : SocketService {
  private val url: String
  private val wsClient: Sockrates
  // private val mainScope = MainScope()

  private val activeEventListeners: MutableMap<Int, MutableSet<SocketServiceEventListener>> =
    HashMap()

  private val connectionStateListeners: MutableSet<SocketServiceConnectionStateListener> =
    LinkedHashSet()

  init {
    url = createSocketServiceUrl(baseUrl, peerId, roomName, authToken, useHive)
    wsClient = Sockrates(url, config = SockratesConfiguration(disconnectOnPingTimeout = false))
  }

  override suspend fun connect() {
    withContext(workContext) {
      wsClient.connect(object : SockratesWSListener {
        override fun onConnectionStateChanged(
          client: Sockrates,
          newState: WebSocketConnectionState
        ) {
          super.onConnectionStateChanged(client, newState)
          try {
            val socketServiceConnectionState =
              SocketServiceConnectionState.createFromSockratesConnectionState(newState)
            notifyConnectionStateListeners(socketServiceConnectionState)
          } catch (e: IllegalArgumentException) {
            // no-op
          }
        }

        override fun onMessage(client: Sockrates, message: SocketMessage) {
          super.onMessage(client, message)
          notifyEventSubscribers(message)
        }
      })
    }
  }

  override suspend fun disconnect() {
    withContext(workContext) {
      wsClient.disconnect()
      clear()
    }
  }

  override suspend fun send(event: Int, payload: ByteArray?, messageId: String?) {
    withContext(workContext) {
      wsClient.send(
        event = event,
        messageId = messageId ?: generateMessageId(peerId),
        payload = payload
      )
    }
  }

  override suspend fun requestResponse(
    event: Int,
    payload: ByteArray?,
    messageId: String?
  ): ByteArray? {
    return withContext(workContext) {
      val result = wsClient.requestResponse(
        event = event,
        messageId = messageId ?: generateMessageId(peerId),
        payload = payload
      )
      when (result) {
        is Success -> {
          return@withContext result.value.payload?.toByteArray()
        }
        is Failure -> {
          val reason = result.reason
          when (reason) {
            is RequestResponseFailureReason.ResponseTimeout -> {
              throw ResponseTimeoutException("SocketService response timeout after ${reason.timeoutInMillis}ms")
            }
            is RequestResponseFailureReason.SocketNotConnected -> {
              return@withContext null
            }
            is RequestResponseFailureReason.Other -> {
              return@withContext null
            }
          }
        }
      }
    }
  }

  override fun subscribe(event: Int, listener: SocketServiceEventListener) {
    val existingEventListeners = activeEventListeners[event]
    if (existingEventListeners == null) {
      val newEventListenersSet = LinkedHashSet<SocketServiceEventListener>()
      newEventListenersSet.add(listener)
      activeEventListeners[event] = newEventListenersSet
    } else {
      existingEventListeners.add(listener)
    }
  }

  override fun unsubscribe(event: Int, listener: SocketServiceEventListener) {
    val existingEventListeners = activeEventListeners[event]
    if (existingEventListeners != null) {
      existingEventListeners.remove(listener)
    }
  }

  override fun addConnectionStateListener(listener: SocketServiceConnectionStateListener) {
    connectionStateListeners.add(listener)
  }

  override fun removeConnectionStateListener(listener: SocketServiceConnectionStateListener) {
    connectionStateListeners.remove(listener)
  }

  private fun notifyEventSubscribers(socketMessage: SocketMessage) {
    if (activeEventListeners.isEmpty()) return

    val eventListeners = activeEventListeners[socketMessage.event]
    if (eventListeners != null && eventListeners.isNotEmpty()) {
      for (listener in eventListeners) {
        listener.onEvent(socketMessage.event, socketMessage.id, socketMessage.payload?.toByteArray())
      }
    }
  }

  private fun notifyConnectionStateListeners(connectionState: SocketServiceConnectionState) {
    connectionStateListeners.forEach {
      it.onConnectionStateChanged(connectionState)
    }
  }

  private fun clear() {
    activeEventListeners.clear()
    connectionStateListeners.clear()
  }

  companion object {
    private fun createSocketServiceUrl(
      baseUrl: String,
      peerId: String,
      roomName: String,
      authToken: String,
      useHive: Boolean,
    ): String {
      return "$baseUrl/ws?roomID=${roomName}&peerID=${peerId}&authToken=${authToken}&useMediaV2=${useHive}"
    }

    private fun generateMessageId(peerId: String): String {
      val randomSuffix = generateRandomString(5)
      return "${peerId}-${randomSuffix}"
    }

    private val characterSet = charArrayOf(
      '0', '1', '2', '3', '4', '5',
      '6', '7', '8', '9', 'a', 'b',
      'c', 'd', 'e', 'f', 'g', 'h',
      'i', 'j', 'k', 'l', 'm', 'n',
      'o', 'p', 'q', 'r', 's', 't',
      'u', 'v', 'w', 'x', 'y', 'z'
    )

    /*
    This is not finalised yet. Currently using the most straightforward method to
    generate a random 5 character alphanumeric string.
    Will need to come-up with a better efficient method or implement Base36 encoding in
    Kotlin.
     */
    private fun generateRandomString(idLength: Int = 5): String {
      val idBuilder: StringBuilder = StringBuilder()
      var randomIndex = 0
      for (i in 0 until idLength) {
        randomIndex = (Random.nextInt(until = characterSet.size))
        idBuilder.append(characterSet[randomIndex])
      }

      return idBuilder.toString()
    }
  }
}