package io.dyte.core.controllers

import defaults.BuildKonfig
import io.dyte.callstats.CallStats
import io.dyte.callstats.events.DeviceInfo
import io.dyte.callstats.events.PeerMetaData
import io.dyte.core.controllers.PermissionType.CAMERA
import io.dyte.core.controllers.PermissionType.MICROPHONE
import io.dyte.core.events.JoinRoomEventPublisher
import io.dyte.core.models.DyteMeetingType.LIVESTREAM
import io.dyte.core.models.DyteMeetingType.WEBINAR
import io.dyte.core.models.WaitListStatus.WAITING
import io.dyte.core.network.BaseApiService
import io.dyte.core.network.models.IceServerData
import io.dyte.core.platform.IDyteMediaSoupUtils
import io.dyte.core.socket.SocketConnectionEvent
import io.dyte.core.socket.events.OutboundMeetingEventType
import io.dyte.core.socket.events.OutboundMeetingEventType.CREATE_WEB_RTC_TRANSPORT
import io.dyte.core.socket.events.OutboundMeetingEventType.GET_ROOM_STATE
import io.dyte.core.socket.events.OutboundMeetingEventType.GET_ROUTER_RTP_CAPABILITIES
import io.dyte.core.socket.events.payloadmodel.inbound.Device
import io.dyte.core.socket.events.payloadmodel.outbound.CreateWebRTCTransportPayloadRequestModel
import io.dyte.core.socket.events.payloadmodel.outbound.JoinRoomPayloadRequestModel
import io.dyte.core.socket.events.payloadmodel.outbound.RouterCapabilitiesModel
import io.dyte.core.socket.events.payloadmodel.outbound.WebRtcCreateTransportModel
import io.dyte.core.socket.events.payloadmodel.outbound.WebSocketJoinRoomModel
import io.dyte.core.socket.events.payloadmodel.outbound.WebSocketRoomStateModel
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import kotlinx.serialization.json.Json
import kotlinx.serialization.json.encodeToJsonElement

internal class RoomNodeController(
  controllerContainer: IControllerContainer,
  private val callStatsClient: CallStats,
  private val joinRoomEventPublisher: JoinRoomEventPublisher? = null
) : IRoomNodeController, BaseController(controllerContainer) {

  private lateinit var iceServers: List<IceServerData>
  private lateinit var jsonParser: Json

  private lateinit var routerCapabilities: RouterCapabilitiesModel
  private lateinit var roomState: WebSocketRoomStateModel
  private lateinit var webRtcCreateTransportModelConsumer: WebRtcCreateTransportModel
  private lateinit var webRtcCreateTransportModelProducer: WebRtcCreateTransportModel
  private lateinit var joinRoom: WebSocketJoinRoomModel

  private var roomJoined: Boolean = false

  private val socketConnectionEventListener = object : SocketConnectionEventListener {
    override fun onConnectionEvent(event: SocketConnectionEvent) {
      onSocketConnectionEvent(event)
    }
  }

  override fun init() {
    jsonParser = BaseApiService.json
    controllerContainer.socketController.addConnectionListener(socketConnectionEventListener)
    serialScope.launch {
      try {
        iceServers = getICEServers()
      } catch (e: Exception) {
        controllerContainer.loggerController.traceError("error in getting ice servers ${e.message}")
      }
    }
  }

  override suspend fun joinRoom() {
    withContext(serialScope.coroutineContext) {
      controllerContainer.loggerController.traceLog("SelfController.joinRoom")

      if (controllerContainer.selfController.getSelf().shouldJoinMediaRoom()) {
        try {
          // Created Media Soup devices and load RTP Capability
          _joinRoom()

          withContext(Dispatchers.Main) {
            (controllerContainer.sfuUtils as? IDyteMediaSoupUtils)
              ?.loadRouterRtpCapabilities(routerCapabilities)
          }

          controllerContainer.loggerController.traceLog("join room done!")

          if (joinRoom.waitlisted == true) {
            controllerContainer.loggerController.traceLog("aww snap, I'm in waiting room!")
            controllerContainer.selfController.getSelf()._waitListStatus = WAITING
            controllerContainer.eventController.triggerEvent(
              DyteEventType.OnSelfWaitListStatusUpdate(
                WAITING
              )
            )
            return@withContext
          }

          controllerContainer.loggerController.traceLog(
            "loaded router rtp capabilities into media soup device"
          )

          val consumerResponse = controllerContainer.socketController.sendMessage(
            CREATE_WEB_RTC_TRANSPORT, jsonParser.encodeToJsonElement(
              CreateWebRTCTransportPayloadRequestModel(
                forceTcp = false,
                producing = false,
                consuming = true
              )
            )
          )


          webRtcCreateTransportModelConsumer =
            (controllerContainer.socketMessageResponseParser.parseResponse(
              consumerResponse
            ).payload as WebRtcCreateTransportModel)
          (controllerContainer.sfuUtils as? IDyteMediaSoupUtils)
            ?.createWebRtcTransportRecv(webRtcCreateTransportModelConsumer, iceServers)


          controllerContainer.loggerController.traceLog("created receive transport")

          val producerResponse = controllerContainer.socketController.sendMessage(
            CREATE_WEB_RTC_TRANSPORT, jsonParser.encodeToJsonElement(
              CreateWebRTCTransportPayloadRequestModel(
                forceTcp = false,
                producing = true,
                consuming = false
              )
            )
          )


          webRtcCreateTransportModelProducer =
            controllerContainer.socketMessageResponseParser.parseResponse(
              producerResponse
            ).payload as WebRtcCreateTransportModel
          (controllerContainer.sfuUtils as? IDyteMediaSoupUtils)
            ?.createWebRtcTransportProd(webRtcCreateTransportModelProducer, iceServers)
          controllerContainer.loggerController.traceLog("created produce transport")

          val roomStateResponse =
            controllerContainer.socketController.sendMessage(GET_ROOM_STATE, null)
          roomState =
            requireNotNull(
              controllerContainer.socketMessageResponseParser.parseResponse(
                roomStateResponse
              ).payload as WebSocketRoomStateModel
            )
          controllerContainer.loggerController.traceLog("room state received")

          // check for remote peers joined before local user
          controllerContainer.participantController.handleRoomState(roomState)
          // Pick out active plugins, if any
          // controllerContainer.pluginsController.handleRoomState(roomState)
          // Notify other components who are listening to join-room event
          joinRoomEventPublisher?.publish()

          controllerContainer.loggerController.traceLog("room state processed")

          controllerContainer.participantController.handleRoomJoined(joinRoom)
          controllerContainer.waitlistController.handleRoomJoined(joinRoom)
          controllerContainer.stageController.handleRoomJoined(joinRoom)

          controllerContainer.metaController.setMeetingStartedTimestamp(
            requireNotNull(joinRoom.startedAt)
          )

          controllerContainer.loggerController.traceLog(
            "can produce audio ? ${controllerContainer.presetController.canPublishAudio()} } video ? ${controllerContainer.presetController.canPublishVideo()}"
          )

          val selfTracks =
            (controllerContainer.sfuUtils as? IDyteMediaSoupUtils)?.getSelfTrack()
              ?: Pair(null, null)
          val selfParticipant = controllerContainer.selfController.getSelf()
          // selfParticipant.audioTrack = selfTracks.first
          selfParticipant._videoTrack = selfTracks.second
          controllerContainer.selfController.onEnteredInRoom()

          controllerContainer.eventController.triggerEvent(
            DyteEventType.OnMeetingRoomJoined(
              controllerContainer.selfController.getSelf()
            )
          )

          controllerContainer.eventController.triggerEvent(DyteEventType.OnMetaUpdate)
          controllerContainer.loggerController.traceLog("joined!")

          if (controllerContainer.metaController.getMeetingType() == WEBINAR && controllerContainer.webinarController.shouldJoinStage()) {
            controllerContainer.stageController.joinStage()
          }

          if (controllerContainer.permissionController.isPermissionGrated(CAMERA).not()) {
            controllerContainer.loggerController.traceLog("joined without camera permission")
            controllerContainer.eventController.triggerEvent(
              DyteEventType.OnMeetingRoomJoinedWithoutCameraPermission
            )
          }
          if (controllerContainer.permissionController.isPermissionGrated(MICROPHONE).not()) {
            controllerContainer.loggerController.traceLog("joined without microphone permission")
            controllerContainer.eventController.triggerEvent(
              DyteEventType.OnMeetingRoomJoinedWithoutMicPermission
            )
          }
          controllerContainer.pollsController.loadPolls()
          controllerContainer.chatController.loadChatMessages()

          controllerContainer.platformUtilsProvider.getMediaUtils().routeAudioFromSpeakerphone()

          controllerContainer.participantController.setPage(0)

          roomJoined = true
        } catch (e: Exception) {
          e.printStackTrace()
          controllerContainer.eventController.triggerEvent(DyteEventType.OnMeetingRoomJoinFailed(e))
        }
      } else {
        /*if(controllerContainer.metaController.getMeetingType() == LIVESTREAM) {
          _joinRoom()
        }*/
        controllerContainer.chatController.loadChatMessages()
        controllerContainer.eventController.triggerEvent(
          DyteEventType.OnMeetingRoomJoined(
            controllerContainer.selfController.getSelf()
          )
        )
        controllerContainer.eventController.triggerEvent(DyteEventType.OnMetaUpdate)
      }
    }
  }

  override fun leaveRoom() {
    roomJoined = false
  }

  private suspend fun getICEServers(): List<IceServerData> {
    return requireNotNull(controllerContainer.apiClient.getICEServers().iceServers)
  }

  private fun onSocketConnectionEvent(event: SocketConnectionEvent) {
    when (event) {
      SocketConnectionEvent.Connected -> {
        controllerContainer.loggerController.traceLog("RoomNodeController | Connected to the meeting room")
        controllerContainer.eventController.triggerEvent(DyteEventType.OnConnectedToMeetingRoom)
      }

      SocketConnectionEvent.Connecting -> {
        controllerContainer.loggerController.traceLog("RoomNodeController | Connecting to the meeting room")
        controllerContainer.eventController.triggerEvent(DyteEventType.OnConnectingToMeetingRoom)
      }

      SocketConnectionEvent.ConnectionFailed -> {
        controllerContainer.loggerController.traceLog("RoomNodeController | Failed to connect to the meeting room")
        controllerContainer.eventController.triggerEvent(DyteEventType.OnMeetingRoomConnectionFailed)
      }

      is SocketConnectionEvent.Disconnected -> {
        controllerContainer.loggerController.traceLog("RoomNodeController | Disconnected from the meeting room")
        controllerContainer.eventController.triggerEvent(DyteEventType.OnDisconnectedFromMeetingRoom)
      }

      SocketConnectionEvent.Reconnected -> {
        serialScope.launch {
          controllerContainer.loggerController.traceLog("RoomNodeController | Reconnected to the meeting room")
          // We shouldn't call joinRoom if the local user was outside the meeting room before the network drop
          val localUserWasInRoom = roomJoined
          if (localUserWasInRoom) {
            (controllerContainer.sfuUtils as? IDyteMediaSoupUtils)?.handlePreReconnection()
            joinRoom()
            controllerContainer.loggerController.traceLog(
              "RoomNodeController | Rejoined meeting room on network reconnection"
            )
          }
          controllerContainer.eventController.triggerEvent(DyteEventType.OnReconnectedToMeetingRoom)
        }
      }

      SocketConnectionEvent.Reconnecting -> {
        serialScope.launch {
          controllerContainer.loggerController.traceLog("RoomNodeController | Detected network issue. Reconnecting to the meeting room")
          // We shouldn't leave call if the local user was outside the meeting room before the network drop
          val localUserWasInRoom = roomJoined
          if (localUserWasInRoom) {
            controllerContainer.platformUtilsProvider.getVideoUtils().destroyAll()
            controllerContainer.sfuUtils.leaveCall()
          }
          controllerContainer.eventController.triggerEvent(DyteEventType.OnReconnectingToMeetingRoom)
        }
      }

      is SocketConnectionEvent.ReconnectionAttempt -> {
        controllerContainer.loggerController.traceLog("RoomNodeController | Attempt ${event.attemptNumber} to reconnect to meeting room")
      }

      is SocketConnectionEvent.ReconnectionAttemptFailed -> {
        controllerContainer.loggerController.traceLog("RoomNodeController | Attempt ${event.attemptNumber} failed to reconnect to meeting room")
      }

      SocketConnectionEvent.ReconnectionFailed -> {
        controllerContainer.loggerController.traceLog("RoomNodeController | Failed to reconnect to the meeting room")
        controllerContainer.eventController.triggerEvent(DyteEventType.OnMeetingRoomReconnectionFailed)
      }
    }
  }

  private suspend fun _joinRoom() {
    val routerCapabilitiesResponse =
      controllerContainer.socketController.sendMessage(GET_ROUTER_RTP_CAPABILITIES, null)
    routerCapabilities =
      (controllerContainer.socketMessageResponseParser.parseResponse(
        routerCapabilitiesResponse
      ).payload as RouterCapabilitiesModel)
    controllerContainer.loggerController.traceLog(
      "received and processed router capabilities"
    )

    val joinRoomPayload = JoinRoomPayloadRequestModel()
    joinRoomPayload.device = Device(
      osName = controllerContainer.platformUtilsProvider.getPlatformUtils().getOsName(),
      osVersionName = controllerContainer.platformUtilsProvider.getPlatformUtils()
        .getOsVersion()
    )
    joinRoomPayload.displayName = controllerContainer.selfController.getSelf().name
    joinRoomPayload.audioMuted = !controllerContainer.selfController.getSelf().audioEnabled
    joinRoomPayload.rtpCapabilities = routerCapabilities
    val payload = jsonParser.encodeToJsonElement(joinRoomPayload)
    val joinRoomResponse = controllerContainer.socketController.sendMessage(
      OutboundMeetingEventType.JOIN_ROOM,
      payload
    )

    joinRoom =
      controllerContainer.socketMessageResponseParser.parseResponse(
        joinRoomResponse
      ).payload as WebSocketJoinRoomModel
  }
}
