package io.dyte.core.controllers

import io.dyte.core.controllers.DyteEventType.OnStageCountUpdated
import io.dyte.core.controllers.DyteEventType.OnViewerCountUpdated
import io.dyte.core.controllers.LiveStreamState.ERRORED
import io.dyte.core.controllers.LiveStreamState.NONE
import io.dyte.core.controllers.LiveStreamState.STARTED
import io.dyte.core.controllers.LiveStreamState.STOPPED
import io.dyte.core.feat.DyteMeetingParticipant
import io.dyte.core.feat.ParticipantFlags
import io.dyte.core.models.DyteJoinedMeetingParticipant
import io.dyte.core.socket.socketservice.SocketService
import io.dyte.core.socket.socketservice.SocketServiceEventListener
import io.dyte.core.socket.socketservice.SocketServiceUtils.RoomEvent
import kotlinx.coroutines.launch
import socket.livestreaming.DenyStageAccessRequest
import socket.livestreaming.GetPeerInfoRequest
import socket.livestreaming.GetStagePeersResponse
import socket.livestreaming.GetStageRequestsResponse
import socket.livestreaming.GrantStageAccessRequest
import socket.livestreaming.LiveStreamingEvent
import socket.livestreaming.PeerInfoResponse
import socket.livestreaming.RoomPeerCountResponse
import socket.livestreaming.StageRequest
import socket.room.RoomPeersInfoResponse

internal class LiveStreamController(
  controllerContainer: IControllerContainer,
  private val socketService: SocketService
) : BaseController(controllerContainer), ILiveStreamController {

  private var _liveStreamUrl: String? = null

  private var _state: LiveStreamState = NONE

  private var _viewerCount: Int = 0
  private var _stagePeerCount: Int = 0

  private var _roomName: String = ""

  private var participants = arrayListOf<DyteMeetingParticipant>()
  private var requestedPeers = arrayListOf<LiveStreamStageRequestPeer>()
  private var stagePeers = arrayListOf<LiveStreamStagePeer>()

  override val stageRequestPeers: List<LiveStreamStageRequestPeer>
    get() = requestedPeers

  override val livestreamData: DyteLivestreamData
    get() = DyteLivestreamData(_liveStreamUrl, _roomName, _state, _viewerCount)

  private val socketEventListener = object : SocketServiceEventListener {
    override fun onEvent(event: Int, eventId: String?, payload: ByteArray?) {
      println("DyteMobileClient | LiveStreamController onEvent $event $eventId")
        when (event) {
          RoomEvent.STARTED.id -> {
            controllerContainer.eventController.triggerEvent(DyteEventType.OnLiveStreamStarting)
            println("DyteMobileClient | LiveStreamController onEvent started")
            _state = STARTED
            val decodedPayload = LiveStreamingEvent.ADAPTER.decode(requireNotNull(payload))
            _roomName = decodedPayload.name
            _liveStreamUrl = decodedPayload.playback_url
            println("DyteMobileClient | LiveStreamController onEvent roomName $_roomName")
            controllerContainer.eventController.triggerEvent(DyteEventType.OnLiveStreamStarted)
          }

          RoomEvent.STOPPED.id -> {
            controllerContainer.eventController.triggerEvent(DyteEventType.OnLiveStreamEnding)
            _state = STOPPED
            controllerContainer.eventController.triggerEvent(DyteEventType.OnLiveStreamEnded)
          }

          RoomEvent.ERRORED.id -> {
            _state = ERRORED
            controllerContainer.eventController.triggerEvent(DyteEventType.OnLiveStreamErrored)
          }

          RoomEvent.ROOM_PEER_COUNT.id -> {
            println("DyteMobileClient | LiveStreamController onEvent room peer count")
            payload?.let {
              val response = RoomPeerCountResponse.ADAPTER.decode(payload)
              println("DyteMobileClient | LiveStreamController onEvent room viewers ${response.count}")
              _viewerCount = response.count.toInt()
              controllerContainer.eventController.triggerEvent(OnViewerCountUpdated(_viewerCount))
            }
          }

          // when new peer joins stage
          RoomEvent.GET_STAGE_PEERS.id -> {
            println("DyteMobileClient | LiveStreamController onEvent stage peers update")
            payload?.let {
              val response = GetStagePeersResponse.ADAPTER.decode(payload)
              println("DyteMobileClient | LiveStreamController onEvent stage peer count ${response.stage_peers.size}")
              stagePeers.clear()
              response.stage_peers.forEach { peerId ->
                val peersToRemove = requestedPeers.filter { it.userId == peerId || it.peerId == peerId }
                requestedPeers.removeAll(peersToRemove.toSet())
                peersToRemove.forEach { peerOnStage ->
                  stagePeers.add(LiveStreamStagePeer(peerOnStage.displayName, peerOnStage.userId, peerOnStage.peerId))
                }
              }
              println("DyteMobileClient | LiveStreamController onEvent update in request peer ${requestedPeers.size}")
              _stagePeerCount = response.stage_peers.size
              controllerContainer.eventController.triggerEvent(OnStageCountUpdated(_viewerCount))
            }
          }

          RoomEvent.GRANT_STAGE_REQUEST.id -> {
            println("DyteMobileClient | LiveStreamController onEvent request stage access grated")
            // val payload = GrantStageAccessRequest.ADAPTER.decode(requireNotNull(payload))
            // println("DyteMobileClient | LiveStreamController onEvent granted ${payload.user_ids}")
            // REQUEST GRANTED
          }

          RoomEvent.UPDATE_PEER_INFO.id -> {
            println("DyteMobileClient | LiveStreamController onEvent update peer info")
            // val payload = UpdatePeerInfoRequest.ADAPTER.decode(requireNotNull(payload))
            // println("DyteMobileClient | LiveStreamController onEvent ${payload.display_name}")
            // REQUEST GRANTED
          }

          RoomEvent.REQUEST_STAGE_ACCESS.id -> {
            println("DyteMobileClient | LiveStreamController onEvent request stage access")
            payload?.let {
              val response = StageRequest.ADAPTER.decode(payload)
              println("DyteMobileClient | LiveStreamController onEvent granted ${response.display_name}")
              requestedPeers.add(LiveStreamStageRequestPeer(response.display_name, response.user_id, response.peer_id))
            }
          }

          RoomEvent.DENY_STAGE_REQUEST.id -> {
            println("DyteMobileClient | LiveStreamController onEvent request stage access denied")
            payload?.let {
              val response = DenyStageAccessRequest.ADAPTER.decode(payload)
              println("DyteMobileClient | LiveStreamController onEvent denied ${response.user_ids}")
              // REQUEST DENIED
            }

          }

          RoomEvent.GET_PEER_INFO.id -> {
            println("DyteMobileClient | LiveStreamController onEvent request stage access denied")
            // val payload = Peer.ADAPTER.decode(requireNotNull(payload))
            // println("DyteMobileClient | LiveStreamController onEvent denied ${payload.display_name} ${payload.stage_type}")
            // REQUEST DENIED
          }

          RoomEvent.JOIN_STAGE.id -> {
            println("DyteMobileClient | LiveStreamController onEvent join stage")
            /*val payload = Joi.ADAPTER.decode(requireNotNull(payload))
            println("DyteMobileClient | LiveStreamController onEvent denied ${payload.display_name} ${payload.stage_type}")*/
            // REQUEST DENIED
          }

          RoomEvent.LEAVE_STAGE.id -> {
            println("DyteMobileClient | LiveStreamController onEvent leave stage")
            /*val payload = Joi.ADAPTER.decode(requireNotNull(payload))
            println("DyteMobileClient | LiveStreamController onEvent denied ${payload.display_name} ${payload.stage_type}")*/
            // REQUEST DENIED
          }

          // when someone requests to be on stage
          RoomEvent.GET_STAGE_REQUESTS.id -> {
            println("DyteMobileClient | LiveStreamController onEvent stage peer update")
            payload?.let {
              val response = GetStageRequestsResponse.ADAPTER.decode(payload)
              println("DyteMobileClient | LiveStreamController onEvent there are ${response.stage_requests.size} stage request pending")
              requestedPeers.clear()
              response.stage_requests.forEach { requestPeer ->
                println("DyteMobileClient | LiveStreamController onEvent request-stage peer ${requestPeer.display_name}")
                val requestedPeer = LiveStreamStageRequestPeer(
                  requestPeer.display_name,
                  requestPeer.user_id,
                  requestPeer.peer_id
                )
                requestedPeers.add(requestedPeer)
              }
            } ?: run {
              // empty response means no more peers requesting to be on stage
              println("DyteMobileClient | LiveStreamController onEvent no requests")
              requestedPeers.clear()
            }

            controllerContainer.eventController.triggerEvent(
              DyteEventType.OnLiveStreamStageRequestsUpdated(
                requestedPeers
              )
            )

            /*serialScope.launch {
              val stagePeers = getStagePeers()
              controllerContainer.eventController.triggerEvent(
                DyteEventType.OnStageCountUpdated(
                  stagePeers.size
                )
              )
            }*/
          }

          RoomEvent.CANCEL_STAGE_REQUEST.id -> {
            println("DyteMobileClient | LiveStreamController onEvent cancel stage request")
          }
          
          else -> {
            println("DyteMobileClient | LiveStreamController onEvent cant process")
          }
        }
    }
  }

  override fun init() {
    socketService.subscribe(RoomEvent.STARTED.id, socketEventListener)
    socketService.subscribe(RoomEvent.STOPPED.id, socketEventListener)
    socketService.subscribe(RoomEvent.ERRORED.id, socketEventListener)
    socketService.subscribe(RoomEvent.ROOM_PEER_COUNT.id, socketEventListener)
    socketService.subscribe(RoomEvent.UPDATE_PEER_INFO.id, socketEventListener)
    socketService.subscribe(RoomEvent.GRANT_STAGE_REQUEST.id, socketEventListener)
    socketService.subscribe(RoomEvent.REQUEST_STAGE_ACCESS.id, socketEventListener)
    socketService.subscribe(RoomEvent.DENY_STAGE_REQUEST.id, socketEventListener)
    socketService.subscribe(RoomEvent.GET_STAGE_PEERS.id, socketEventListener)
    socketService.subscribe(RoomEvent.GET_PEER_INFO.id, socketEventListener)
    socketService.subscribe(RoomEvent.GET_STAGE_REQUESTS.id, socketEventListener)
    socketService.subscribe(RoomEvent.JOIN_STAGE.id, socketEventListener)
    socketService.subscribe(RoomEvent.LEAVE_STAGE.id, socketEventListener)
    socketService.subscribe(RoomEvent.CANCEL_STAGE_REQUEST.id, socketEventListener)
    loadLiveStreamUrl()
  }

  private fun loadLiveStreamUrl() {
    try {
      serialScope.launch {
        try {
          _liveStreamUrl = controllerContainer.apiClient.getLiveStreamUrl()
          _state = STARTED
          controllerContainer.eventController.triggerEvent(DyteEventType.OnLiveStreamStarted)
        } catch (e: Exception) {
          // ignore exception
        }
      }
    } catch (e: Exception) {
      _state = NONE
      e.printStackTrace()
    }
  }

  override suspend fun loadData() {
      socketService.send(RoomEvent.GET_STAGE_PEERS.id, null)
      socketService.send(RoomEvent.GET_STAGE_REQUESTS.id, null)
  }

  override suspend fun start() {
    controllerContainer.eventController.triggerEvent(DyteEventType.OnLiveStreamStarting)
    controllerContainer.apiClient.startLiveStream()
  }

  override suspend fun stop() {
    controllerContainer.eventController.triggerEvent(DyteEventType.OnLiveStreamEnding)
    controllerContainer.apiClient.stopLiveStream()
  }

  /*override suspend fun joinStage() {
    println("DyteMobileClient | LiveStreamController joinStage ")
    if (controllerContainer.webinarController.canJoinStage()) {
      println("DyteMobileClient | LiveStreamController joinStage request join stage")
      socketService.send(RoomEvent.REQUEST_STAGE_ACCESS.id, null)
    } else {
      println("DyteMobileClient | LiveStreamController joinStage direct")
      _joinRoom = true
      socketService.send(RoomEvent.JOIN_STAGE.id, null)
      controllerContainer.roomNodeController.joinRoom()
    }
  }

  override suspend fun leaveStage() {
    println("DyteMobileClient | LiveStreamController leaveStage ")
    _joinRoom = false
    socketService.send(RoomEvent.JOIN_STAGE.id, null)
  }*/

  override suspend fun acceptRequest(id: String) {
    socketService.send(
      RoomEvent.GRANT_STAGE_REQUEST.id,
      GrantStageAccessRequest(listOf(id)).encode()
    )
  }

  override suspend fun rejectRequest(id: String) {
    socketService.send(RoomEvent.DENY_STAGE_REQUEST.id, DenyStageAccessRequest(listOf(id)).encode())
  }

  override suspend fun getStagePeers(): List<LiveStreamStagePeer> {
    println("DyteMobileClient | LiveStreamController getStagePeers ${stagePeers.size}")
    return stagePeers
  }

  override suspend fun getStageRequests(): List<LiveStreamStageRequestPeer> {
    println("DyteMobileClient | LiveStreamController getStageRequests ")
    val response = socketService.requestResponse(RoomEvent.GET_STAGE_REQUESTS.id, null)
    val parsedResponse = GetStageRequestsResponse.ADAPTER.decode(requireNotNull(response))
    val stagePeers = arrayListOf<LiveStreamStageRequestPeer>()
    parsedResponse.stage_requests.forEach { requestPeer ->
      stagePeers.add(
        LiveStreamStageRequestPeer(
          requestPeer.display_name,
          requestPeer.user_id,
          requestPeer.peer_id
        )
      )
    }
    return stagePeers
  }

  private suspend fun loadLiveStreamPeers() {
    val response = socketService.requestResponse(RoomEvent.GET_ROOM_PEERS_INFO.id, null)
    response?.let {
      val parsedResponse = RoomPeersInfoResponse.ADAPTER.decode(requireNotNull(response))
      println("DyteMobileClient | LiveStreamController loadLiveStreamPeers we've ${parsedResponse.peers.size} peers")
      participants.clear()
      parsedResponse.peers.forEach {
        val peerInfoResponse = socketService.requestResponse(
          RoomEvent.GET_PEER_INFO.id,
          GetPeerInfoRequest(it.peer_id).encode()
        )
        val peerInfo = PeerInfoResponse.ADAPTER.decode(requireNotNull(peerInfoResponse))
        participants.add(
          DyteJoinedMeetingParticipant(
            it.peer_id,
            peerInfo.peer!!.user_id,
            it.display_name,
            it.display_picture_url,
            false,
            it.custom_participant_id,
            ParticipantFlags(false, false, false),
            it.preset_id ?: "",
            controllerContainer
          )
        )
      }
    }
  }
}

interface ILiveStreamController {
  val livestreamData: DyteLivestreamData
  val stageRequestPeers: List<LiveStreamStageRequestPeer>

  suspend fun start()
  suspend fun stop()

  /*suspend fun joinStage()
  suspend fun leaveStage()*/

  suspend fun getStagePeers(): List<LiveStreamStagePeer>

  suspend fun getStageRequests(): List<LiveStreamStageRequestPeer>

  suspend fun acceptRequest(id: String)

  suspend fun rejectRequest(id: String)

  suspend fun loadData()
}

enum class LiveStreamState {
  NONE,
  STARTING,
  STARTED,
  STOPPING,
  STOPPED,
  ERRORED;

  companion object {
    fun fromMap(stage: String): LiveStreamState {
      return when (stage) {
        "none" -> NONE
        "starting" -> STARTING
        "started" -> STARTED
        "stopping" -> STOPPING
        "stopped" -> STOPPED
        "errored" -> ERRORED
        else -> NONE
      }
    }
  }

  fun toMap(): Map<String, String> {
    val strState =
      when (this) {
        NONE -> "none"
        STARTING -> "stopping"
        STARTED -> "started"
        STOPPING -> "stopping"
        STOPPED -> "stopped"
        ERRORED -> "errored"
      }

    val map: HashMap<String, String> = HashMap()

    map["state"] = strState

    return map;
  }
}

data class DyteLivestreamData(
  val liveStreamUrl: String?,
  val roomName: String,
  val state: LiveStreamState,
  val viewerCount: Int,
) {
  companion object {
    fun fromMap(map: Map<String, Any?>): DyteLivestreamData {

      val lvsUrl = map["livestreamUrl"] as String
      val roomName = map["roomName"] as String
      val viewerCount = map["viewerCount"] as Int
      val state = LiveStreamState.fromMap(map["state"] as String)

      return DyteLivestreamData(lvsUrl, roomName, state, viewerCount)
    }
  }

  fun toMap(): Map<String, Any?> {
    val map = HashMap<String, Any?>()
    map["livestreamUrl"] = liveStreamUrl
    map["roomName"] = roomName
    map["viewerCount"] = viewerCount
    map["state"] = state.toMap()["state"]
    return map;
  }
}

data class LiveStreamStageRequestPeer(
  val displayName: String,
  val userId: String,
  val peerId: String
) {
  companion object {
    fun fromMap(map: Map<String, Any?>): LiveStreamStageRequestPeer {

      val displayName = map["displayName"] as String
      val userId = map["userId"] as String
      val peerId = map["peerId"] as String

      return LiveStreamStageRequestPeer(displayName, userId, peerId)
    }
  }

  fun toMap(): Map<String, Any?> {
    val map = HashMap<String, Any?>()
    map["displayName"] = displayName
    map["userId"] = userId
    map["peerId"] = peerId
    return map
  }
}

data class LiveStreamStagePeer(
  val displayName: String,
  val userId: String,
  val peerId: String
) {
  companion object {
    fun fromMap(map: Map<String, Any?>): LiveStreamStagePeer {

      val displayName = map["displayName"] as String
      val userId = map["userId"] as String
      val peerId = map["peerId"] as String

      return LiveStreamStagePeer(displayName, userId, peerId)
    }
  }

  fun toMap(): Map<String, Any?> {
    val map = HashMap<String, Any?>()
    map["displayName"] = displayName
    map["userId"] = userId
    map["peerId"] = peerId
    return map
  }
}