/*
 * Copyright (c) 2017 Henry Lin @zxcpoiu
 * 
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
package io.dyte.core.incallmanager

import android.annotation.SuppressLint
import android.content.Context
import android.hardware.Sensor
import android.hardware.SensorManager
import android.os.Build.VERSION
import android.os.PowerManager
import android.os.PowerManager.WakeLock
import android.util.Log
import io.dyte.core.incallmanager.apprtc.AppRTCProximitySensor
import io.dyte.core.incallmanager.apprtc.AppRTCProximitySensor.Companion.create
import java.lang.reflect.Method

class InCallProximityManager private constructor(
  context: Context,
  inCallManager: InCallManagerModule
) {
  private var mProximityLock: WakeLock? = null
  private var mPowerManagerRelease: Method? = null
  var isProximitySupported = false
    private set
  private var proximitySensor: AppRTCProximitySensor? = null
  @SuppressLint("InvalidWakeLockTag") private fun checkProximitySupport(context: Context) {
    val sensorManager = context.getSystemService(Context.SENSOR_SERVICE) as SensorManager
    if (sensorManager.getDefaultSensor(Sensor.TYPE_PROXIMITY) == null) {
      isProximitySupported = false
      return
    }
    val powerManager = context.getSystemService(Context.POWER_SERVICE) as PowerManager
    isProximitySupported = true

    // --- Check if PROXIMITY_SCREEN_OFF_WAKE_LOCK is implemented.
    try {
      val field = PowerManager::class.java.getDeclaredField("PROXIMITY_SCREEN_OFF_WAKE_LOCK")
      val proximityScreenOffWakeLock = field[null] as Int
      val proximitySupported = if (VERSION.SDK_INT < 17) {
        val method = powerManager.javaClass.getDeclaredMethod("getSupportedWakeLockFlags")
        val powerManagerSupportedFlags = method.invoke(powerManager) as Int
        powerManagerSupportedFlags and proximityScreenOffWakeLock != 0x0
      } else {
        // --- android 4.2+
        val method = powerManager.javaClass.getDeclaredMethod(
          "isWakeLockLevelSupported",
          Int::class.javaPrimitiveType
        )
        method.invoke(powerManager, proximityScreenOffWakeLock) as Boolean
      }
      if (proximitySupported) {
        mProximityLock = powerManager.newWakeLock(proximityScreenOffWakeLock, TAG)
        mProximityLock!!.setReferenceCounted(false)
      }
    } catch (e: Exception) {
      Log.d(TAG, "Failed to get proximity screen locker. exception: ", e)
    }
    if (mProximityLock != null) {
      Log.d(TAG, "use native screen locker...")
      try {
        mPowerManagerRelease =
          mProximityLock!!.javaClass.getDeclaredMethod("release", Int::class.javaPrimitiveType)
      } catch (e: Exception) {
        Log.d(TAG, "failed to get proximity screen locker: `release()`. exception: ", e)
      }
    } else {
      Log.d(TAG, "fallback to old school screen locker...")
    }
  }

  fun start(): Boolean {
    if (!isProximitySupported) {
      return false
    }
    proximitySensor!!.start()
    return true
  }

  fun stop() {
    proximitySensor!!.stop()
  }

  val isProximityWakeLockSupported: Boolean
    get() = mProximityLock != null
  val proximityIsNear: Boolean
    get() = if (isProximitySupported) proximitySensor!!.sensorReportsNearState() else false

  fun acquireProximityWakeLock() {
    if (!isProximityWakeLockSupported) {
      return
    }
    synchronized(mProximityLock!!) {
      if (!mProximityLock!!.isHeld) {
        Log.d(TAG, "acquireProximityWakeLock()")
        mProximityLock!!.acquire()
      }
    }
  }

  fun releaseProximityWakeLock(waitForNoProximity: Boolean) {
    if (!isProximityWakeLockSupported) {
      return
    }
    synchronized(mProximityLock!!) {
      if (mProximityLock!!.isHeld) {
        try {
          val flags = if (waitForNoProximity) PowerManager.RELEASE_FLAG_WAIT_FOR_NO_PROXIMITY else 0
          mPowerManagerRelease!!.invoke(mProximityLock, flags)
          Log.d(TAG, "releaseProximityWakeLock()")
        } catch (e: Exception) {
          Log.e(TAG, "failed to release proximity lock. e: ", e)
        }
      }
    }
  }

  companion object {
    private const val TAG = "InCallProximityManager"

    /** Construction  */
    fun create(context: Context, inCallManager: InCallManagerModule): InCallProximityManager {
      return InCallProximityManager(context, inCallManager)
    }
  }

  init {
    Log.d(TAG, "InCallProximityManager")
    checkProximitySupport(context)
    if (isProximitySupported) {
      proximitySensor = create(context) {
        inCallManager.onProximitySensorChangedState(
          proximitySensor!!.sensorReportsNearState()
        )
      }
    }
  }
}