package io.dyte.core.models

import io.dyte.core.controllers.IControllerContainer
import io.dyte.core.feat.DyteMeetingParticipant
import io.dyte.core.feat.ParticipantFlags
import io.dyte.core.platform.VideoView
import kotlinx.coroutines.launch

open class DyteJoinedMeetingParticipant
internal constructor(
  override val id: String,
  override val userId: String,
  override var name: String,
  override val picture: String?,
  override val isHost: Boolean,
  override val clientSpecificId: String?,
  override val flags: ParticipantFlags,
  override val presetName: String,
  controllerContainer: IControllerContainer,
) :
  DyteMeetingParticipant(
    id,
    userId,
    name,
    picture,
    isHost,
    clientSpecificId,
    flags,
    presetName,
    controllerContainer
  ) {

  companion object {
    fun get(participant: DyteMeetingParticipant): DyteJoinedMeetingParticipant {
      return DyteJoinedMeetingParticipant(
        participant.id,
        participant.userId,
        participant.name, // null in case of recorder
        participant.picture,
        participant.isHost,
        participant.clientSpecificId,
        participant.flags,
        participant.presetName,
        participant.controllerContainer
      )
    }
  }

  /**
   * Disables the audio of the participant if it is enabled
   *
   * Requires *localUser.permissions.host.canMuteAudio*
   */
  @Throws(Exception::class)
  open fun disableAudio() {
    val self = this
    serialScope.launch { controllerContainer.hostController.muteAudio(self) }
  }

  /**
   * Disables the video of the participant if it is enabled
   *
   * Requires *localUser.permissions.host.canMuteVideo*
   */
  @Throws(Exception::class)
  open fun disableVideo() {
    serialScope.launch {
      controllerContainer.hostController.muteVideo(this@DyteJoinedMeetingParticipant)
    }
  }

  /**
   * Pins the participant, so the participant is always in the active list
   *
   * Requires *localUser.permissions.host.canPinParticipant*
   */
  @Throws(Exception::class)
  open fun pin() {
    serialScope.launch {
      controllerContainer.hostController.pinParticipant(this@DyteJoinedMeetingParticipant)
    }
  }

  /**
   * Unpins the participant
   *
   * Requires *localUser.permissions.host.canPinParticipant*
   */
  @Throws(Exception::class)
  open fun unpin() {
    serialScope.launch { controllerContainer.hostController.unpinParticipant() }
  }

  val isPinned: Boolean
    get() = controllerContainer.participantController.pinnedParticipant?.id == this.id

  /**
   * Removes the participant from the meeting
   *
   * Requires *localUser.permissions.host.canKickParticipant*
   */
  @Throws(Exception::class)
  open fun kick() {
    serialScope.launch { controllerContainer.hostController.kick(id) }
  }

  open fun getVideoView(): VideoView? {
    return controllerContainer.participantController.getVideoView(this)
  }
}
