package io.dyte.core.host

import io.dyte.core.observability.DyteLogger
import io.dyte.core.socket.socketservice.SocketService
import io.dyte.core.socket.socketservice.SocketServiceUtils
import io.dyte.webrtc.MediaStreamTrackKind
import media.edge.GlobalPeerPinningRequest
import media.edge.HostMediaControlForAllPeerRequest
import media.edge.HostMediaControlForAllPeerResponse
import media.edge.HostMediaControlForPeerRequest
import media.edge.HostMediaControlForPeerResponse
import media.edge.KickAllPeersResponse
import media.edge.KickPeerRequest
import media.edge.KickPeerResponse

internal class HostSocketServiceHandler(
  private val socketService: SocketService,
) : HostSocketHandler {
  override suspend fun kickPeer(peerId: String): Boolean {
    return try {
      val req = KickPeerRequest(participant_id = peerId)

      val res =
        socketService.requestResponse(
          event = SocketServiceUtils.MediaEvent.KICK_PEER.id,
          payload = KickPeerRequest.ADAPTER.encode(req)
        )

      !(res == null || KickPeerResponse.ADAPTER.decode(res).status != "success")
    } catch (e: Exception) {
      DyteLogger.error("HostSocketService::kick_peer_failed::", e)
      false
    }
  }

  override suspend fun kickAll(): Boolean {
    return try {
      val res =
        socketService.requestResponse(
          event = SocketServiceUtils.MediaEvent.KICK_ALL.id,
          payload = null
        )

      !(res == null || KickAllPeersResponse.ADAPTER.decode(res).status != "success")
    } catch (e: Exception) {
      DyteLogger.error("HostSocketService::kick_all_failed::", e)
      false
    }
  }

  override suspend fun muteAllAudio(allowUnmute: Boolean?): Boolean {
    return hostControlForAll(MediaStreamTrackKind.Audio)
  }

  override suspend fun muteAllVideo(): Boolean {
    return hostControlForAll(MediaStreamTrackKind.Video)
  }

  override suspend fun muteAudio(peerId: String): Boolean {
    return hostControlForPeer(peerId, MediaStreamTrackKind.Audio)
  }

  override suspend fun muteVideo(peerId: String): Boolean {
    return hostControlForPeer(peerId, MediaStreamTrackKind.Video)
  }

  override suspend fun pinPeer(peerId: String): Boolean {
    return pinPeerInternal(peerId)
  }

  override suspend fun unpinPeer(): Boolean {
    return pinPeerInternal()
  }

  private suspend fun hostControlForAll(kind: MediaStreamTrackKind): Boolean {
    return try {
      val req =
        HostMediaControlForAllPeerRequest(
          audio = kind == MediaStreamTrackKind.Audio,
          screen_share = false,
          video = kind == MediaStreamTrackKind.Video
        )

      val res =
        socketService.requestResponse(
          event = SocketServiceUtils.MediaEvent.HOST_CONTROL_ALL_PEERS.id,
          payload = HostMediaControlForAllPeerRequest.ADAPTER.encode(req)
        )

      !(res == null || HostMediaControlForAllPeerResponse.ADAPTER.decode(res).status != "success")
    } catch (e: Exception) {
      DyteLogger.error("HostSocketService::host_control_for_all_failed::", e)
      false
    }
  }

  private suspend fun hostControlForPeer(peerId: String, kind: MediaStreamTrackKind): Boolean {
    return try {
      val req =
        HostMediaControlForPeerRequest(
          audio = kind == MediaStreamTrackKind.Audio,
          scree_share = false,
          video = kind == MediaStreamTrackKind.Video,
          participant_id = peerId
        )

      val res =
        socketService.requestResponse(
          event = SocketServiceUtils.MediaEvent.HOST_CONTROL_PEER.id,
          payload = HostMediaControlForPeerRequest.ADAPTER.encode(req)
        )

      !(res == null || HostMediaControlForPeerResponse.ADAPTER.decode(res).status != "success")
    } catch (e: Exception) {
      DyteLogger.error("HostSocketService::host_control_for_peer_failed::", e)
      false
    }
  }

  private suspend fun pinPeerInternal(peerId: String = ""): Boolean {
    return try {
      val req =
        GlobalPeerPinningRequest(
          participant_id = peerId,
        )

      socketService.send(
        event = SocketServiceUtils.MediaEvent.GLOBAL_PIN_PEER.id,
        payload = GlobalPeerPinningRequest.ADAPTER.encode(req)
      )

      true
    } catch (e: Exception) {
      DyteLogger.error("HostSocketService::pin_peer_failed::", e)
      false
    }
  }
}
