package io.dyte.core.controllers

import io.dyte.core.feat.DyteMeetingParticipant
import io.dyte.core.host.HostSocketHandler
import io.dyte.core.observability.DyteLogger

internal class HostController(
  controllerContainer: IControllerContainer,
  private val hostSocketHandler: HostSocketHandler
) : IHostController, BaseController(controllerContainer) {
  override suspend fun kick(id: String) {
    DyteLogger.info("DyteParticipants.kickPeer")
    if (controllerContainer.presetController.permissions.host.canKickParticipant.not()) {
      DyteLogger.error("DyteParticipant::unauthorized_kick")
      throw UnsupportedOperationException("not allowed to kick participant")
    }

    val res = hostSocketHandler.kickPeer(id)

    if (!res) {
      DyteLogger.error("DyteParticipant::kick_peer_failed")
      throw Error("Failed to kick participant")
    }
  }

  // TODO : is there any differance between kick single and kick all ?
  override suspend fun kickAll() {
    DyteLogger.info("DyteParticipants.kickAll")
    if (controllerContainer.presetController.permissions.host.canKickParticipant.not()) {
      DyteLogger.error("DyteParticipant::unauthorized_kick_all")
      throw UnsupportedOperationException("not allowed to kick all")
    }

    val res = hostSocketHandler.kickAll()

    if (!res) {
      DyteLogger.error("DyteParticipant::kick_all_failed")
      throw Error("Failed to kick all participants")
    }
  }

  override suspend fun muteAllAudio(allowUnmute: Boolean) {
    DyteLogger.info("DyteParticipants.disableAllAudio")
    if (controllerContainer.presetController.permissions.host.canMuteAudio.not()) {
      DyteLogger.error("DyteParticipant::unauthorized_disable_all_audio")
      throw UnsupportedOperationException("not allowed to mute all audio")
    }

    val res = hostSocketHandler.muteAllAudio(allowUnmute)

    if (!res) {
      DyteLogger.error("DyteParticipant::mute_all_audio_failed")
      throw Error("Failed to mute all audio")
    }
  }

  override suspend fun muteAllVideo() {
    DyteLogger.info("DyteParticipants.disableAllVideo")
    if (controllerContainer.presetController.permissions.host.canMuteVideo.not()) {
      DyteLogger.error("DyteParticipants::unauthorized_disable_all_video")
      throw UnsupportedOperationException("not allowed to mute all video")
    }

    val res = hostSocketHandler.muteAllVideo()

    if (!res) {
      DyteLogger.error("DyteParticipant::mute_all_video_failed")
      throw Error("Failed to mute all video")
    }
  }

  override suspend fun muteAudio(dyteMeetingParticipant: DyteMeetingParticipant) {
    DyteLogger.info("DyteParticipants.disablePeerAudio")
    if (controllerContainer.presetController.permissions.host.canMuteAudio.not()) {
      DyteLogger.error("DyteParticipant::unauthorized_disable_audio")
      throw UnsupportedOperationException("cant disable participant audio")
    }

    val res = hostSocketHandler.muteAudio(dyteMeetingParticipant.id)

    if (!res) {
      DyteLogger.error("DyteParticipant::mute_audio_failed")
      throw Error("Failed to mute audio")
    }
  }

  override suspend fun muteVideo(dyteMeetingParticipant: DyteMeetingParticipant) {
    DyteLogger.info("DyteParticipants.disablePeerVideo")
    if (controllerContainer.presetController.permissions.host.canMuteVideo.not()) {
      DyteLogger.error("DyteParticipant::unauthorized_disable_video")
      throw UnsupportedOperationException("cant produce video")
    }

    val res = hostSocketHandler.muteVideo(dyteMeetingParticipant.id)

    if (!res) {
      DyteLogger.error("DyteParticipant::mute_video_failed")
      throw Error("Failed to mute video")
    }
  }

  override suspend fun pinParticipant(dyteMeetingParticipant: DyteMeetingParticipant) {
    DyteLogger.info("DyteParticipants.pinPeer")
    if (controllerContainer.presetController.permissions.host.canPinParticipant.not()) {
      DyteLogger.error("DyteParticipant::unauthorized_pin_peer")
      throw UnsupportedOperationException("not allowed to pin participant")
    }

    val res = hostSocketHandler.pinPeer(dyteMeetingParticipant.id)

    if (!res) {
      DyteLogger.error("DyteParticipant::pin_peer_failed")
      throw Error("Failed to pin participant")
    }
  }

  override suspend fun unpinParticipant() {
    DyteLogger.info("DyteParticipants.unpinPeer")
    if (controllerContainer.presetController.permissions.host.canPinParticipant.not()) {
      DyteLogger.error("DyteParticipant::unauthorized_unpin_peer")
      throw UnsupportedOperationException("not allowed to un-pin participant")
    }

    val res = hostSocketHandler.unpinPeer()

    if (!res) {
      DyteLogger.error("DyteParticipant::unpin_peer_failed")
      throw Error("Failed to unpin participant")
    }
  }
}

interface IHostController {
  suspend fun muteAllAudio(allowUnmute: Boolean)

  suspend fun muteAudio(dyteMeetingParticipant: DyteMeetingParticipant)

  suspend fun muteAllVideo()

  suspend fun muteVideo(dyteMeetingParticipant: DyteMeetingParticipant)

  suspend fun kickAll()

  suspend fun kick(id: String)

  suspend fun pinParticipant(dyteMeetingParticipant: DyteMeetingParticipant)

  suspend fun unpinParticipant()
}
