package io.dyte.core.controllers

import io.dyte.core.events.JoinRoomEventPublisher
import io.dyte.core.hive.HiveNodeSocketHandler
import io.dyte.core.media.IDyteHive
import io.dyte.core.models.DyteMeetingType
import io.dyte.core.network.models.IceServerData
import io.dyte.core.observability.DyteLogger
import io.dyte.core.socket.events.payloadmodel.inbound.MeetingPeerFlags
import io.dyte.core.socket.events.payloadmodel.inbound.MeetingPeerMetadata
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketMeetingPeerUser
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketPeerLeftModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketPeerMuteModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketPeerPinnedModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketSelectedPeersModel
import io.dyte.core.socket.events.payloadmodel.outbound.RoomState
import io.dyte.core.socket.events.payloadmodel.outbound.WebSocketRoomStateModel
import io.dyte.core.socket.socketservice.SocketServiceEventListener
import io.dyte.media.hive.HiveConsumerOptions
import io.dyte.media.hive.HiveTransport
import io.dyte.webrtc.MediaStreamTrackKind
import io.dyte.webrtc.RtpTransceiverDirection
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.ExperimentalCoroutinesApi
import kotlinx.coroutines.launch
import kotlinx.coroutines.newSingleThreadContext
import kotlinx.coroutines.runBlocking
import kotlinx.coroutines.sync.Mutex
import kotlinx.coroutines.sync.withLock
import media.ProducerKind
import media.ProducerState
import media.edge.*
import socket.room.BroadcastMessage

internal class HiveController(
  controllerContainer: IControllerContainer,
  private val socketHandler: HiveNodeSocketHandler,
  private val joinRoomEventPublisher: JoinRoomEventPublisher? = null
) : IRoomNodeController, BaseController(controllerContainer) {

  private val logger = DyteLogger

  private lateinit var iceServers: List<IceServerData>

  private val hiveUtils = controllerContainer.sfuUtils as IDyteHive

  // FIXME(itzmanish): hardcoding room's maxPreferredStreams to 6 for now.
  // Though it should not be use after completely shifting to hive node
  private val maxPreferredStreams = 6

  private var activatedProducingPeerIds = mutableSetOf<String>()

  private var roomJoined: Boolean = false

  private var peerProducers = mutableListOf<PeerProducerMeta>()

  private val peerId: String = controllerContainer.selfController.getSelf().id

  private var roomJoiningInProgress = false

  private val consumerMutex = Mutex()

  @OptIn(ExperimentalCoroutinesApi::class)
  private val coroutineScope = CoroutineScope(newSingleThreadContext("HiveController"))

  init {
    coroutineScope.launch { handleHiveInternalEvents() }

    handleSocketEvents()
  }

  override fun init() {
    // TODO("Socket service on state events")
  }

  override suspend fun joinRoom() {
    hiveUtils.stopAllTransports()

    this.roomJoiningInProgress = true

    DyteLogger.info("HiveController: Join room started")
    controllerContainer.eventController.triggerEvent(DyteEventType.OnMeetingRoomJoinStarted)

    try {
      val peerJoinRequest =
        PeerJoinRequest(
          display_name = controllerContainer.selfController.getSelf().name,
          prejoined = false,
          room_uuid = "",
        )

      socketHandler.joinRoom(peerJoinRequest)

      hiveUtils.setupTransports()

      completeJoinRoom()
    } catch (e: Error) {
      logger.error("HiveController: Error on sending join room request: $e")
    }
  }

  private suspend fun completeJoinRoom() {
    try {
      val req = PeerJoinCompleteRequest()

      val peerJoinResponse = socketHandler.completeJoinRoom()!!
      // var pluginResponseRaw: ByteArray? = ByteArray(0)
      //      runBlocking {
      //        peerJoinResponseRaw = socketService.requestResponse(
      //          event = SocketServiceUtils.MediaEvent.JOIN_ROOM.id,
      //          payload = PeerJoinRequest.ADAPTER.encode(peerJoinRequest)
      //        )
      //
      ////                pluginResponseRaw = socketService.requestResponse(
      ////                    event = SocketServiceEvents.PluginEvent.GET_PLUGINS.id,
      ////                    payload = null
      ////                )
      //      }
      //
      //      val peerJoinResponse = PeerJoinResponse.ADAPTER.decode(peerJoinResponseRaw!!)
      // val pluginResponse = EnablePluginsResponse.ADAPTER.decode(pluginResponseRaw!!)
      // TODO: update proto files
      // val is_recorder = peerJoinResponse.room_state.recorder_participant_id == it.peer_id
      // TODO : fix MeetingPeerMetadata
      val oldPeers =
        peerJoinResponse.participants.map {
          WebSocketMeetingPeerUser(
            id = it.peer_id,
            isHost = false,
            name = it.display_name,
            flags = MeetingPeerFlags(recordere = false),
            userId = it.user_id,
            audioMuted = false,
            participantMetadata = MeetingPeerMetadata("", "")
          )
        }
      //            val oldPlugins = pluginResponse.plugins.map {
      //                RoomStatePlugin(
      //                    id= it.plugin_id,
      //                    enabledBy = it.enabled_by,
      //                    staggered = false,
      //                    stores = listOf(),
      //                    aclAccessGrantedUsers = listOf(),
      //                    aclAccessRemovedUsers = listOf(),
      //                    aclType = ""
      //                )
      //            }

      if (peerJoinResponse.room_state?.is_recording == true) {
        controllerContainer.recordingController.onRecordingPeerJoined()
        // peerJoinResponse.room_state.recorder_participant_id
      }

      val oldRoomState =
        WebSocketRoomStateModel(
          roomState =
            RoomState(
              displayTitle = peerJoinResponse.room_state?.display_title,
              roomUUID = peerJoinResponse.room_state?.room_uuid,
              roomName = peerJoinResponse.room_state?.room_name,
              peers = oldPeers,
              // plugins = oldPlugins
            )
        )

      // check for remote peers joined before local user
      controllerContainer.participantController.handleRoomState(oldRoomState)

      // Pick out active plugins, if any
      // controllerContainer.pluginsController.handleRoomState(oldRoomState)
      // Notify other components who are listening to join-room event
      joinRoomEventPublisher?.publish()

      // TODO: Set Meeting Started Time
      // controllerContainer.metaController.setMeetingStartedTimestamp()

      // TODO: Handle Waitlisted State
      // controllerContainer.selfController.getSelf()._waitListStatus = WAITING
      //        controllerContainer.eventController.triggerEvent(
      //          DyteEventType.OnSelfWaitListStatusUpdate(
      //            WAITING
      //          )
      //        )

      this.roomJoiningInProgress = false
      this.roomJoined = true

      val audioEnableByDeveloper = controllerContainer.metaController.isAudioEnabled()

      if (controllerContainer.presetController.canPublishAudio() && audioEnableByDeveloper) {
        hiveUtils.shareMic(
          controllerContainer.platformUtilsProvider.getMediaUtils().createAudioTrack()
        )

        if (controllerContainer.selfController.getSelf()._audioEnabled) {
          controllerContainer.selfController.getSelf()._audioEnabled = true
          controllerContainer.eventController.triggerEvent(DyteEventType.OnSelfAudioUpdate)
        } else {
          hiveUtils.pauseMic()
        }
      }

      val videoEnableByDeveloper = controllerContainer.metaController.isVideoEnabled()

      if (
        controllerContainer.presetController.canPublishVideo() &&
          videoEnableByDeveloper &&
          controllerContainer.selfController.getSelf()._videoEnabled
      ) {
        hiveUtils.shareCam(
          controllerContainer.platformUtilsProvider.getMediaUtils().getVideoTrack()
        )
      }

      val selfTracks = hiveUtils.getSelfTrack()
      val selfParticipant = controllerContainer.selfController.getSelf()
      // selfParticipant.audioTrack = selfTracks.first
      selfParticipant._videoTrack = selfTracks.second

      //      controllerContainer.participantController.onSelfJoined()

      controllerContainer.participantController.onSelectedPeers(
        WebSocketSelectedPeersModel(
          peerIds = peerJoinResponse.selected_peers?.audio_peers,
          compulsoryPeers = peerJoinResponse.selected_peers?.compulsory_peers
        )
      )

      controllerContainer.selfController.onEnteredInRoom()

      if (controllerContainer.metaController.getMeetingType() == DyteMeetingType.LIVESTREAM) {
        controllerContainer.liveStreamController.loadData()
      }

      controllerContainer.eventController.triggerEvent(
        DyteEventType.OnMeetingRoomJoined(controllerContainer.selfController.getSelf())
      )

      peerJoinResponse.participants.forEach { participant ->
        participant.producer_states.forEach { state ->
          val producerMeta =
            PeerProducerMeta(
              producerId = state.producer_id,
              kind = if (state.kind == ProducerKind.AUDIO) "audio" else "video",
              paused = state.pause,
              screenShare = state.screen_share,
              peerId = participant.peer_id
            )

          createConsumer(producerMeta)
        }
      }

      controllerContainer.participantController.refreshGridParticipantsHive()

      if (
        controllerContainer.permissionController.isPermissionGrated(PermissionType.CAMERA).not()
      ) {
        DyteLogger.info("HiveController: Joined without camera permission")
        controllerContainer.eventController.triggerEvent(
          DyteEventType.OnMeetingRoomJoinedWithoutCameraPermission
        )
      }
      if (
        controllerContainer.permissionController.isPermissionGrated(PermissionType.MICROPHONE).not()
      ) {
        DyteLogger.info("HiveController: Joined without microphone permission")
        controllerContainer.eventController.triggerEvent(
          DyteEventType.OnMeetingRoomJoinedWithoutMicPermission
        )
      }

      controllerContainer.pollsController.loadPolls()
      controllerContainer.chatController.loadChatMessages()
      //
      //            DyteLogger.info("can produce audio ?
      // ${controllerContainer.presetController.canPublishAudio()} } video ?
      // ${controllerContainer.presetController.canPublishVideo()}")
      //
      //            if (controllerContainer.presetController.permissions.hostPermissions.canPresent
      // && (controllerContainer.presetController.canPublishAudio() ||
      // controllerContainer.presetController.canPublishVideo())) {
      //                connectProducerTransport()
      //            }
      //

      logger.info("HiveController: Join room completed")
    } catch (e: Exception) {
      DyteLogger.error("HiveController: join room failed", e)
      controllerContainer.eventController.triggerEvent(DyteEventType.OnMeetingRoomJoinFailed(e))
    }
  }

  private fun createConsumer(producerMeta: PeerProducerMeta) {
    peerProducers.add(producerMeta)

    if (
      producerMeta.kind == "audio" ||
        producerMeta.screenShare ||
        controllerContainer.participantController.currentActiveGridParticipantsHive.contains(
          producerMeta.peerId
        )
    ) {
      logger.info("HiveDebug: handling new consumer")
      coroutineScope.launch {
        consumerMutex.withLock {
          hiveUtils.handleNewConsumer(
            HiveConsumerOptions(
              id = null,
              paused = producerMeta.paused,
              producerId = producerMeta.producerId,
              kind =
                if (producerMeta.kind == "audio") MediaStreamTrackKind.Audio
                else MediaStreamTrackKind.Video,
              producingPeerId = producerMeta.peerId,
              appData = mutableMapOf("screenShare" to producerMeta.screenShare)
            )
          )
        }
      }
    }
  }

  private fun handleSocketEvents() {
    val onPeerJoinedBroadcast =
      object : SocketServiceEventListener {
        override fun onEvent(event: Int, eventId: String?, payload: ByteArray?) {
          logger.info("HiveDebug: Peer Joined Broadcast")
          if (!roomJoined) return

          try {
            val participant = PeerJoinBroadcastResponse.ADAPTER.decode(payload!!).participant
            logger.info("HiveController: Got peer joined with Id: ${participant?.peer_id}")

            if (peerId == participant?.peer_id) return

            // TODO : fix MeetingPeerMetadata
            val dyteParticipant =
              WebSocketMeetingPeerUser(
                id = participant?.peer_id,
                isHost = false,
                name = participant?.display_name,
                flags = MeetingPeerFlags(),
                userId = participant?.user_id, // Setting peer id as userid for now
                participantMetadata = MeetingPeerMetadata("", "")
                //                        audioMuted = true,
                // videoEnabled
              )

            controllerContainer.participantController.onPeerJoined(dyteParticipant)

            serialScope.launch {
              controllerContainer.participantController.refreshGridParticipantsHive()
            }
          } catch (e: Error) {
            logger.info("HiveController: Error in peer joined broadcast")
          }
        }
      }

    val onPeerProducerCreateBroadcast =
      object : SocketServiceEventListener {
        override fun onEvent(event: Int, eventId: String?, payload: ByteArray?) {
          if (!roomJoined) return

          try {
            val response = PeerProducerCreateBroadcastResponse.ADAPTER.decode(payload!!)

            if (response.participant_id == peerId) return

            val producerState = response.producer_state

            logger.info(
              "HiveController: Producer created broadcast: " +
                response.participant_id +
                "Producer id: ${producerState?.producer_id}"
            )

            val participant =
              controllerContainer.participantController.meetingRoomParticipants.joined.find {
                it.id == response.participant_id
              }

            if (participant == null) {
              logger.info("Peer not found for producer ${producerState?.producer_id} ")
            }

            val producerMeta =
              PeerProducerMeta(
                producerId = producerState!!.producer_id,
                kind = if (producerState.kind == ProducerKind.AUDIO) "audio" else "video",
                paused = producerState.pause,
                screenShare = producerState.screen_share,
                peerId = response.participant_id
              )

            logger.info("HiveDebug: Producer meta ${producerMeta.kind}")

            createConsumer(producerMeta)
          } catch (e: Error) {
            logger.info("HiveController: Error in peer producer create broadcast")
          }
        }
      }

    val onPeerProducerToggleBroadcast =
      object : SocketServiceEventListener {
        override fun onEvent(event: Int, eventId: String?, payload: ByteArray?) {
          if (!roomJoined) return

          try {
            val response = PeerProducerToggleBroadcastResponse.ADAPTER.decode(payload!!)

            if (response.participant_id == peerId) return

            val producerState = response.producer_state

            val convertedKind: MediaStreamTrackKind =
              if (producerState?.kind == ProducerKind.AUDIO) MediaStreamTrackKind.Audio
              else MediaStreamTrackKind.Video

            logger.info(
              "HiveController: Producer toggle broadcast: " +
                "${response.participant_id} producer id: " +
                "${producerState?.producer_id}"
            )
            val participant =
              controllerContainer.participantController.meetingRoomParticipants.joined.find {
                it.id == response.participant_id
              }
                ?: return

            if (producerState!!.kind == ProducerKind.AUDIO) {
              if (producerState.pause) {
                controllerContainer.participantController.onPeerAudioMuted(
                  WebSocketPeerMuteModel(participant.id)
                )
              } else {
                controllerContainer.participantController.onPeerAudioUnmuted(
                  WebSocketPeerMuteModel(participant.id)
                )
              }
            }

            val producer = peerProducers.find { it.peerId == response.participant_id }
            if (producer != null) {
              producer.paused = producerState.pause
            }

            val consumersForThisProducer =
              hiveUtils.getConsumers().values.filter {
                it.getProducerId() == producerState.producer_id && it.getKind() == convertedKind
              }

            consumersForThisProducer.forEach {
              if (it.getPaused() != producerState.pause) {
                logger.info("HiveController: Consumer state is mismatched for $")

                CoroutineScope(Dispatchers.Default).launch {
                  if (producerState.pause) {
                    it.pause()
                    hiveUtils.handlePauseConsumer(it.getId())
                  } else {
                    it.resume()
                    hiveUtils.handleResumeConsumer(it.getId())
                  }
                }
              }
            }
          } catch (e: Error) {
            logger.info("HiveController: Error in producer toggle broadcast handler")
          }
        }
      }

    val onSelectedPeer =
      object : SocketServiceEventListener {
        override fun onEvent(event: Int, eventId: String?, payload: ByteArray?) {
          if (!roomJoined) return

          try {
            val response = SelectedPeersResponse.ADAPTER.decode(payload!!)
            logger.info(
              "HiveController: Selected peers: ${response.audio_peers} " +
                "${response.compulsory_peers}"
            )

            controllerContainer.participantController.onSelectedPeers(
              WebSocketSelectedPeersModel(
                peerIds = response.audio_peers,
                compulsoryPeers = response.compulsory_peers
              )
            )
          } catch (e: Error) {
            logger.info("HiveController: Error in selected peer $e")
          }
        }
      }

    val onSelectedPeerDiff =
      object : SocketServiceEventListener {
        override fun onEvent(event: Int, eventId: String?, payload: ByteArray?) {
          if (!roomJoined) return

          try {
            val entries = SelectedPeersDiffResponse.ADAPTER.decode(payload!!).entries

            val peerIdsWithPriorities =
              entries.map {
                PeerWithPriority(
                  peerId = it.peer_id,
                  priority = it.priority,
                )
              }

            controllerContainer.participantController.onSelectedPeersDiff(peerIdsWithPriorities)
          } catch (e: Error) {
            logger.info("HiveController: Error in selected peer diff $e")
          }
        }
      }

    val onPeerLeaveBroadcast =
      object : SocketServiceEventListener {
        override fun onEvent(event: Int, eventId: String?, payload: ByteArray?) {
          if (!roomJoined) return

          try {
            val participantId = PeerLeaveBroadcastResponse.ADAPTER.decode(payload!!).participant_id

            if (participantId == peerId) return

            logger.info("HiveController: Peer left broadcast: $participantId")

            runBlocking { hiveUtils.cleanupConsumers(participantId) }

            controllerContainer.participantController.onPeerLeft(
              WebSocketPeerLeftModel(id = participantId)
            )
          } catch (e: Error) {
            logger.info("HiveController: Error in peer left broadcast $e")
          }
        }
      }

    val onPeerProducerCloseBroadcast =
      object : SocketServiceEventListener {
        override fun onEvent(event: Int, eventId: String?, payload: ByteArray?) {
          if (!roomJoined) return

          try {
            val response = PeerProducerCloseBroadcastResponse.ADAPTER.decode(payload!!)

            if (response.participant_id == peerId) return

            val producerState = response.producer_state

            logger.info("HiveController: Producer closed broadcast ${response.participant_id}")

            peerProducers =
              peerProducers.filter { it.producerId != producerState!!.producer_id }.toMutableList()

            val producerId = producerState!!.producer_id

            val consumerId = hiveUtils.producerIdToConsumerIdMap[producerId]

            if (consumerId == null) {
              logger.info("HiveController: No consumer found for producer: $producerId")
              // return
            }

            logger.info("HiveController: Closing consumer $consumerId for $producerId")

            runBlocking {
              consumerId?.let { hiveUtils.closeConsumer(it, null) }
              // hiveUtils.closeConsumer(consumerId, null)
            }

            if (producerState.screen_share) {
              val participant =
                controllerContainer.participantController.meetingRoomParticipants.joined.find {
                  it.id == response.participant_id
                }

              participant?.let {
                val screenShareParticipant =
                  controllerContainer.participantController.meetingRoomParticipants.screenshares
                    .find { ss -> ss.id == response.participant_id }

                screenShareParticipant?.let {
                  controllerContainer.participantController.onPeerScreenSharedEnded(
                    screenShareParticipant
                  )
                }
              }
            }

            logger.info("HiveController: Closed consumer $consumerId")

            hiveUtils.producerIdToConsumerIdMap.remove(producerId)
            // todo hive

            //          CoroutineScope(Dispatchers.Default).async {
            //            hiveUtils.closeConsumer(consumerId)
            //          }
          } catch (e: Error) {
            logger.info("HiveController: Error on Producer close broadcast $e")
          }
        }
      }

    val onGlobalPeerPinBroadcast =
      object : SocketServiceEventListener {
        override fun onEvent(event: Int, eventId: String?, payload: ByteArray?) {
          if (!roomJoined) return

          try {
            if (payload != null) {
              val peerId = GlobalPeerPinningBroadcastResponse.ADAPTER.decode(payload).participant_id
              logger.info("HiveController: Pinning peerId: $peerId")

              controllerContainer.participantController.onPeerPinned(
                WebSocketPeerPinnedModel(peerId = peerId)
              )
            } else {
              controllerContainer.participantController.onPeerUnpinned()
            }
          } catch (e: Error) {
            logger.info("HiveController: Error on global peer pin broadcast $e")
          }
        }
      }

    val onRecordingStartedBroadcast =
      object : SocketServiceEventListener {
        override fun onEvent(event: Int, eventId: String?, payload: ByteArray?) {
          if (!roomJoined) return

          try {
            var peerId: String? = null

            if (payload != null) {
              val data = RecordingStartedBroadcastResponse.ADAPTER.decode(payload)
              peerId = data.participant_id
            }

            controllerContainer.recordingController.onRecordingPeerJoined()
          } catch (e: Error) {
            logger.error("HiveController: Error in recording started broadcast $e")
          }
        }
      }

    val onRecordingStoppedBroadcast =
      object : SocketServiceEventListener {
        override fun onEvent(event: Int, eventId: String?, payload: ByteArray?) {
          if (!roomJoined) return

          try {
            var peerId: String? = null

            if (payload != null) {
              val data = RecordingStoppedBroadcastResponse.ADAPTER.decode(payload)
              peerId = data.participant_id
            }

            controllerContainer.recordingController.onRecordingPeerLeft()
          } catch (e: Error) {
            logger.error("HiveController: Error in recording stopped broadcast $e")
          }
        }
      }

    val onBroadcastMessage =
      object : SocketServiceEventListener {
        override fun onEvent(event: Int, eventId: String?, payload: ByteArray?) {
          try {
            val res = BroadcastMessage.ADAPTER.decode(payload!!)
            controllerContainer.eventController.triggerEvent(
              DyteEventType.OnRoomMessage(res.payload.utf8())
            )
          } catch (e: Exception) {
            logger.error("HiveController: Error in broadcast message $e")
          }
        }
      }

    socketHandler.subscribeToBroadcasts(
      onPeerJoinedBroadcast = onPeerJoinedBroadcast,
      onPeerProducerCreateBroadcast = onPeerProducerCreateBroadcast,
      onPeerProducerToggleBroadcast = onPeerProducerToggleBroadcast,
      onSelectedPeer = onSelectedPeer,
      onSelectedPeerDiff = onSelectedPeerDiff,
      onPeerLeaveBroadcast = onPeerLeaveBroadcast,
      onPeerProducerCloseBroadcast = onPeerProducerCloseBroadcast,
      onGlobalPeerPinBroadcast = onGlobalPeerPinBroadcast,
      onRecordingStartedBroadcast = onRecordingStartedBroadcast,
      onRecordingStoppedBroadcast = onRecordingStoppedBroadcast,
      onBroadcastMessage = onBroadcastMessage
    )
  }

  override suspend fun closeConsumers(peers: List<String>) {
    val consumerIds =
      hiveUtils
        .getConsumers()
        .filter { peers.contains(it.value.getPeerId()) }
        .map { itt -> itt.value.getId() }

    peers.forEach { activatedProducingPeerIds.remove(it) }

    return hiveUtils.closeConsumers(consumerIds)
  }

  override suspend fun consumePeer(producingPeerId: String, force: Boolean?) {
    if (!activatedProducingPeerIds.contains(producingPeerId) || force!!) {
      try {
        hiveUtils.consumePeer(producingPeerId)
        this.activatedProducingPeerIds.add(producingPeerId)
      } catch (e: Error) {
        logger.info("HiveController: consumePeer failed $e")
      }
    } else {
      logger.info("HiveController: Not creating consumer for peerId = $producingPeerId")
    }
  }

  override fun leaveRoom() {
    roomJoined = false
  }

  suspend fun broadcastMessage(broadcastMessage: BroadcastMessage) =
    socketHandler.broadcastMessage(broadcastMessage)

  suspend fun activatePeer(peerId: String, force: Boolean? = false) {
    logger.info("HiveDebug: in activate peer")
    val producers = peerProducers.filter { it.peerId == peerId }
    if (producers.isEmpty()) {
      logger.info("HiveUtils: activatePeer $peerId: Not an active peer")

      return
    }

    if (force == true) {
      consumerMutex.withLock { this.hiveUtils.consumePeer(peerId) }
      return
    }

    val consumersToResume = arrayListOf<String>()

    val allConsumersExist =
      producers.all { producer ->
        val consumerId =
          this.hiveUtils.producerIdToConsumerIdMap.get(producer.producerId) ?: return@all false
        val consumer = this.hiveUtils.getConsumers()[consumerId]

        if (consumer != null && !producer.paused && consumer.getPaused()) {
          consumersToResume.add(consumerId)
        }

        return@all true
      }

    if (!allConsumersExist) {
      consumerMutex.withLock { this.hiveUtils.consumePeer(peerId) }
      return
    }

    consumersToResume.forEach { this.hiveUtils.handleResumeConsumer(it) }
  }

  suspend fun deactivatePeer(peerId: String) {
    val producers = peerProducers.filter { it.peerId == peerId }
    logger.info("HiveUtils: deactivatePeer $peerId")

    producers.forEach {
      if (it.kind == "audio" || (it.kind == "video" && it.screenShare)) {
        logger.info("HiveUtils: We don't want to pause audio or screenshare")

        return
      }

      val consumerId = this.hiveUtils.producerIdToConsumerIdMap[it.producerId]

      if (consumerId == null) {
        logger.info("HiveUtils: Consumer not found in deactivate producers ${it.producerId}")

        return
      }

      this.hiveUtils.closeConsumer(consumerId)
      this.activatedProducingPeerIds.remove(peerId)
    }
  }

  override suspend fun reconnectTransport(transport: HiveTransport) {
    // TODO: Socket is connected or not
    if (this.roomJoiningInProgress) return

    val iceServerData = controllerContainer.apiClient.getICEServers().iceServers

    transport.close()

    if (transport.getDirection() == RtpTransceiverDirection.SendOnly) {
      this.hiveUtils.createWebRtcTransportProd(null, iceServerData)
      logger.info("HiveUtils: Transport connected now need to start creating producers")

      this.hiveUtils.resetVideoProducers(
        controllerContainer.platformUtilsProvider.getMediaUtils().getVideoTrack(),
        null
      )

      return
    }

    this.hiveUtils.cleanupConsumers(null)

    this.hiveUtils.createWebRtcTransportRecv(null, iceServerData)

    //    val lastActivePeers = this.activatedProducingPeerIds
    this.activatedProducingPeerIds = mutableSetOf()

    //    lastActivePeers.forEach {
    //      this.consumePeer(it, true)
    //    }
    controllerContainer.participantController.refreshGridParticipantsHive(true)
  }

  private suspend fun handleHiveInternalEvents() {
    this.hiveUtils.getObserverFlow().collect {
      if (it.eventName == "reconnect_transport") {
        val transport = it.data as HiveTransport

        try {
          this.reconnectTransport(transport)
          logger.info("HiveController: Transport reconnected ${transport.getId()}")
        } catch (e: Error) {
          logger.info("HiveController: Error on reconnecting transports: $e")
        }
      }
    }
  }

  override suspend fun connectToMediaProduction() {}
}

interface IRoomNodeController {
  fun init() {}

  suspend fun joinRoom()

  suspend fun connectToMediaProduction()

  fun connectProducerTransport() {}

  // Hive specific methods
  suspend fun reconnectTransport(transport: HiveTransport) {}

  suspend fun activatePeer(peerId: String, producers: List<ProducerState>, force: Boolean?) {}

  suspend fun deactivatePeer(peerId: String, producers: List<ProducerState>) {}

  suspend fun closeConsumers(peers: List<String>) {}

  suspend fun consumePeer(producingPeerId: String, force: Boolean? = null) {}

  fun leaveRoom()
}

data class PeerProducerMeta(
  var producerId: String,
  var kind: String,
  var paused: Boolean,
  var screenShare: Boolean,
  var peerId: String,
)
