package io.dyte.core.chat

import io.dyte.core.chat.SocketChatMessage.MessageType
import io.dyte.core.controllers.IMetaController
import io.dyte.core.controllers.ISocketController
import io.dyte.core.observability.DyteLogger
import io.dyte.core.platform.IDytePlatformUtilsProvider
import io.dyte.core.socket.ISocketMessageResponseParser
import io.dyte.core.socket.events.OutboundMeetingEventType.GET_CHAT_MESSAGES
import io.dyte.core.socket.events.OutboundMeetingEventType.SEND_CHAT_MESSAGE
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketChatMessage
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketChatMessagesModel
import io.dyte.core.socket.socketservice.SocketService
import io.dyte.core.socket.socketservice.SocketServiceEventListener
import kotlinx.serialization.encodeToString
import kotlinx.serialization.json.Json
import kotlinx.serialization.json.JsonObject
import kotlinx.serialization.json.buildJsonObject
import kotlinx.serialization.json.put
import socket.chat.ChatMessage
import socket.chat.GetChatMessagesResponse
import socket.chat.SendChatMessageToRoomRequest
import socket.chat.SendChatMessageToRoomResponse

/**
 * Chat socket interface which is used by ChatController or can be directly used by DyteChat client
 * to exchange chat messages with the Sockets.
 *
 * Note(swapnil): This interface should have 2 implementations: SocketIO and SocketService. And
 * based on the feature-flag, the corresponding implementation should be passed to the
 * ChatController or DyteClient.
 */
internal interface ChatSocketHandler {

  fun setOnMessageListener(listener: OnMessageListener)

  fun removeOnMessageListener()

  /** Fetches chat messages from the meeting. */
  suspend fun getChatMessages(): List<SocketChatMessage>

  /** Sends a text message for all the participants in the meeting room. */
  suspend fun sendTextMessage(message: String)

  /** Sends an image message for all the participants in the meeting room. */
  suspend fun sendImageMessage(imageUrl: String)

  /** Sends a file message for all the participants in the meeting room. */
  suspend fun sendFileMessage(fileUrl: String, fileName: String, fileSize: Long)

  interface OnMessageListener {
    /** Called when a new message is received for all the participants in the room. */
    fun onMessageToRoom(message: SocketChatMessage)

    /*
     * Note(swapnil): This method will be used when we implement private chat.
     * TODO: confirm the usage with web folks
     *  */
    // fun onMessageToPeer(message: SocketChatMessage)
  }
}

/** Currently combines the Request/Outgoing chat messages to both Socket-service and SocketIO. */
internal class DefaultChatSocketHandler(
  private val chatSocketServer: String,
  private val roomNodeSocket: ISocketController,
  private val socketService: SocketService,
  private val metaController: IMetaController,
  private val platformUtilsProvider: IDytePlatformUtilsProvider,
  private val socketIoMessageResponseParser: ISocketMessageResponseParser,
) : ChatSocketHandler {
  private var onMessageListener: ChatSocketHandler.OnMessageListener? = null

  private val roomMessageEventListener =
    object : SocketServiceEventListener {
      override fun onEvent(event: Int, eventId: String?, payload: ByteArray?) {
        payload?.let { onSocketServiceMessageToRoom(payload) }
      }
    }

  override fun setOnMessageListener(listener: ChatSocketHandler.OnMessageListener) {
    if (chatSocketServer == MEETING_SOCKET_SOCKET_SERVICE) {
      onMessageListener = listener
      subscribeToSocketServiceChatEvents()
    }
  }

  override fun removeOnMessageListener() {
    unSubscribeToSocketServiceChatEvents()
    onMessageListener = null
  }

  override suspend fun getChatMessages(): List<SocketChatMessage> {
    if (chatSocketServer == MEETING_SOCKET_SOCKET_SERVICE) {
      return getChatMessagesViaSocketService()
    }

    return getChatMessagesViaSocketIO()
  }

  override suspend fun sendTextMessage(message: String) {
    if (chatSocketServer == MEETING_SOCKET_SOCKET_SERVICE) {
      sendTextMessageViaSocketService(message)
      return
    }

    sendTextMessageViaSocketIO(message)
  }

  override suspend fun sendImageMessage(imageUrl: String) {
    if (chatSocketServer == MEETING_SOCKET_SOCKET_SERVICE) {
      sendImageMessageViaSocketService(imageUrl)
      return
    }

    sendImageMessageViaSocketIO(imageUrl)
  }

  override suspend fun sendFileMessage(fileUrl: String, fileName: String, fileSize: Long) {
    if (chatSocketServer == MEETING_SOCKET_SOCKET_SERVICE) {
      sendFileMessageViaSocketService(fileUrl, fileName, fileSize)
      return
    }

    sendFileMessageViaSocketIO(fileUrl, fileName, fileSize)
  }

  private suspend fun getChatMessagesViaSocketService(): List<SocketChatMessage> {
    val response =
      socketService.requestResponse(event = ChatSocketEvent.GET_MESSAGES.id, payload = null)
        ?: throw Exception("Failed to get messages from server")

    val getChatMessagesResponse = GetChatMessagesResponse.ADAPTER.decode(response)
    val socketChatMessages = mutableListOf<SocketChatMessage>()
    for (socketServiceChatMessage in getChatMessagesResponse.messages) {
      try {
        val socketChatMessage = fromSocketServiceChatMessage(socketServiceChatMessage)
        socketChatMessages.add(socketChatMessage)
      } catch (e: Exception) {
        DyteLogger.error("ChatSocketHandler::getChatMessages::error", e)
      }
    }
    return socketChatMessages.toList()
  }

  private suspend fun getChatMessagesViaSocketIO(): List<SocketChatMessage> {
    val chatMessagesResponse = roomNodeSocket.sendMessage(GET_CHAT_MESSAGES, null)
    val parsedResponse = socketIoMessageResponseParser.parseResponse(chatMessagesResponse)
    val chatMessages = parsedResponse.payload as WebSocketChatMessagesModel
    if (chatMessages.messages == null) {
      throw Exception("Failed to get messages from server")
    }

    val socketChatMessages = mutableListOf<SocketChatMessage>()
    for (socketIoChatMessage in chatMessages.messages) {
      try {
        val socketChatMessage = fromSocketIoChatMessage(socketIoChatMessage)
        socketChatMessages.add(socketChatMessage)
      } catch (e: Exception) {
        DyteLogger.error("ChatSocketHandler::getChatMessagesSocketIO::error", e)
      }
    }
    return socketChatMessages.toList()
  }

  private suspend fun sendTextMessageViaSocketService(message: String) {
    sendMessageToRoomViaSocketService(MessageType.TEXT.type, message)
  }

  private suspend fun sendImageMessageViaSocketService(imageUrl: String) {
    sendMessageToRoomViaSocketService(MessageType.IMAGE.type, imageUrl)
  }

  private suspend fun sendFileMessageViaSocketService(
    fileUrl: String,
    fileName: String,
    fileSize: Long
  ) {
    val message = SocketServiceFileMessage(fileUrl, fileName, fileSize)
    sendMessageToRoomViaSocketService(MessageType.FILE.type, Json.encodeToString(message))
  }

  private suspend fun sendMessageToRoomViaSocketService(messageType: Int, message: String) {
    val sendChatMessageToRoomRequest =
      SendChatMessageToRoomRequest(payload_type = messageType, payload = message)
    socketService.send(
      event = ChatSocketEvent.SEND_MESSAGE_TO_ROOM.id,
      payload = SendChatMessageToRoomRequest.ADAPTER.encode(sendChatMessageToRoomRequest)
    )
  }

  private suspend fun sendTextMessageViaSocketIO(message: String) {
    val messagePayload = buildJsonObject {
      put(PAYLOAD_KEY_USER_ID, metaController.getPeerId())
      put(PAYLOAD_KEY_DISPLAY_NAME, metaController.getDisplayName())
      put(PAYLOAD_KEY_MESSAGE_TYPE, MessageType.TEXT.type)
      put(PAYLOAD_KEY_MESSAGE, message)
      put(PAYLOAD_KEY_TIME, platformUtilsProvider.getPlatformUtils().getCurrentTime())
    }
    sendMessageToRoomViaSocketIO(messagePayload)
  }

  private suspend fun sendImageMessageViaSocketIO(imageUrl: String) {
    val messagePayload = buildJsonObject {
      put(PAYLOAD_KEY_USER_ID, metaController.getPeerId())
      put(PAYLOAD_KEY_DISPLAY_NAME, metaController.getDisplayName())
      put(PAYLOAD_KEY_MESSAGE_TYPE, MessageType.IMAGE.type)
      put(PAYLOAD_KEY_LINK, imageUrl)
      put(PAYLOAD_KEY_TIME, platformUtilsProvider.getPlatformUtils().getCurrentTime())
    }
    sendMessageToRoomViaSocketIO(messagePayload)
  }

  private suspend fun sendFileMessageViaSocketIO(
    fileUrl: String,
    fileName: String,
    fileSize: Long
  ) {
    val messagePayload = buildJsonObject {
      put(PAYLOAD_KEY_USER_ID, metaController.getPeerId())
      put(PAYLOAD_KEY_DISPLAY_NAME, metaController.getDisplayName())
      put(PAYLOAD_KEY_MESSAGE_TYPE, MessageType.FILE.type)
      put(PAYLOAD_KEY_LINK, fileUrl)
      put(PAYLOAD_KEY_FILE_NAME, fileName)
      put(PAYLOAD_KEY_FILE_SIZE, fileSize)
      put(PAYLOAD_KEY_TIME, platformUtilsProvider.getPlatformUtils().getCurrentTime())
    }
    sendMessageToRoomViaSocketIO(messagePayload)
  }

  private suspend fun sendMessageToRoomViaSocketIO(messagePayload: JsonObject) {
    roomNodeSocket.sendMessageAsync(SEND_CHAT_MESSAGE, messagePayload)
  }

  private fun subscribeToSocketServiceChatEvents() {
    subscribeToSendMessageToRoomEvent()
  }

  private fun unSubscribeToSocketServiceChatEvents() {
    unSubscribeToSendMessageToRoomEvent()
  }

  private fun subscribeToSendMessageToRoomEvent() {
    socketService.subscribe(
      event = ChatSocketEvent.SEND_MESSAGE_TO_ROOM.id,
      listener = roomMessageEventListener
    )
  }

  private fun unSubscribeToSendMessageToRoomEvent() {
    socketService.unsubscribe(
      event = ChatSocketEvent.SEND_MESSAGE_TO_ROOM.id,
      listener = roomMessageEventListener
    )
  }

  private fun onSocketServiceMessageToRoom(payload: ByteArray) {
    try {
      val sendChatMessageToRoomResponse = SendChatMessageToRoomResponse.ADAPTER.decode(payload)
      val message = sendChatMessageToRoomResponse.message ?: return
      val socketChatMessage = fromSocketServiceChatMessage(message)
      onMessageListener?.onMessageToRoom(socketChatMessage)
    } catch (e: Exception) {
      DyteLogger.error("ChatSocketHandler::getChatMessagesSocketService::error", e)
    }
  }

  private fun fromSocketServiceChatMessage(message: ChatMessage): SocketChatMessage {
    return SocketChatMessageMapper.fromSocketServiceChatMessage(message)
  }

  private fun fromSocketIoChatMessage(
    socketIoChatMessage: WebSocketChatMessage
  ): SocketChatMessage {
    return SocketChatMessageMapper.fromSocketIoChatMessage(socketIoChatMessage)
  }

  @kotlinx.serialization.Serializable
  private data class SocketServiceFileMessage(val link: String, val name: String, val size: Long)

  companion object {
    // Note(swapnil): These constants can be later tied to the FeatureFlagService file itself
    const val MEETING_SOCKET_SOCKET_SERVICE = "socket-service"
    const val MEETING_SOCKET_ROOM_NODE = "room-node"

    private const val PAYLOAD_KEY_USER_ID = "userId"
    private const val PAYLOAD_KEY_DISPLAY_NAME = "displayName"
    private const val PAYLOAD_KEY_MESSAGE_TYPE = "type"
    private const val PAYLOAD_KEY_LINK = "link"
    private const val PAYLOAD_KEY_FILE_NAME = "name"
    private const val PAYLOAD_KEY_FILE_SIZE = "size"
    private const val PAYLOAD_KEY_TIME = "time"
    private const val PAYLOAD_KEY_MESSAGE = "message"
  }
}
