package io.dyte.core.models

import io.dyte.core.controllers.IControllerContainer
import io.dyte.core.feat.DyteMeetingParticipant
import io.dyte.core.feat.DyteWaitlistedParticipant
import kotlinx.coroutines.*
import kotlinx.serialization.json.JsonObject

/**
 * Dyte room participants
 *
 * @property waitlisted A map that contains all the participants waiting to join the meeting.
 * @property joined A list that contains all the participants who have joined the meeting.
 * @property active A map that contains all the participants except the local user who are supposed to be on the screen at the moment
 * @property pinned A map that contains all the pinned participants of the meeting
 *
 * @constructor Create empty Dyte room participants
 */
data class DyteRoomParticipants internal constructor(
  val waitlisted: List<DyteWaitlistedParticipant>,
  val joined: List<DyteJoinedMeetingParticipant>,
  private val _active: List<DyteJoinedMeetingParticipant>,
  val screenshares: List<DyteScreenShareMeetingParticipant>,
  private val controllerContainer: IControllerContainer,
) {

  val  serialScope = CoroutineScope(newSingleThreadContext("DyteRoomParticipants"))

  val pinned: DyteJoinedMeetingParticipant?
    get() = runBlocking { withContext(serialScope.coroutineContext) {
      controllerContainer.participantController.pinnedParticipant
    } }

  val active: List<DyteJoinedMeetingParticipant>
    get() = _active.distinctBy { it.id }

  val activeSpeaker: DyteJoinedMeetingParticipant?
    get() = runBlocking { withContext(serialScope.coroutineContext) {
      controllerContainer.participantController.activeSpeaker
    } }

  val currentPageNumber : Int
    get() = controllerContainer.participantController.getGridPagesInfo().currentPageNumber

  val canGoNextPage : Boolean
    get() = controllerContainer.participantController.getGridPagesInfo().isNextPagePossible

  val canGoPreviousPage : Boolean
    get() = controllerContainer.participantController.getGridPagesInfo().isPreviousPagePossible

  val pageCount: Int
    get() = controllerContainer.participantController.getGridPagesInfo().pageCount

  fun toMap(): Map<String, Any?> {
    val map = HashMap<String, Any?>()
    map["waitlisted"] = listDyteMeetingParticipantToMap(waitlisted)
    map["joined"] = listDyteMeetingParticipantToMap(joined)
    map["active"] = listDyteMeetingParticipantToMap(active)
    map["screenshares"] = listDyteMeetingParticipantToMap(screenshares)
    map["pinned"] = pinned?.toMap()
    map["activeSpeaker"] = activeSpeaker?.toMap()
    map["grid"] = controllerContainer.participantController.getGridPagesInfo().toMap()
    return map
  }

  private fun listDyteMeetingParticipantToMap(participants: List<DyteMeetingParticipant>): List<Map<String, Any?>> {
    return participants.map { it.toMap() }
  }

  @Throws(Exception::class)
  fun setPage(pageNumber: Int) {
    if (pageNumber < 0) {
      throw IllegalArgumentException("page number cant be negative")
    }
    serialScope.launch {
      controllerContainer.participantController.setPage(pageNumber)
    }
  }

  fun disableAllAudio() {
    // TODO : fix this hardcoding
    serialScope.launch {
      controllerContainer.hostController.muteAllAudio(true)
    }
  }

  @Throws(Exception::class)
  fun disableAllVideo() {
    serialScope.launch {
      controllerContainer.hostController.muteAllVideo()
    }
  }

  @Throws(Exception::class)
  fun acceptAllWaitingRequests() {
    serialScope.launch {
      controllerContainer.waitlistController.acceptAllWaitingRequests()
    }
  }

  /**
   * remove all participants from the meeting
   */
  fun kickAll() {
    serialScope.launch {
      controllerContainer.hostController.kickAll()
    }
  }

  fun broadcastMessage(payload: JsonObject) {
    serialScope.launch {
      controllerContainer.participantController.broadcastMessage(payload)
    }
  }
}