package io.dyte.core.media

import io.dyte.callstats.CallStats
import io.dyte.callstats.media.ConsumerFacade
import io.dyte.callstats.media.ProducerFacade
import io.dyte.callstats.media.TransportFacade
import io.dyte.callstats.models.AuthPayload
import io.dyte.core.controllers.DyteEventType
import io.dyte.core.controllers.IControllerContainer
import io.dyte.core.controllers.PermissionType
import io.dyte.core.feat.DyteMeetingParticipant
import io.dyte.core.network.models.IceServerData
import io.dyte.core.observability.ILoggerController
import io.dyte.core.platform.IDyteSFUUtils
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketConsumerClosedModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketConsumerResumedModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketPeerMuteModel
import io.dyte.core.socket.events.payloadmodel.outbound.WebRtcCreateTransportModel
import io.dyte.core.socket.socketservice.SocketService
import io.dyte.core.socket.socketservice.SocketServiceUtils
import io.dyte.media.hive.*
import io.dyte.media.hive.handlers.HiveCodecOptions
import io.dyte.media.utils.IMediaClientLogger
import io.dyte.media.utils.sdp.SDPUtils
import io.dyte.webrtc.*
import io.ktor.utils.io.core.*
import kotlinx.coroutines.*
import kotlinx.coroutines.flow.MutableSharedFlow
import kotlinx.coroutines.flow.launchIn
import kotlinx.coroutines.flow.onEach
import kotlinx.serialization.json.Json
import kotlinx.serialization.json.JsonPrimitive
import media.CreateTransportRequest
import media.CreateTransportResponse
import media.ProducerKind
import media.RenegotiateRequest
import media.Target
import media.SessionDescription
import media.edge.*
import kotlin.Pair

class DyteHive internal constructor(
  controllerContainer: IControllerContainer,
  socket: SocketService,
  private val callStatsClient: CallStats
) : IDyteHive {
  @OptIn(ExperimentalCoroutinesApi::class)
  private val coroutineScope = CoroutineScope(newSingleThreadContext("DyteHive"))
  private val defScope = CoroutineScope(Dispatchers.Default)

  private val logger = controllerContainer.loggerController

  private var _device: HiveDevice? = null

  private var _sendTransport: HiveTransport? = null

  private var _recvTransport: HiveTransport? = null

  private val _consumers = mutableMapOf<String, HiveConsumer>()

  private val _producers = mutableMapOf<String, HiveProducer>()

  override val producerIdToConsumerIdMap = mutableMapOf<String, String>()

  private val consumersToParticipants = HashMap<String, DyteMeetingParticipant>()

  private var _socket = socket

  private var _controllerContainer = controllerContainer

  private var localAudioTrack: AudioStreamTrack? = null

  private var localVideoTrack: VideoStreamTrack? = null

  val observer = MutableSharedFlow<HiveEmitData>()

  override fun init() {
    val hiveLogger = HiveLogger(_controllerContainer.loggerController)
    _controllerContainer.platformUtilsProvider.getMediaUtils().setLogger(_controllerContainer.loggerController)
    _controllerContainer.platformUtilsProvider.getMediaUtils().setPlatform(_controllerContainer.platformUtilsProvider.getPlatformUtils().getOsName())

    _device = HiveDevice(HiveDeviceOptions(), coroutineScope, hiveLogger)

    startCallStats()

    if (_controllerContainer.permissionController.isPermissionGrated(PermissionType.CAMERA) && _controllerContainer.presetController.canPublishVideo()) {
      logger.traceLog("DyteMobileClient | DyteHive init creating video track")
      localVideoTrack =
        _controllerContainer.platformUtilsProvider.getMediaUtils().getVideoTrack()
      _controllerContainer.selfController.getSelf()._videoTrack = localVideoTrack
      _controllerContainer.selfController.getSelf()._videoEnabled = true
    } else {
      logger.traceLog("DyteMobileClient | DyteHive init no camera")
    }

    if (_controllerContainer.permissionController.isPermissionGrated(PermissionType.MICROPHONE) && _controllerContainer.presetController.canPublishAudio()) {
      logger.traceLog("DyteMobileClient | DyteHive init creating audio track")
      localAudioTrack =
        _controllerContainer.platformUtilsProvider.getMediaUtils().createAudioTrack()
      _controllerContainer.selfController.getSelf()._audioEnabled = true
    } else {
      logger.traceLog("DyteMobileClient | DyteHive init no microphone")
    }
  }

  private fun startCallStats() {
    callStatsClient.authenticate(AuthPayload(_controllerContainer.metaController.getPeerId()))

    callStatsClient.startPingStats(interval = 7000)
  }

  override fun getObserverFlow() = observer

  fun getProducers() = this._producers

  override fun getConsumers() = this._consumers

  private fun stringToSessionDescriptionType(str: String): SessionDescriptionType? {
    when (str.lowercase()) {
      "offer" -> return SessionDescriptionType.Offer
      "pranswer" -> return SessionDescriptionType.Pranswer
      "answer" -> return SessionDescriptionType.Answer
      "rollback" -> return SessionDescriptionType.Rollback
    }

    return null
  }

  override suspend fun setupTransports(iceServers: List<IceServerData>?) {
    logger.traceLog("HiveDebug: Inside setup transports")
    val iceServersData = iceServers ?: _controllerContainer.apiClient.getICEServers().iceServers

    this.createWebRtcTransportProd(model = null, iceServers = iceServersData)
    this.createWebRtcTransportRecv(model = null, iceServers = iceServersData)
  }

  override suspend fun stopAllTransports() {
    logger.traceLog("HiveUtils: Closing all the transports")
    this._sendTransport?.close()
    this._recvTransport?.close()

    this._sendTransport = null
    this._recvTransport = null
  }

  private suspend fun negotiate(transport: HiveTransport, offer: io.dyte.webrtc.SessionDescription)
    : io.dyte.webrtc.SessionDescription {
    logger.traceLog("HiveUtils: Initiating negotiation with offer = ${offer.sdp}")
    logger.traceLog("Setting remote offer on transport ${transport.getServerId()}")
    val answer = transport.setRemoteOffer(offer)

    val regReq = RenegotiateRequest(
      transport_id = transport.getId()!!,
      description = SessionDescription(
        sdp = answer.sdp,
        type = answer.type.toString().lowercase(),
        target = Target.SUBSCRIBER
      )
    )

    logger.traceLog("HiveUtils: Sending renegotiate request")
    this._socket.send(
      event = SocketServiceUtils.MediaEvent.RENEGOTIATE_SESSION_DESCRIPTION.id,
      payload = RenegotiateRequest.ADAPTER.encode(regReq)
    )

    logger.traceLog("HiveUtils: Renegotiation done for ${transport.getServerId()}")
    return answer
  }

  private suspend fun handleTransport(transport: HiveTransport, consuming: Boolean) {
    logger.traceLog("HiveDebug: Handling transport")
    transport.observer.onEach {
      when (it.eventName) {
        "connect" -> {
          logger.traceLog("HiveDebug: Received connect event")
          val offer = it.data as io.dyte.webrtc.SessionDescription

          logger.traceLog("HiveDebug: Offer = ${offer.sdp}")

          val req = CreateTransportRequest(
            consuming = consuming,
            description = SessionDescription(
              sdp = offer.sdp,
              type = offer.type.toString().lowercase(),
              target = if (consuming) Target.SUBSCRIBER else Target.PUBLISHER
            )
          )

          try {
            val response = _socket.requestResponse(
              event = SocketServiceUtils.MediaEvent.CREATE_WEBRTC_TRANSPORT.id,
              payload = CreateTransportRequest.ADAPTER.encode(req)
            )

            logger.traceLog("HiveDebug: Recived response for create transport $response")

            val responseDecoded = CreateTransportResponse.ADAPTER.decode(response!!)

            logger.traceLog("HiveDebug: decoded response for create transport $responseDecoded")

            transport.setServerId(responseDecoded.transport_id)
            logger.traceLog(
              "HiveDebug: Server ID for ${transport.getDirection()} = " + responseDecoded.transport_id
            )

            val description = responseDecoded.description
            val answer = SessionDescription(
              type = stringToSessionDescriptionType(
                description!!.type
              )!!,
              sdp = description.sdp
            )

            transport.externalObserver.emit(
              HiveEmitData(
                eventName = "returnConnect",
                data = answer
              )
            )
          } catch (e: Error) {
            logger.traceLog("HiveUtils: ${transport.getDirection()} connection error")
          }
        }

        "connectionstatechange" -> {
          val state = it.data as HiveConnectionState

          when (state) {
            HiveConnectionState.Failed -> {
              if (transport.getClosed()) return@onEach
              logger.traceLog("HiveUtils: ${transport.getDirection()} transport failed")

              // TODO: Flagsmith stuff for reconnect transport
            }

            HiveConnectionState.Disconnected -> {
              if (transport.getClosed()) return@onEach
              logger.traceLog("HiveUtils: ${transport.getDirection()} transport disconencted")

              // TODO: Flagsmith stuff for reconnect transport
            }

            else -> {}
          }
        }

        "icecandidate" -> {
          val candidate = it.data as IceCandidate

          logger.traceLog("HiveUtils: Sending IceCandidate: ${candidate.candidate}")
        }

        "datachannel" -> {
          logger.traceLog("HiveDebug: Received data channel event in mobile-core")
          val data = it.data as Map<String, Any>

          val channel = data["channel"] as DataChannel
          val msg = data["parsedMessage"] as DCMessage

          logger.traceLog(
            "HiveUtils: Got data channel message on event = ${channel.label}: " +
              "${msg.payload}"
          )

          if (channel.label == "negotiation") {
            if (!consuming) {
              logger.traceLog("HiveUtils: Not a consuming transport")
              return@onEach
            }

            // TODO: Check what different thing is to be done
          }

          try {
            when (msg.type) {
              "offer" -> {
                val offer = SessionDescription(
                  type = SessionDescriptionType.Offer,
                  sdp = msg.payload["sdp"]?.content!!.replace("\\r\\n", "\n")
                )

                lateinit var resPayload: DCMessage

                try {
                  val ans = transport.setRemoteOffer(offer)
                  resPayload = DCMessage(
                    type = "answer",
                    payload = mapOf(
                      "type" to JsonPrimitive(ans.type.toString().lowercase()),
                      "sdp" to JsonPrimitive(ans.sdp)
                    )
                  )
                } catch (e: Error) {
                  logger.traceLog("HiveUtils: datachannel:events error: $e")
                  resPayload = DCMessage(
                    type = "error",
                    payload = mapOf(
                      "error" to JsonPrimitive(e.message!!)
                    )
                  )
                }

                logger.traceLog("HiveUtils: Datachannel answer: ${resPayload.payload}")

                val resPayloadByteArray = (Json.encodeToString(
                  DCMessage.serializer(), resPayload
                )).toByteArray()
                channel.send(resPayloadByteArray)
              }
              "consumer_toggle" -> {
                val mute = msg.payload["mute"]?.content == "true"
                val trackId = msg.payload["trackId"]?.content

                logger.traceLog("HiveUtils: Consmer toggled for $trackId, muted = $mute")

                val consumer = this.getConsumers()[trackId]

                if (consumer == null) {
                  logger.traceLog("HiveUtils: Consumer with Id = $trackId not found")
                  return@onEach
                }

                if (consumer.getPaused() != mute) {
                  logger.traceLog("HiveUtils: Consumer state is not same")

                  if (mute) {
                    handlePauseConsumer(consumer.getId())
                  } else {
                    handleResumeConsumer(consumer.getId())
                  }
                }
              }
              else -> logger.traceLog(
                "HiveUtils: Unknown event received from hive node:" +
                  " ${msg.type}"
              )
            }
          } catch (e: Error) {
            logger.traceLog(
              "HiveUtils: Unable to handle the incoming datachannel message on " +
                channel.label
            )
          }
        }

        "negotiate" -> {
          val offer = it.data as io.dyte.webrtc.SessionDescription

          val answer = negotiate(transport, offer)

          transport.externalObserver.emit(
            HiveEmitData(
              eventName = "returnNegotiate",
              data = answer
            )
          )
        }

        "close" -> {
          logger.traceLog(
            "HiveUtils: ${transport.getDirection()} with id = " +
              "${transport.getId()} closed"
          )

          if (consuming) {
            this._consumers.values.forEach { consumer ->
              callStatsClient.registerConsumerListenerOnTransportClose(
                ConsumerFacade(
                  isHive = true,
                  hiveConsumer = consumer
                )
              )
            }

            this._recvTransport = null
            return@onEach
          }

          this._producers.values.forEach { producer ->
            callStatsClient.registerProducerListenerOnTransportClose(
              ProducerFacade(
                isHive = true,
                hiveProducer = producer
              )
            )
          }
          this._sendTransport = null
        }

        "produce" -> {
          logger.traceLog("HiveDebug: Produce event received")

          if (consuming) {
            logger.traceLog("HiveUtils: Not a producing transport")
            return@onEach
          }

          logger.traceLog("HiveDebug: Producing transport verified")

          val data = it.data as Map<String, Any>

          val offer = data["offer"] as io.dyte.webrtc.SessionDescription
          val kind = data["kind"] as MediaStreamTrackKind
          val paused = data["paused"] as Boolean
          val appData = data["appData"] as Map<String, Any>?

          val parsedSdp = SDPUtils.parse(offer.sdp)
          val msid = parsedSdp.media.last { media ->
            if (kind == MediaStreamTrackKind.Video) {
              media.type == "video"
            } else {
              media.type == "audio"
            }
          }.msid

          val req = ProducerCreateRequest(
            description = SessionDescription(
              sdp = offer.sdp,
              type = offer.type.toString().lowercase(),
              target = Target.PUBLISHER
            ),
            paused = paused,
            kind = kind.toString().lowercase(),
            msid = msid!!,
            screen_share = appData?.get("screenShare") as? Boolean ?: false
          )

          logger.traceLog("HiveDebug: produce req created")

          try {
            val response = _socket.requestResponse(
              event = SocketServiceUtils.MediaEvent.PRODUCE.id,
              payload = ProducerCreateRequest.ADAPTER.encode(req)
            )

            logger.traceLog("HiveDebug: Received response for producer create")

            val responseData = ProducerCreateResponse.ADAPTER.decode(response!!)
            logger.traceLog("HiveUtils: Producer Create Request: $kind ProducerId = ${responseData.producer_id}")
            logger.traceLog("HiveUtils: Producer Create Request: $kind SDP = ${responseData.description!!.sdp}")

            val description = responseData.description
            val answer = SessionDescription(
              type = stringToSessionDescriptionType(
                description!!.type
              )!!,
              sdp = description.sdp
            )

            transport.externalObserver.emit(
              HiveEmitData(
                eventName = "returnProduce",
                data = mapOf(
                  "answer" to answer,
                  "producerId" to responseData.producer_id
                )
              )
            )
          } catch (e: Error) {
            logger.traceLog("HiveUtils: Create Producer Error: $e")
          }
        }

        "consumePeer" -> {
          val producingPeerId = it.data as String

          val req = ConsumePeerRequest(
            producing_peer_id = producingPeerId
          )

          try {
            val response = _socket.requestResponse(
              event = SocketServiceUtils.MediaEvent.CONSUME.id,
              payload = ConsumePeerRequest.ADAPTER.encode(req)
            )

            val consumerStateMap = ConsumePeerResponse.ADAPTER.decode(response!!)
              .consumer_ids_map!!.map

            val consumersMap = mutableMapOf<String, HiveConsumerStateObject>()

            consumerStateMap.forEach { entry ->
              consumersMap[entry.key] = HiveConsumerStateObject(
                consumerId = entry.value.consumer_id,
                trackId = entry.value.producer_track!!.track_id,
                streamId = entry.value.producer_track!!.stream_id,
                kind = if (entry.value.producer_state!!.kind == ProducerKind.VIDEO)
                  MediaStreamTrackKind.Video else MediaStreamTrackKind.Audio,
                screenShare = entry.value.producer_state!!.screen_share,
                paused = entry.value.producer_state!!.pause
              )
            }

            transport.externalObserver.emit(
              HiveEmitData(
                eventName = "returnConsumePeer",
                data = consumersMap
              )
            )
          } catch (e: Error) {
            logger.traceLog("HiveUtils: consumePeer error: $e")
          }
        }

        "consume" -> {
          logger.traceLog("HiveDebug: on consume event")
          val data = it.data as Map<String, Any>

          val producerId = data["producerId"] as String
          val producingPeerId = data["producingPeerId"] as String

          val req = ConsumePeerRequest(
            producing_peer_id = producingPeerId,
            producer_id = producerId
          )

          logger.traceLog("HiveDebug: request created")

          try {
            val response = _socket.requestResponse(
              event = SocketServiceUtils.MediaEvent.CONSUME.id,
              payload = ConsumePeerRequest.ADAPTER.encode(req)
            )

            //logger.traceLog("HiveDebug: $response")

            val consumerState = ConsumePeerResponse.ADAPTER.decode(response!!)
              .consumer_ids_map!!.map[producerId]

            val producerState = consumerState?.producer_state
            val producerTrack = consumerState?.producer_track

            logger.traceLog("HiveDebug: $consumerState")

            logger.traceLog(
              "HiveDebug: ${consumerState?.consumer_id} " +
                "${producerState?.screen_share}" +
                "${producerTrack?.track_id}" +
                "${producerTrack?.stream_id}" +
                "${producerState?.kind}" +
                "${producerState?.pause}"
            )

            transport.externalObserver.emit(
              HiveEmitData(
                eventName = "returnConsume",
                data = HiveConsumerStateObject(
                  consumerId = consumerState!!.consumer_id,
                  screenShare = producerState!!.screen_share,
                  trackId = producerTrack!!.track_id,
                  streamId = producerTrack.stream_id,
                  kind = if (producerState.kind == ProducerKind.VIDEO)
                    MediaStreamTrackKind.Video else MediaStreamTrackKind.Audio,
                  paused = producerState.pause
                )
              )
            )
          } catch (e: Error) {
            logger.traceLog("HiveUtils: Error during consuming on server")
          }
        }
      }
    }.launchIn(defScope)
  }

  override suspend fun createWebRtcTransportRecv(
    model: WebRtcCreateTransportModel?,
    iceServers: List<IceServerData>,
  ) {
    logger.traceLog("HiveDebug: creating transport for receiving")

    if (this._recvTransport != null && this._recvTransport!!.getConnected()) return

    val hiveIceServers = mutableListOf<IceServer>()

    iceServers.filter { it.username?.isNotBlank() ?: false }.forEach {
      hiveIceServers.add(
        IceServer(
          urls = listOf(it.url),
          username = it.username!!,
          password = it.credential!!
        )
      )
    }
    logger.traceLog("HiveDebug: ice servers data ready for creating transport")

    val transport = _device!!.createRecvTransport(
      HiveTransportOptions(iceServers = hiveIceServers)
    )
    logger.traceLog("HiveDebug: created transport for recv $transport")

    transport.init()

    handleTransport(transport, true)

    transport.connect()

    callStatsClient.registerConsumingTransport(
      TransportFacade(
        isHive = true,
        hiveTransport = transport
      )
    )

    logger.traceLog("HiveDebug: receive transport created: $transport")
    _recvTransport = transport

    this.producerIdToConsumerIdMap.clear()
    logger.traceLog("HiveDebug: Recv Transport ID = ${_recvTransport?.getId()}")
  }

  override suspend fun createWebRtcTransportProd(
    model: WebRtcCreateTransportModel?,
    iceServers: List<IceServerData>,
  ) {
    logger.traceLog("HiveDebug: creating transport for producing")

    if (this._sendTransport != null && this._sendTransport!!.getConnected()) return

    val hiveIceServers = mutableListOf<IceServer>()

    iceServers.filter { it.username?.isNotBlank() ?: false }.forEach {
      hiveIceServers.add(
        IceServer(
          urls = listOf(it.url),
          username = it.username!!,
          password = it.credential!!
        )
      )
    }
    logger.traceLog("HiveDebug: ice servers data ready for creating transport")

    val transport = _device!!.createSendTransport(
      HiveTransportOptions(iceServers = hiveIceServers)
    )
    logger.traceLog("HiveDebug: created transport for send $transport")

    transport.init()

    handleTransport(transport, false)

    transport.connect()

    callStatsClient.registerProducingTransport(
      TransportFacade(
        isHive = true,
        hiveTransport = transport
      )
    )

    logger.traceLog("HiveDebug: send transport created: $transport")
    _sendTransport = transport
  }

  override fun handlePauseConsumer(id: String) {
    _controllerContainer.platformUtilsProvider.getPlatformUtils().runOnMainThread {
      val consumer = _consumers[id]
      consumer?.let {
        val participant = requireNotNull(consumersToParticipants[id])

        val isAudioTrack = consumer.getTrack() is AudioStreamTrack

        try  {
          val req = ConsumerToggleRequest(
            consumer_id = id,
            pause = true
          )

          coroutineScope.launch {
            _socket.send(
              event = SocketServiceUtils.MediaEvent.TOGGLE_CONSUMER.id,
              payload = ConsumerToggleRequest.ADAPTER.encode(req)
            )
          }
        } catch (e: Error) {
          logger.traceLog("HiveUtils: Error on pausing consumer $e")
        }

        coroutineScope.launch {
          it.pause()
        }
        if (isAudioTrack) {
          // Audio consumers may be paused even if the audio is being produced at the other end
          // so do not set the state as audioEnabled = false
        } else {
          val appData = consumer.getAppData()
          if (appData?.get("screenShare") == true) {
            val screenShareParticipant = _controllerContainer.participantController.meetingRoomParticipants.screenshares.find { ss ->
              ss.id == participant.id
            }

            screenShareParticipant?.let {
              _controllerContainer.participantController.onPeerScreenSharedEnded(
                screenShareParticipant,
              )
            }
          } else {
            _controllerContainer.participantController.onParticipantVideoMuted(
              participant,
            )
          }
        }
      } ?: run {
        // throw IllegalArgumentException("no consumer found for $id")
        _controllerContainer.loggerController.traceError("no consumer found for $id")
      }
    }
  }

  override fun handleResumeConsumer(id: String) {
    _controllerContainer.platformUtilsProvider.getPlatformUtils().runOnMainThread {
      val consumer = _consumers[id]
      consumer?.let {
        val participant = requireNotNull(consumersToParticipants[id])

        val isVideoTrack = consumer.getTrack() is VideoStreamTrack

        try  {
          val req = ConsumerToggleRequest(
            consumer_id = id,
            pause = false
          )

          coroutineScope.launch {
            _socket.send(
              event = SocketServiceUtils.MediaEvent.TOGGLE_CONSUMER.id,
              payload = ConsumerToggleRequest.ADAPTER.encode(req)
            )
          }
        } catch (e: Error) {
          logger.traceLog("HiveUtils: Error on pausing consumer $e")
        }

        coroutineScope.launch {
          it.resume()
        }

        if (isVideoTrack) {
          val appData = consumer.getAppData()
          if (appData?.get("screenShare") == true) {
            val screenShareParticipant = _controllerContainer.participantController.meetingRoomParticipants.screenshares.find { ss ->
              ss.id == participant.id
            }

            screenShareParticipant?.let {
              _controllerContainer.participantController.onPeerScreenShareStarted(
                screenShareParticipant,
                consumer.getTrack() as VideoStreamTrack
              )
            }
          } else {
            _controllerContainer.participantController.onParticipantVideoUnmuted(
              participant,
              consumer.getTrack() as VideoStreamTrack
            )
          }
        }
      } ?: run {
        // throw IllegalArgumentException("no consumer found for $id")
        _controllerContainer.loggerController.traceError("no consumer found for $id")
      }
    }
  }

  private suspend fun handleProducer(
    producerType: ProducerType,
    producer: HiveProducer,
    onDisconnect: () -> Unit,
  ) {
    producer.observer.collect {
      when (it.eventName) {
        "close" -> {
          if (it.data != null) {
            val data = it.data as Map<String, Any>

            val offer = data["offerSdp"] as io.dyte.webrtc.SessionDescription
            val cb = data["callback"] as suspend (answer: io.dyte.webrtc.SessionDescription) -> Any
            val reason = data["reason"] as String

            logger.traceLog(
              "HiveUtils: Producer Id = ${producer.getId()} closed with " +
                      "reason: $reason"
            )

            val req = ProducerCloseRequest(
              producer_id = producer.getId(),
              description = SessionDescription(
                sdp = offer.sdp,
                type = offer.type.toString().lowercase(),
                target = Target.PUBLISHER
              )
            )

            try {
              val response = _socket.requestResponse(
                event = SocketServiceUtils.MediaEvent.CLOSE_PRODUCER.id,
                payload = ProducerCloseRequest.ADAPTER.encode(req)
              )

              val description = ProducerClosingResponse.ADAPTER.decode(
                response!!
              ).description

              val answer = SessionDescription(
                type = stringToSessionDescriptionType(description!!.type)!!,
                sdp = description.sdp
              )

              cb.invoke(answer)
            } catch (e: Error) {
              logger.traceLog("HiveUtils: Producer close error: $e")
            }
          }

          this._producers.remove(producerType.toString())
          onDisconnect.invoke()

          // TODO: Fix setMaxSpatialLayer
//          if (producerType == ProducerType.SCREENSHARE_VIDEO) {
//            val camProducer = this.getProducers()[ProducerType.CAM.toString()]
//
//            camProducer?.let {
//              try {
//                it.setMaxSpatialLayer(3)
//              } catch (e: Error) {
//                logger.traceLog("HiveDebug: Failed to switch spatial layer: $e")
//              }
//            }
//          }
        }
        "trackended" -> {
          logger.traceLog("HiveUtils: Producer Trackended for $producerType")

          if (!(producerType == ProducerType.CAM || producerType == ProducerType.MIC)) {
            onDisconnect.invoke()
          }
        }
      }
    }
  }

  suspend fun createProducer(
    producerType: ProducerType,
    producerOptions: HiveProducerOptions,
    onDisconnect: () -> Unit,
  ) {
    logger.traceLog("HiveDebug: creating producer")
    if (this._sendTransport == null || this._sendTransport!!.getClosed()) {
      logger.traceLog("HiveDebug: send transport closed = ${this._sendTransport!!.getClosed()}")

      return
    }

    logger.traceLog("HiveDebug: create producer: transport is send and open")
    val producer = this._sendTransport!!.produce(producerOptions)
    logger.traceLog("HiveDebug: producer formed")


    coroutineScope.launch {
      handleProducer(producerType, producer, onDisconnect)
    }
    logger.traceLog("HiveDebug: handled producer")

    this._producers[producerType.toString()] = producer
    logger.traceLog("HiveDebug: create producer, all done!")

    callStatsClient.registerProducer(
      ProducerFacade(
        isHive = true,
        hiveProducer = producer
      )
    )

    // TODO: Fix setMaxSpatialLayer
//    if (this._producers.keys.contains(ProducerType.SCREENSHARE_VIDEO.toString())) {
//      val camProducer = this.getProducers()[ProducerType.CAM.toString()]
//
//      camProducer?.let {
//        try {
//          it.setMaxSpatialLayer(0)
//        } catch (e: Error) {
//          logger.traceLog("HiveDebug: Failed to switch spatial layer: $e")
//        }
//      }
//    }
  }

  override suspend fun pauseProducer(type: String) {
    val producer = this._producers[type]

    if (producer == null) {
      logger.traceLog("HiveUtils: Producer type = $type not found")
      return
    }

    producer.pause()
    logger.traceLog("HiveUtils: $type producer id = ${producer.getId()} paused")
  }

  override suspend fun resumeProducer(type: String) {
    val producer = this._producers[type]

    if (producer == null) {
      logger.traceLog("HiveUtils: Producer type = $type not found")
      return
    }

    producer.resume()
    logger.traceLog("HiveUtils: $type producer id = ${producer.getId()} resumed")
  }

  override fun removeProducer(type: String, stopTracks: Boolean?) {
    val producer = this._producers[type]

    if (producer == null) {
      logger.traceLog("HiveUtils: Producer type = $type not found")
      return
    }

    if (stopTracks != null) {
      if (stopTracks) {
        producer.getTrack().stop()
      }
    }

    coroutineScope.launch {
      producer.close(null)
    }
  }

  override suspend fun stopAllProducers() {
    logger.traceLog("HiveUtils: Stopping all available producers")
    this._producers.forEach {
      logger.traceLog("HiveUtils: Closing producer type = ${it.key}")
      it.value.close(null)
    }
  }

  override suspend fun resetVideoProducers(
    videoTrack: MediaStreamTrack?,
    screenShareTrack: MediaStreamTrack?,
  ) {
    if (videoTrack != null) {
      this.removeProducer(ProducerType.CAM.toString(), false)
      this.shareCam(videoTrack)
    }

    if (screenShareTrack != null) {
      this.removeProducer(ProducerType.SCREENSHARE_VIDEO.toString(), false)
      this.shareScreen(screenShareTrack, null)
    }
  }

  override suspend fun pauseCam() {
    val camProducer = this.getProducers()[ProducerType.CAM.toString()]

    if (camProducer == null) {
      logger.traceLog("HiveUtils: PauseMic: could not find cam producer")
      return
    }

    camProducer.pause()

    val req = ProducerToggleRequest(
      producer_id = camProducer.getId(),
      pause = true
    )

    _socket.send(
      event = SocketServiceUtils.MediaEvent.TOGGLE_PRODUCER.id,
      payload = ProducerToggleRequest.ADAPTER.encode(req)
    )

  }

  override suspend fun resumeCam() {
    val camProducer = this.getProducers()[ProducerType.CAM.toString()]

    if (camProducer == null) {
      logger.traceLog("HiveUtils: PauseMic: could not find cam producer")
      return
    }


    camProducer.resume()

    val req = ProducerToggleRequest(
      producer_id = camProducer.getId(),
      pause = false
    )

    _socket.send(
      event = SocketServiceUtils.MediaEvent.TOGGLE_PRODUCER.id,
      payload = ProducerToggleRequest.ADAPTER.encode(req)
    )

  }

  override suspend fun pauseMic() {
    val micProducer = this.getProducers()[ProducerType.MIC.toString()]

    if (micProducer == null) {
      logger.traceLog("HiveUtils: PauseMic: could not find mic producer")
      return
    }

    if (micProducer.getPaused()) {
      logger.traceLog("HiveUtils: PauseMic: mic producer already paused")
      return
    }

    micProducer.pause()

    val req = ProducerToggleRequest(
      producer_id = micProducer.getId(),
      pause = true
    )

    _socket.send(
      event = SocketServiceUtils.MediaEvent.TOGGLE_PRODUCER.id,
      payload = ProducerToggleRequest.ADAPTER.encode(req)
    )
  }

  override suspend fun resumeMic() {
    val micProducer = this.getProducers()[ProducerType.MIC.toString()]

    if (micProducer == null) {
      logger.traceLog("HiveUtils: PauseMic: could not find mic producer")
      return
    }

    if (!micProducer.getPaused()) {
      logger.traceLog("HiveUtils: PauseMic: mic producer already resumed")
      return
    }


    micProducer.resume()

    val req = ProducerToggleRequest(
      producer_id = micProducer.getId(),
      pause = false
    )

    _socket.send(
      event = SocketServiceUtils.MediaEvent.TOGGLE_PRODUCER.id,
      payload = ProducerToggleRequest.ADAPTER.encode(req)
    )

  }

  override fun disableCam() {
    this.removeProducer(ProducerType.CAM.toString(), null)
  }

  override fun disableMic() {
    this.removeProducer(ProducerType.MIC.toString(), null)
  }

  override fun disableScreenShare() {
    logger.traceLog("HiveUtils: Screen sharing stopped")

    this.removeProducer(ProducerType.SCREENSHARE_VIDEO.toString(), null)
    this.removeProducer(ProducerType.SCREENSHARE_AUDIO.toString(), null)
  }

  override suspend fun replaceTrack(type: String, track: MediaStreamTrack) {
    val producer = this._producers[type]

    if (producer == null) {
      logger.traceLog("HiveUtils: Producer type = $type not found")
      return
    }

    producer.replaceTrack(track)
    logger.traceLog("HiveUtils: Track replaced with id = ${track.id} on producer $type")
  }

  private fun handleConsumer(consumer: HiveConsumer) {
    consumer.observer.onEach {
      when (it.eventName) {
        "close" -> {
          handleCloseConsumer(consumer.getId())
        }
      }
    }.launchIn(coroutineScope)
  }

  fun handleCloseConsumer(consumerId: String) {
    val consumer = _consumers[consumerId]
    consumer?.let {
      val appData = consumer.getAppData()
      val participant = requireNotNull(consumersToParticipants[consumerId])
      if (appData?.get("screenShare") == true) {
        val screenShareParticipant = _controllerContainer.participantController.meetingRoomParticipants.screenshares.find { ss ->
          ss.id == participant.id
        }

        screenShareParticipant?.let {
          screenShareParticipant._screenShareTrack = null
          _controllerContainer.participantController.onPeerScreenSharedEnded(screenShareParticipant)
        }
      } else if (consumer.getKind() == MediaStreamTrackKind.Video) {
        _controllerContainer.participantController.onParticipantVideoMuted(participant)
      } else {
        _controllerContainer.participantController.onPeerAudioMuted(
          WebSocketPeerMuteModel(participant.id)
        )
      }
      _consumers.remove(consumer.getId())
      consumersToParticipants.remove(consumerId)
    }
  }

  private fun _initConsumer(consumer: HiveConsumer?) {
    if (consumer == null) return

    callStatsClient.registerConsumer(
      ConsumerFacade(
        isHive = true,
        hiveConsumer = consumer
      )
    )

    handleConsumer(consumer)

    this._consumers[consumer.getId()] = consumer
    this.producerIdToConsumerIdMap[consumer.getProducerId()] = consumer.getId()
  }

  override suspend fun consumePeer(producingPeerId: String) {
    if (this._recvTransport == null || this._recvTransport!!.getClosed())
      throw Error("Receiving transport not connected")

    val results = this._recvTransport!!.consumePeer(producingPeerId, emptyMap())
    val failedTasks = mutableListOf<HiveConsumerCreationTaskException>()

    results.forEach {
      try {
        val consumer = it.await()

        val participant =
          _controllerContainer.participantController.meetingRoomParticipants.joined
            .find { participant -> participant.id == consumer.getPeerId() }

        participant?.let { participant ->
          logger.traceLog("HiveDebug: Updating participant")
          if (consumer.getTrack().kind == MediaStreamTrackKind.Video) {
            if (consumer.getAppData()?.get("screenShare") as? Boolean == true) {
              val screenShareParticipant = _controllerContainer.participantController.meetingRoomParticipants.screenshares.find { ss ->
                ss.id == participant.id
              }

              screenShareParticipant?.let {
                _controllerContainer.participantController.onPeerScreenShareStarted(screenShareParticipant, consumer.getTrack() as VideoStreamTrack)
              }
            } else {
              logger.traceLog("HiveDebug: setting video info")
              participant._videoEnabled = consumer.getPaused().not() ?: true
              participant._videoTrack = consumer.getTrack() as VideoStreamTrack
              if (participant._videoEnabled)
                _controllerContainer.participantController.onPeerVideoUnmuted(
                  WebSocketConsumerResumedModel(id = participant.id)
                )
              else
                _controllerContainer.participantController.onPeerVideoMuted(
                  WebSocketConsumerClosedModel(id = participant.id)
                )
              logger.traceLog("HiveDebug: video info set")
            }
          } else {
            logger.traceLog("HiveDebug: setting audio info")
            val audioTrack = consumer.getTrack()
            logger.traceLog("AudioConsumer: ${audioTrack.enabled} ${audioTrack.muted} ${audioTrack.readyState}")
            logger.traceLog("AudioConsumer: ${consumer.getPaused()}")
            participant._audioEnabled = consumer.getPaused().not() ?: true
            logger.traceLog("AudioConsumer: ${participant._audioEnabled}")
            if (participant._audioEnabled)
              _controllerContainer.participantController.onPeerAudioUnmuted(WebSocketPeerMuteModel(peerId = participant.id))
            else
              _controllerContainer.participantController.onPeerAudioMuted(WebSocketPeerMuteModel(peerId = participant.id))
          }

          _controllerContainer.eventController.triggerEvent(DyteEventType.OnPeerUpdate(participant))
          consumersToParticipants.put(consumer.getId(), participant)
        }

        this._initConsumer(consumer)
      } catch (e: HiveConsumerCreationTaskException) {
        failedTasks.add(e)
      }
    }

    val retryConsumers = this._recvTransport!!.retryFailedConsumerCreationTasks(failedTasks)

    retryConsumers.forEach {
      try {
        val consumer = it.await()
        this._initConsumer(consumer)
      } catch (e: HiveConsumerCreationTaskException) {
        logger.traceLog("HiveUtils: Consumer creation failed second time")
      }
    }
  }

  override suspend fun handleNewConsumer(hiveConsumerOptions: HiveConsumerOptions) {
    logger.traceLog("HiveDebug: inside handle new consumer")
    if (this._recvTransport == null || this._recvTransport!!.getClosed()) return

    logger.traceLog("HiveDebug: recv transport is not null")

//        val producer  = this.getProducer(consumerOpts.producerId)
//
//        if (producer != null) {
//            logger.traceLog("HiveUtils: Why are you creating a consumer for local producer?")
//            return
//        }
//
//        logger.traceLog("HiveDebug: producer = producer is null")

    try {
      logger.traceLog("HiveDebug: consuming")
      val consumerDeferred = this._recvTransport!!.consume(hiveConsumerOptions)

      logger.traceLog("HiveDebug: active=${consumerDeferred.isActive} cancelled=${consumerDeferred.isCancelled}" +
              " completed=${consumerDeferred.isCompleted}")

      val consumer = consumerDeferred.await()
      logger.traceLog("HiveDebug: here")

      val participant =
        _controllerContainer.participantController.meetingRoomParticipants.joined
          .find { it.id == consumer.getPeerId() }

      participant?.let {
        logger.traceLog("HiveDebug: Updating participant")
        if (consumer.getTrack().kind == MediaStreamTrackKind.Video) {
          if (hiveConsumerOptions.appData?.get("screenShare") as? Boolean == true) {
            val screenShareParticipant = _controllerContainer.participantController.meetingRoomParticipants.screenshares.find { ss ->
              ss.id == participant.id
            }

            screenShareParticipant?.let {
              _controllerContainer.participantController.onPeerScreenShareStarted(screenShareParticipant, consumer.getTrack() as VideoStreamTrack)
            }
          } else {
            logger.traceLog("HiveDebug: setting video info")
            it._videoEnabled = hiveConsumerOptions.paused?.not() ?: true
            it._videoTrack = consumer.getTrack() as VideoStreamTrack
            if (it._videoEnabled)
              _controllerContainer.participantController.onPeerVideoUnmuted(
                WebSocketConsumerResumedModel(id = it.id)
              )
            else
              _controllerContainer.participantController.onPeerVideoMuted(
                WebSocketConsumerClosedModel(id = it.id)
              )
            logger.traceLog("HiveDebug: video info set")
          }
        } else {
          logger.traceLog("HiveDebug: setting audio info")
          val audioTrack = consumer.getTrack()
          logger.traceLog("AudioConsumer: ${audioTrack.enabled} ${audioTrack.muted} ${audioTrack.readyState}")
          logger.traceLog("AudioConsumer: ${hiveConsumerOptions.paused}")
          it._audioEnabled = hiveConsumerOptions.paused?.not() ?: true
          logger.traceLog("AudioConsumer: ${it._audioEnabled}")
          if (it._audioEnabled)
            _controllerContainer.participantController.onPeerAudioUnmuted(WebSocketPeerMuteModel(peerId = it.id))
          else
            _controllerContainer.participantController.onPeerAudioMuted(WebSocketPeerMuteModel(peerId = it.id))
        }

        _controllerContainer.eventController.triggerEvent(DyteEventType.OnPeerUpdate(it))
        consumersToParticipants.put(consumer.getId(), it)
      }



      this._initConsumer(consumer)
//      logger.traceLog("Hivedebug: here")
//
//      if (consumer.getCompletionExceptionOrNull() == null) {
//        logger.traceLog("HiveDebug: consumer created")
//        this._initConsumer(consumer.getCompleted())
//      } else {
//        logger.traceLog("HiveDebug: consumer creation failed")
//        throw consumer.getCompletionExceptionOrNull()!!
//      }
    } catch (e: HiveConsumerCreationTaskException) {
      val retryConsumer = this._recvTransport!!.retryFailedConsumerCreationTasks(
        listOf(e)
      ).first()

      try {
        val consumer = retryConsumer.await()
        this._initConsumer(consumer)
      } catch (e: HiveConsumerCreationTaskException) {
        logger.traceLog("HiveUtils: Failed to complete consumer creation task again")
      }
    }
  }

  override suspend fun closeConsumer(consumerId: String, force: Boolean?) {
    this.closeConsumers(listOf(consumerId), force)
  }

  override suspend fun closeConsumers(consumerIds: List<String>, force: Boolean?) {
    var closeProvidedConsumers = true
    val availableConsumerIds = consumerIds.filter { id ->
      val consumer = this._consumers[id]

      if (consumer == null) {
        logger.traceLog("HiveUtils: Consumer Id = $id not found")
        return@filter false
      }

      return@filter true
    }

    try {
      val req = ConsumerCloseRequest(
        consumer_ids = availableConsumerIds
      )

      this._recvTransport!!.limitedScope.launch {
        // TODO(anunaym14): Revisit this
        _socket.send(
          event = SocketServiceUtils.MediaEvent.CLOSE_CONSUMER.id,
          payload = ConsumerCloseRequest.ADAPTER.encode(req)
        )

//        val response = _socket.requestResponse(
//          event = SocketServiceUtils.MediaEvent.CLOSE_CONSUMER.id,
//          payload = ConsumerCloseRequest.ADAPTER.encode(req)
//        )
//
//        response?.let {
//          val resp = ConsumerClosingResponse.ADAPTER.decode(it)
//          logger.traceLog("HiveUtils: Consumer close response for $resp")
//        }
      }
      // FIXME(itzmanish): here we should get a list of closed consumers from server
      // because there is possibility that if the request somehow fails or server
      // is only able to close a subset of consumers then we should not close all the
      // consumers in the client side, Closing all the consumers in the client side
      // without confirming how many consumers actually closed will produce leak of media.
    } catch (e: Error) {
      logger.traceLog("HiveUtils: Error on Closing Consumer: $e")
      closeProvidedConsumers = force ?: false
    }

    if (closeProvidedConsumers) {
      availableConsumerIds.forEach { id ->
        val consumer = this.getConsumers()[id]

        this.producerIdToConsumerIdMap.remove(consumer?.getProducerId())
        consumer?.close("Manually closed")
      }
    }
  }

  override suspend fun cleanupConsumers(peerId: String?) {
    val consumers = arrayListOf<String>()

    this._consumers.values.forEach { consumer ->
      if (peerId != null) {
        if (consumer.getPeerId() == peerId)
          consumers.add(consumer.getId())
      } else {
        consumers.add(consumer.getId())
      }
    }

    this.closeConsumers(consumers, true)
  }

  override suspend fun shareCam(videoTrack: MediaStreamTrack?) {
    if (videoTrack == null) return

    if (this.getProducers().contains(ProducerType.CAM.toString())) {
      val camProducer = this.getProducers()[ProducerType.CAM.toString()]

      if (!camProducer?.getClosed()!!) {
        camProducer.replaceTrack(videoTrack)
        this.resumeCam()
        return
      }

      this.removeProducer(camProducer.getId(), null)
    }

    // TODO: Feature flag for Simulcast
    val simulcastEnbaled = false
    val simulcastEncodings = mutableListOf<CommonRtpEncodingParameters>()

    if (simulcastEnbaled) {
      // TODO: Fix hardcoded value for video track width
      var encoding = CommonRtpEncodingParameters()
      when(480) {
        320 -> {
          encoding.rid = "q"
          encoding.maxBitrateBps = 250_000
          encoding.maxFramerate = 30
          // Scalability mode

          simulcastEncodings.add(encoding)
        }

        640 -> {
          encoding.rid = "h"
          encoding.maxBitrateBps = 750_000
          encoding.maxFramerate = 30
          // Scalability mode

          simulcastEncodings.add(encoding)

          encoding = CommonRtpEncodingParameters()

          encoding.rid = "q"
          encoding.scaleResolutionDownBy = 2.0
          encoding.maxBitrateBps = 250_000
          encoding.maxFramerate = 30
          // Scalability mode

          simulcastEncodings.add(encoding)
        }

        1280 -> {
          encoding.rid = "f"
          encoding.maxBitrateBps = 1_500_000
          encoding.maxFramerate = 30
          // Scalability mode

          simulcastEncodings.add(encoding)

          encoding = CommonRtpEncodingParameters()

          encoding.rid = "h"
          encoding.scaleResolutionDownBy = 2.0
          encoding.maxBitrateBps = 750_000
          encoding.maxFramerate = 30
          // Scalability mode

          simulcastEncodings.add(encoding)

          encoding = CommonRtpEncodingParameters()

          encoding.rid = "q"
          encoding.scaleResolutionDownBy = 4.0
          encoding.maxBitrateBps = 250_000
          encoding.maxFramerate = 30
          // Scalability mode

          simulcastEncodings.add(encoding)
        }

        else -> {
          encoding.rid = "q"
          encoding.maxBitrateBps = 250_000
          encoding.maxFramerate = 30
          // Scalability mode

          simulcastEncodings.add(encoding)
        }
      }
    }

    val nonSimulcastEncodings = CommonRtpEncodingParameters()
    nonSimulcastEncodings.active = true
    nonSimulcastEncodings.scaleResolutionDownBy = 1.0

    val producerOptions = HiveProducerOptions(
      track = videoTrack,
      codecOptions = HiveCodecOptions(name = "VP8", parameters = null),
      appData = mapOf("screenShare" to false),
      stopTracks = false,
      disableTrackOnPause = null,
      encodings = if (simulcastEnbaled) simulcastEncodings else listOf(nonSimulcastEncodings),
      zeroRtpOnPause = null,
      stream = _controllerContainer.platformUtilsProvider.getMediaUtils().getVideoStream()!!
    )

    val onDisconnect: () -> Unit = {
      runBlocking {
        disableCam()
      }
    }

    this.createProducer(ProducerType.CAM, producerOptions, onDisconnect)

    _controllerContainer.selfController.getSelf()._videoEnabled = true
    _controllerContainer.selfController.getSelf()._videoTrack = localVideoTrack
    _controllerContainer.eventController.triggerEvent(DyteEventType.OnSelfVideoUpdate)
  }

  override suspend fun shareScreen(videoTrack: MediaStreamTrack?, audioTrack: MediaStreamTrack?) {
    if (videoTrack == null) return

    val videoProducerOptions = HiveProducerOptions(
      track = videoTrack,
      codecOptions = HiveCodecOptions(name = "VP8", parameters = null),
      appData = mapOf(
        "screenShare" to true,
        //"supportsRemoteControl"
      ),
      stopTracks = false,
      disableTrackOnPause = null,
      encodings = null,
      zeroRtpOnPause = null,
      stream = _controllerContainer.platformUtilsProvider.getMediaUtils().getVideoStream()!!
    )

    val videoOnDisconnect: () -> Unit = {
      runBlocking {
        disableScreenShare()
      }
    }

    this.createProducer(ProducerType.SCREENSHARE_VIDEO, videoProducerOptions, videoOnDisconnect)

    if (audioTrack != null) {
      val encoding = CommonRtpEncodingParameters()
      encoding.bitratePriority = 4.0

      val audioProducerOptions = HiveProducerOptions(
        track = audioTrack,
        codecOptions = HiveCodecOptions(name = "opus", parameters = null),
        appData = mapOf(
          "screenShare" to true,
          //"supportsRemoteControl"
        ),
        stopTracks = false,
        disableTrackOnPause = null,
        encodings = listOf(encoding),
        zeroRtpOnPause = null,
        stream = _controllerContainer.platformUtilsProvider.getMediaUtils().getAudioStream()!!
      )

      val audioOnDisconnect: () -> Unit = {
        // Do nothing as video will already stop the track
      }

      this.createProducer(
        ProducerType.SCREENSHARE_AUDIO, audioProducerOptions,
        audioOnDisconnect)
    }
  }

  override suspend fun shareMic(audioTrack: MediaStreamTrack?) {
    logger.traceLog("HiveDebug: Inside share mic")

    if (audioTrack == null) return

    if (this.getProducers().contains(ProducerType.MIC.toString())) {
      logger.traceLog("HiveDebug: Mic producer already exists")
      val micProducer = this.getProducers()[ProducerType.MIC.toString()]

      if (!micProducer?.getClosed()!!) {
        micProducer.replaceTrack(audioTrack)
        this.resumeMic()
        return
      }
      this.removeProducer(ProducerType.MIC.toString(), false)
    }

    logger.traceLog("HiveDebug: Mic producer doesn't exist, creating producer")
    val producerOptions = HiveProducerOptions(
      track = audioTrack,
      codecOptions = HiveCodecOptions(name = "opus", parameters = null),
      appData = null,
      stopTracks = false,
      disableTrackOnPause = null,
      encodings = null,
      zeroRtpOnPause = null,
      stream = _controllerContainer.platformUtilsProvider.getMediaUtils().getAudioStream()!!
    )

    val onDisconnect: () -> Unit = {
      runBlocking {
        disableMic()
      }
    }

    this.createProducer(ProducerType.MIC, producerOptions, onDisconnect)
    logger.traceLog("HiveDebug: successfully created mic producer")
  }

  override fun getSelfTrack(): Pair<AudioStreamTrack?, VideoStreamTrack?> {
    return Pair(localAudioTrack, localVideoTrack)
  }

  override suspend fun muteSelfAudio(): Boolean {
    try {
      pauseMic()

      _controllerContainer.selfController.getSelf()._audioEnabled = false
      _controllerContainer.eventController.triggerEvent(DyteEventType.OnSelfAudioUpdate)
    } catch (e: Error) {
      return false
    }

    callStatsClient.sendAudioToggleEvent(false)
    return true
  }

  override suspend fun unmuteSelfAudio(): Boolean {
    logger.traceLog("FixBroken: In unmute self mic")
    try {
      if (!_producers.contains(ProducerType.MIC.toString())) {
        coroutineScope.launch {
          shareMic(localAudioTrack)
        }
      } else {
        resumeMic()
      }

      callStatsClient.sendAudioToggleEvent(true)

      _controllerContainer.selfController.getSelf()._audioEnabled = true
      _controllerContainer.eventController.triggerEvent(DyteEventType.OnSelfAudioUpdate)
      return true
    } catch (e: Error) {
      return false
    }
  }

  override suspend fun muteSelfVideo(): Boolean {
    try {
      pauseCam()

      _controllerContainer.selfController.getSelf()._videoEnabled = false
      _controllerContainer.eventController.triggerEvent(DyteEventType.OnSelfVideoUpdate)
    } catch (e: Error) {
      return false
    }

    callStatsClient.sendVideoToggleEvent(false)
    return true
  }

  override suspend fun unmuteSelfVideo(): Boolean {
    logger.traceLog("FixBroken: In unmute self vid")
    try {
      if (!_producers.contains(ProducerType.CAM.toString())) {
        coroutineScope.launch {
          shareCam(localVideoTrack)
        }
      } else {
        resumeCam()
      }

      callStatsClient.sendVideoToggleEvent(true)

      _controllerContainer.selfController.getSelf()._videoEnabled = true
      _controllerContainer.selfController.getSelf()._videoTrack = localVideoTrack
      _controllerContainer.eventController.triggerEvent(DyteEventType.OnSelfVideoUpdate)
      return true
    } catch (e: Error) {
      return false
    }
  }

  override suspend fun leaveCall() {
    stopAllTransports()
    callStatsClient.stopPingStats()
    callStatsClient.sendDisconnectEvent()

    val req = PeerLeaveRequest(
      close_room = false
    )

    try {
      val response = _socket.requestResponse(
        event = SocketServiceUtils.MediaEvent.LEAVE_ROOM.id,
        payload = PeerLeaveRequest.ADAPTER.encode(req)
      )

      val closed = PeerLeaveResponse.ADAPTER.decode(response!!).closed

      if (!closed)
        logger.traceLog("HiveUtils: Weird state on peer closed, should not happen")
    } catch (e: Error) {
      logger.traceLog("HiveUtils: Error on sending leave room request: $e")
    }
  }

  override fun switchCamera() {
    val camProducer = this.getProducers()[ProducerType.CAM.toString()]

    if (camProducer == null) {
      logger.traceLog("HiveUtils: PauseMic: could not find cam producer")
      return
    }

    runBlocking {
      (camProducer.getTrack() as VideoStreamTrack).switchCamera()
    }
  }

  override suspend fun produceMedia() {
    TODO("Not yet implemented")
  }
}

interface IDyteHive : IDyteSFUUtils {

  val producerIdToConsumerIdMap: MutableMap<String, String>

  fun init()

  suspend fun setupTransports(iceServers: List<IceServerData>? = null)
  suspend fun stopAllTransports()
  suspend fun createWebRtcTransportRecv(
    model: WebRtcCreateTransportModel?,
    iceServers: List<IceServerData>,
  )

  suspend fun createWebRtcTransportProd(
    model: WebRtcCreateTransportModel?,
    iceServers: List<IceServerData>,
  )

  suspend fun pauseProducer(type: String)
  suspend fun resumeProducer(type: String)
  fun removeProducer(type: String, stopTracks: Boolean?)
  suspend fun stopAllProducers()
  suspend fun resetVideoProducers(
    videoTrack: MediaStreamTrack?,
    screenShareTrack: MediaStreamTrack?,
  )

  suspend fun pauseCam()
  suspend fun resumeCam()
  suspend fun pauseMic()
  suspend fun resumeMic()

  fun disableCam()
  fun disableMic()
  fun disableScreenShare()

  suspend fun replaceTrack(type: String, track: MediaStreamTrack)

  suspend fun closeConsumers(consumerIds: List<String>, force: Boolean? = null)
  suspend fun cleanupConsumers(peerId: String?)

  suspend fun shareCam(videoTrack: MediaStreamTrack?)

  suspend fun shareScreen(videoTrack: MediaStreamTrack?, audioTrack: MediaStreamTrack?)

  suspend fun shareMic(audioTrack: MediaStreamTrack?)

  fun getSelfTrack(): Pair<AudioStreamTrack?, VideoStreamTrack?>
  suspend fun handleNewConsumer(hiveConsumerOptions: HiveConsumerOptions)
  fun getConsumers(): Map<String, HiveConsumer>
  suspend fun closeConsumer(consumerId: String, force: Boolean? = null)
  suspend fun consumePeer(producingPeerId: String)
  fun getObserverFlow(): MutableSharedFlow<HiveEmitData>
}

enum class ProducerType {
  CAM,
  MIC,
  SCREENSHARE_VIDEO,
  SCREENSHARE_AUDIO
}

class HiveLogger(private val logger: ILoggerController): IMediaClientLogger {
  override fun traceError(message: String) {
    logger.traceError(message)
  }

  override fun traceLog(message: String) {
    logger.traceLog(message)
  }

  override fun traceWarning(message: String) {
    logger.traceWarning(message)
  }
}
