package io.dyte.core.feat

import io.dyte.core.controllers.*
import io.dyte.core.controllers.Controller
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.launch
import kotlinx.coroutines.newSingleThreadContext

class DyteLiveStream internal constructor(private val controller: Controller) {
  private val serialScope = CoroutineScope(newSingleThreadContext("DyteLiveStream"))

  val liveStreamUrl: String?
    get() = controller.liveStreamController.livestreamData.liveStreamUrl
  val state: LiveStreamState
    get() = controller.liveStreamController.livestreamData.state
  val roomName: String
    get() = controller.metaController.getMeetingTitle()
  val stageRequestPeers : List<LiveStreamStageRequestPeer>
    get() = controller.liveStreamController.stageRequestPeers
  val stageStatus: StageStatus
    get() = controller.stageController.stageStatus

  fun getStagePeers(onPeersLoaded : (stagePeers : List<LiveStreamStagePeer>) -> Unit) {
    serialScope.launch {
      val peers = controller.liveStreamController.getStagePeers()
      onPeersLoaded.invoke(peers)
    }
  }

  fun getStageRequestPeers(onPeersLoaded : (stageRequests : List<LiveStreamStageRequestPeer>) -> Unit) {
    serialScope.launch {
      val requestPeers = controller.liveStreamController.getStageRequests()
      onPeersLoaded.invoke(requestPeers)
    }
  }

  fun start() {
    serialScope.launch {
      controller.liveStreamController.start()
    }
  }

  fun stop() {
    serialScope.launch {
      controller.liveStreamController.stop()
    }
  }

  fun joinStage() {
    serialScope.launch {
      controller.liveStreamController.joinStage()
    }
  }

  fun withdrawRequestToJoin() {
    serialScope.launch {
      controller.liveStreamController.withdrawRequestToJoin()
    }
  }

  fun acceptStageRequest() {
    serialScope.launch { controller.liveStreamController.acceptJoinStage()
    }
  }

  fun leaveStage() {
    serialScope.launch {
      controller.liveStreamController.leaveStage()
    }
  }

  fun acceptAllRequests() {
    println("DyteMobileClient | DyteLiveStream acceptAllRequests ")
    serialScope.launch {
      controller.liveStreamController.acceptAllRequests()
    }
  }

  fun acceptRequest(peer: LiveStreamStageRequestPeer) {
    println("DyteMobileClient | DyteLiveStream acceptRequest ")
    serialScope.launch {
      controller.liveStreamController.acceptRequest(peer.userId)
    }
  }

  fun rejectRequest(peer: LiveStreamStageRequestPeer) {
    println("DyteMobileClient | DyteLiveStream rejectRequest ")
    serialScope.launch {
      controller.liveStreamController.rejectRequest(peer.userId)
    }
  }

  fun rejectAllRequests() {
    println("DyteMobileClient | DyteLiveStream rejectAllRequests ")
    serialScope.launch {
      controller.liveStreamController.rejectAllRequests()
    }
  }
}