package io.dyte.core.controllers

import io.dyte.core.controllers.DyteEventType.OnStageCountUpdated
import io.dyte.core.controllers.DyteEventType.OnViewerCountUpdated
import io.dyte.core.controllers.LiveStreamState.ERRORED
import io.dyte.core.controllers.LiveStreamState.NONE
import io.dyte.core.controllers.LiveStreamState.STARTED
import io.dyte.core.controllers.LiveStreamState.STOPPED
import io.dyte.core.controllers.StageStatus.ACCEPTED_TO_JOIN_STAGE
import io.dyte.core.controllers.StageStatus.OFF_STAGE
import io.dyte.core.feat.DyteMeetingParticipant
import io.dyte.core.socket.socketservice.SocketService
import io.dyte.core.socket.socketservice.SocketServiceEventListener
import io.dyte.core.socket.socketservice.SocketServiceUtils.RoomEvent
import kotlinx.coroutines.launch
import socket.livestreaming.*
import socket.room.RoomPeerCountResponse

internal class LiveStreamController(
  controllerContainer: IControllerContainer,
  private val socketService: SocketService
) : BaseController(controllerContainer), ILiveStreamController {

  private var _liveStreamUrl: String? = null

  private var _state: LiveStreamState = NONE

  private var _viewerCount: Int = 0
  private var _stagePeerCount: Int = 0

  private var _roomName: String = ""

  private var participants = arrayListOf<DyteMeetingParticipant>()
  private var requestedPeers = arrayListOf<LiveStreamStageRequestPeer>()
  private var stagePeers = arrayListOf<LiveStreamStagePeer>()

  override val stageRequestPeers: List<LiveStreamStageRequestPeer>
    get() = requestedPeers

  override val livestreamData: DyteLivestreamData
    get() = DyteLivestreamData(_liveStreamUrl, _roomName, _state, _viewerCount)

  private val socketEventListener = object : SocketServiceEventListener {
    override fun onEvent(event: Int, eventId: String?, payload: ByteArray?) {
        when (event) {
          RoomEvent.STARTED.id -> {
            controllerContainer.eventController.triggerEvent(DyteEventType.OnLiveStreamStarting)
            _state = STARTED
            val decodedPayload = LiveStreamingEvent.ADAPTER.decode(requireNotNull(payload))
            _roomName = decodedPayload.name
            _liveStreamUrl = decodedPayload.playback_url
            controllerContainer.eventController.triggerEvent(DyteEventType.OnLiveStreamStarted)
          }

          RoomEvent.STOPPED.id -> {
            controllerContainer.eventController.triggerEvent(DyteEventType.OnLiveStreamEnding)
            _state = STOPPED
            controllerContainer.eventController.triggerEvent(DyteEventType.OnLiveStreamEnded)
          }

          RoomEvent.ERRORED.id -> {
            _state = ERRORED
            controllerContainer.eventController.triggerEvent(DyteEventType.OnLiveStreamErrored)
          }

          RoomEvent.ROOM_PEER_COUNT.id -> {
            payload?.let {
              val response = RoomPeerCountResponse.ADAPTER.decode(payload)
              _viewerCount = response.count.toInt()
              controllerContainer.eventController.triggerEvent(OnViewerCountUpdated(_viewerCount))
            }
          }

          // when new peer joins stage
          RoomEvent.GET_STAGE_PEERS.id -> {
            payload?.let {
              val response = GetStagePeersResponse.ADAPTER.decode(payload)
              stagePeers.clear()
              response.stage_peers.forEach { peerId ->
                val peersToRemove = requestedPeers.filter { it.userId == peerId || it.peerId == peerId }
                requestedPeers.removeAll(peersToRemove.toSet())
                peersToRemove.forEach { peerOnStage ->
                  stagePeers.add(LiveStreamStagePeer(peerOnStage.displayName, peerOnStage.userId, peerOnStage.peerId))
                }
              }
              _stagePeerCount = response.stage_peers.size
              controllerContainer.eventController.triggerEvent(OnStageCountUpdated(_viewerCount))
            }
          }

          RoomEvent.GRANT_STAGE_REQUEST.id -> {
            serialScope.launch {
              controllerContainer.stageController.setStageStatus(ACCEPTED_TO_JOIN_STAGE)
            }
          }

          // when someone requests to be on stage
          RoomEvent.GET_STAGE_REQUESTS.id -> {
            payload?.let {
              val response = GetStageRequestsResponse.ADAPTER.decode(payload)
              requestedPeers.clear()
              response.stage_requests.forEach { requestPeer ->
                val requestedPeer = LiveStreamStageRequestPeer(
                  requestPeer.display_name,
                  requestPeer.user_id,
                  requestPeer.peer_id
                )
                requestedPeers.add(requestedPeer)
              }
            } ?: run {
              // empty response means no more peers requesting to be on stage
              requestedPeers.clear()
            }

            controllerContainer.eventController.triggerEvent(
              DyteEventType.OnLiveStreamStageRequestsUpdated(
                requestedPeers
              )
            )
          }

          else -> {}
        }
    }
  }

  override fun init() {
    socketService.subscribe(RoomEvent.STARTED.id, socketEventListener)
    socketService.subscribe(RoomEvent.STOPPED.id, socketEventListener)
    socketService.subscribe(RoomEvent.ERRORED.id, socketEventListener)
    socketService.subscribe(RoomEvent.ROOM_PEER_COUNT.id, socketEventListener)
    socketService.subscribe(RoomEvent.GRANT_STAGE_REQUEST.id, socketEventListener)
    socketService.subscribe(RoomEvent.GET_STAGE_PEERS.id, socketEventListener)
    socketService.subscribe(RoomEvent.GET_STAGE_REQUESTS.id, socketEventListener)
    socketService.subscribe(RoomEvent.LEAVE_STAGE.id, socketEventListener)
    loadLiveStreamUrl()
  }

  private fun loadLiveStreamUrl() {
    try {
      serialScope.launch {
        try {
          _liveStreamUrl = controllerContainer.apiClient.getLiveStreamUrl()
          _state = STARTED
          controllerContainer.eventController.triggerEvent(DyteEventType.OnLiveStreamStarted)
        } catch (e: Exception) {
          // ignore exception
        }
      }
    } catch (e: Exception) {
      _state = NONE
      e.printStackTrace()
    }
  }

  override suspend fun loadData() {
      socketService.send(RoomEvent.GET_STAGE_PEERS.id, null)
      socketService.send(RoomEvent.GET_STAGE_REQUESTS.id, null)
  }

  override suspend fun start() {
    controllerContainer.eventController.triggerEvent(DyteEventType.OnLiveStreamStarting)
    controllerContainer.apiClient.startLiveStream()
  }

  override suspend fun stop() {
    controllerContainer.eventController.triggerEvent(DyteEventType.OnLiveStreamEnding)
    controllerContainer.apiClient.stopLiveStream()
  }

  override suspend fun joinStage() {
    controllerContainer.stageController.setStageStatus(StageStatus.REQUESTED_TO_JOIN_STAGE)
    if (controllerContainer.webinarController.canJoinStage()) {
      socketService.send(RoomEvent.REQUEST_STAGE_ACCESS.id, null)
    } else {
      acceptJoinStage()
    }
  }

  override suspend fun acceptJoinStage() {
    if (controllerContainer.roomNodeController.joinedRoom.not()) {
      controllerContainer.socketController.connect()
    }
    controllerContainer.stageController.setStageStatus(StageStatus.ON_STAGE)
    socketService.send(RoomEvent.JOIN_STAGE.id, null)
    controllerContainer.roomNodeController.joinRoom()
  }

  override suspend fun rejectRequestToJoin() {
    socketService.send(RoomEvent.CANCEL_STAGE_REQUEST.id, null)
    controllerContainer.stageController.setStageStatus(OFF_STAGE)
  }

  override suspend fun withdrawRequestToJoin() {
    socketService.send(RoomEvent.CANCEL_STAGE_REQUEST.id, null)
    controllerContainer.stageController.setStageStatus(OFF_STAGE)
  }

  override suspend fun leaveStage() {
    controllerContainer.stageController.setStageStatus(OFF_STAGE)
    val request = LeaveStageRequest(listOf(controllerContainer.selfController.getSelf().userId)).encode()
    socketService.send(RoomEvent.LEAVE_STAGE.id, request)
    controllerContainer.roomNodeController.leaveRoom()
    controllerContainer.platformUtilsProvider.getVideoUtils().destroyAll()
    controllerContainer.sfuUtils.leaveCall()
    controllerContainer.socketController.disconnect()
    controllerContainer.stageController.leaveStage()
  }

  override suspend fun acceptRequest(id: String) {
    socketService.send(
      RoomEvent.GRANT_STAGE_REQUEST.id,
      GrantStageAccessRequest(listOf(id)).encode()
    )
  }

  override suspend fun acceptAllRequests() {
    stageRequestPeers.forEach {
      acceptRequest(it.userId)
    }
  }

  override suspend fun rejectRequest(id: String) {
    socketService.send(RoomEvent.DENY_STAGE_REQUEST.id, DenyStageAccessRequest(listOf(id)).encode())
  }

  override suspend fun rejectAllRequests() {
    stageRequestPeers.forEach {
      rejectRequest(it.userId)
    }
  }

  override suspend fun getStagePeers(): List<LiveStreamStagePeer> {
    return stagePeers
  }

  override suspend fun getStageRequests(): List<LiveStreamStageRequestPeer> {
    val response = socketService.requestResponse(RoomEvent.GET_STAGE_REQUESTS.id, null)
    val parsedResponse = GetStageRequestsResponse.ADAPTER.decode(requireNotNull(response))
    val stagePeers = arrayListOf<LiveStreamStageRequestPeer>()
    parsedResponse.stage_requests.forEach { requestPeer ->
      stagePeers.add(
        LiveStreamStageRequestPeer(
          requestPeer.display_name,
          requestPeer.user_id,
          requestPeer.peer_id
        )
      )
    }
    return stagePeers
  }
}

interface ILiveStreamController {
  val livestreamData: DyteLivestreamData
  val stageRequestPeers: List<LiveStreamStageRequestPeer>

  suspend fun start()
  suspend fun stop()
  suspend fun joinStage()
  suspend fun acceptJoinStage()
  suspend fun leaveStage()

  suspend fun rejectRequestToJoin()
  suspend fun withdrawRequestToJoin()

  suspend fun getStagePeers(): List<LiveStreamStagePeer>

  suspend fun getStageRequests(): List<LiveStreamStageRequestPeer>

  suspend fun acceptRequest(id: String)
  suspend fun acceptAllRequests()

  suspend fun rejectRequest(id: String)
  suspend fun rejectAllRequests()

  suspend fun loadData()
}

enum class LiveStreamState {
  NONE,
  STARTING,
  STARTED,
  STOPPING,
  STOPPED,
  ERRORED;

  companion object {
    fun fromMap(stage: String): LiveStreamState {
      return when (stage) {
        "none" -> NONE
        "starting" -> STARTING
        "started" -> STARTED
        "stopping" -> STOPPING
        "stopped" -> STOPPED
        "errored" -> ERRORED
        else -> NONE
      }
    }
  }

  fun toMap(): Map<String, String> {
    val strState =
      when (this) {
        NONE -> "none"
        STARTING -> "stopping"
        STARTED -> "started"
        STOPPING -> "stopping"
        STOPPED -> "stopped"
        ERRORED -> "errored"
      }

    val map: HashMap<String, String> = HashMap()

    map["state"] = strState

    return map;
  }
}

data class DyteLivestreamData(
  val liveStreamUrl: String?,
  val roomName: String,
  val state: LiveStreamState,
  val viewerCount: Int,
) {
  companion object {
    fun fromMap(map: Map<String, Any?>): DyteLivestreamData {

      val lvsUrl = map["livestreamUrl"] as String
      val roomName = map["roomName"] as String
      val viewerCount = map["viewerCount"] as Int
      val state = LiveStreamState.fromMap(map["state"] as String)

      return DyteLivestreamData(lvsUrl, roomName, state, viewerCount)
    }
  }

  fun toMap(): Map<String, Any?> {
    val map = HashMap<String, Any?>()
    map["livestreamUrl"] = liveStreamUrl
    map["roomName"] = roomName
    map["viewerCount"] = viewerCount
    map["state"] = state.toMap()["state"]
    return map;
  }
}

data class LiveStreamStageRequestPeer(
  val displayName: String,
  val userId: String,
  val peerId: String
) {
  companion object {
    fun fromMap(map: Map<String, Any?>): LiveStreamStageRequestPeer {

      val displayName = map["displayName"] as String
      val userId = map["userId"] as String
      val peerId = map["peerId"] as String

      return LiveStreamStageRequestPeer(displayName, userId, peerId)
    }
  }

  fun toMap(): Map<String, Any?> {
    val map = HashMap<String, Any?>()
    map["displayName"] = displayName
    map["userId"] = userId
    map["peerId"] = peerId
    return map
  }
}

data class LiveStreamStagePeer(
  val displayName: String,
  val userId: String,
  val peerId: String
) {
  companion object {
    fun fromMap(map: Map<String, Any?>): LiveStreamStagePeer {

      val displayName = map["displayName"] as String
      val userId = map["userId"] as String
      val peerId = map["peerId"] as String

      return LiveStreamStagePeer(displayName, userId, peerId)
    }
  }

  fun toMap(): Map<String, Any?> {
    val map = HashMap<String, Any?>()
    map["displayName"] = displayName
    map["userId"] = userId
    map["peerId"] = peerId
    return map
  }
}