package io.dyte.core.controllers

import io.dyte.core.controllers.DyteEventType.OnActiveParticipantsChanged
import io.dyte.core.controllers.DyteEventType.OnActiveSpeakerChanged
import io.dyte.core.controllers.DyteEventType.OnAudioDevicesChanged
import io.dyte.core.controllers.DyteEventType.OnCameraClosed
import io.dyte.core.controllers.DyteEventType.OnCameraDisconnected
import io.dyte.core.controllers.DyteEventType.OnCameraError
import io.dyte.core.controllers.DyteEventType.OnCameraFreezed
import io.dyte.core.controllers.DyteEventType.OnCameraOpening
import io.dyte.core.controllers.DyteEventType.OnConnectedToMeetingRoom
import io.dyte.core.controllers.DyteEventType.OnConnectingToMeetingRoom
import io.dyte.core.controllers.DyteEventType.OnDisconnectedFromMeetingRoom
import io.dyte.core.controllers.DyteEventType.OnFirstFrameAvailable
import io.dyte.core.controllers.DyteEventType.OnHostKicked
import io.dyte.core.controllers.DyteEventType.OnLiveStreamEnded
import io.dyte.core.controllers.DyteEventType.OnLiveStreamEnding
import io.dyte.core.controllers.DyteEventType.OnLiveStreamErrored
import io.dyte.core.controllers.DyteEventType.OnLiveStreamStageRequestsUpdated
import io.dyte.core.controllers.DyteEventType.OnLiveStreamStarted
import io.dyte.core.controllers.DyteEventType.OnLiveStreamStarting
import io.dyte.core.controllers.DyteEventType.OnLivestreamUpdate
import io.dyte.core.controllers.DyteEventType.OnMeetingMessagesReceived
import io.dyte.core.controllers.DyteEventType.OnMeetingPollsReceived
import io.dyte.core.controllers.DyteEventType.OnMeetingRecordingError
import io.dyte.core.controllers.DyteEventType.OnMeetingRecordingStarted
import io.dyte.core.controllers.DyteEventType.OnMeetingRecordingStateUpdate
import io.dyte.core.controllers.DyteEventType.OnMeetingRecordingStopped
import io.dyte.core.controllers.DyteEventType.OnMeetingRoomConnectionFailed
import io.dyte.core.controllers.DyteEventType.OnMeetingRoomDisconnected
import io.dyte.core.controllers.DyteEventType.OnMeetingRoomInitCompleted
import io.dyte.core.controllers.DyteEventType.OnMeetingRoomInitFailed
import io.dyte.core.controllers.DyteEventType.OnMeetingRoomInitStarted
import io.dyte.core.controllers.DyteEventType.OnMeetingRoomJoinFailed
import io.dyte.core.controllers.DyteEventType.OnMeetingRoomJoinStarted
import io.dyte.core.controllers.DyteEventType.OnMeetingRoomJoined
import io.dyte.core.controllers.DyteEventType.OnMeetingRoomJoinedWithoutCameraPermission
import io.dyte.core.controllers.DyteEventType.OnMeetingRoomJoinedWithoutMicPermission
import io.dyte.core.controllers.DyteEventType.OnMeetingRoomLeave
import io.dyte.core.controllers.DyteEventType.OnMeetingRoomLeaveStarted
import io.dyte.core.controllers.DyteEventType.OnMeetingRoomReconnectionFailed
import io.dyte.core.controllers.DyteEventType.OnMeetingStateChanged
import io.dyte.core.controllers.DyteEventType.OnMetaUpdate
import io.dyte.core.controllers.DyteEventType.OnMicrophoneError
import io.dyte.core.controllers.DyteEventType.OnMicrophoneInitError
import io.dyte.core.controllers.DyteEventType.OnMicrophoneStartError
import io.dyte.core.controllers.DyteEventType.OnNewMeetingMessageReceived
import io.dyte.core.controllers.DyteEventType.OnNewMeetingPollReceived
import io.dyte.core.controllers.DyteEventType.OnNoActiveSpeaker
import io.dyte.core.controllers.DyteEventType.OnParticipantsUpdate
import io.dyte.core.controllers.DyteEventType.OnPeerAudioUpdate
import io.dyte.core.controllers.DyteEventType.OnPeerJoin
import io.dyte.core.controllers.DyteEventType.OnPeerLeft
import io.dyte.core.controllers.DyteEventType.OnPeerPinned
import io.dyte.core.controllers.DyteEventType.OnPeerScreenShareEnded
import io.dyte.core.controllers.DyteEventType.OnPeerScreenShareStarted
import io.dyte.core.controllers.DyteEventType.OnPeerUnpinned
import io.dyte.core.controllers.DyteEventType.OnPeerUpdate
import io.dyte.core.controllers.DyteEventType.OnPeerVideoUpdate
import io.dyte.core.controllers.DyteEventType.OnPluginDisabled
import io.dyte.core.controllers.DyteEventType.OnPluginEnabled
import io.dyte.core.controllers.DyteEventType.OnPluginFileRequest
import io.dyte.core.controllers.DyteEventType.OnPluginMessage
import io.dyte.core.controllers.DyteEventType.OnProximityChanged
import io.dyte.core.controllers.DyteEventType.OnReconnectedToMeetingRoom
import io.dyte.core.controllers.DyteEventType.OnReconnectingToMeetingRoom
import io.dyte.core.controllers.DyteEventType.OnRoomMessage
import io.dyte.core.controllers.DyteEventType.OnSelfAudioUpdate
import io.dyte.core.controllers.DyteEventType.OnSelfUpdate
import io.dyte.core.controllers.DyteEventType.OnSelfVideoUpdate
import io.dyte.core.controllers.DyteEventType.OnSelfWaitListStatusUpdate
import io.dyte.core.controllers.DyteEventType.OnStageCountUpdated
import io.dyte.core.controllers.DyteEventType.OnViewerCountUpdated
import io.dyte.core.controllers.DyteEventType.OnWaitListPeerAccepted
import io.dyte.core.controllers.DyteEventType.OnWaitListPeerClosed
import io.dyte.core.controllers.DyteEventType.OnWaitListPeerJoined
import io.dyte.core.controllers.DyteEventType.OnWaitListPeerRejected
import io.dyte.core.controllers.DyteEventType.OnWebinarPresentRequest
import io.dyte.core.controllers.DyteEventType.OnWebinarPresentRequestAccepted
import io.dyte.core.controllers.DyteEventType.OnWebinarPresentRequestAdded
import io.dyte.core.controllers.DyteEventType.OnWebinarPresentRequestClosed
import io.dyte.core.controllers.DyteEventType.OnWebinarPresentRequestRejected
import io.dyte.core.controllers.DyteEventType.OnWebinarPresentRequestWithdrawn
import io.dyte.core.controllers.DyteEventType.OnWebinarStartedPresenting
import io.dyte.core.controllers.DyteEventType.OnWebinarStoppedPresenting
import io.dyte.core.controllers.LiveStreamState.ERRORED
import io.dyte.core.controllers.LiveStreamState.STARTED
import io.dyte.core.controllers.LiveStreamState.STOPPED
import io.dyte.core.feat.*
import io.dyte.core.listeners.DyteCameraEventsListener
import io.dyte.core.listeners.DyteChatEventsListener
import io.dyte.core.listeners.DyteDataUpdateListener
import io.dyte.core.listeners.DyteLiveStreamEventsListener
import io.dyte.core.listeners.DyteMeetingRoomEventsListener
import io.dyte.core.listeners.DyteMicrophoneEventsListener
import io.dyte.core.listeners.DyteParticipantEventsListener
import io.dyte.core.listeners.DyteParticipantUpdateListener
import io.dyte.core.listeners.DytePermissionEventsListener
import io.dyte.core.listeners.DytePluginEventsListener
import io.dyte.core.listeners.DytePollEventsListener
import io.dyte.core.listeners.DyteRecordingEventsListener
import io.dyte.core.listeners.DyteSelfEventsListener
import io.dyte.core.listeners.DyteWaitlistEventsListener
import io.dyte.core.listeners.DyteWebinarEventsListener
import io.dyte.core.models.DyteJoinedMeetingParticipant
import io.dyte.core.models.DyteRoomParticipants
import io.dyte.core.models.WaitListStatus
import io.dyte.core.models.*
import io.dyte.core.platform.IDyteMediaSoupUtils
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import kotlinx.serialization.json.JsonObject

internal class EventController(controllerContainer: IControllerContainer) :
  BaseController(controllerContainer), IEventController {
  private val selfEventsListeners = arrayListOf<DyteSelfEventsListener>()
  private val roomEventsListeners = arrayListOf<DyteMeetingRoomEventsListener>()
  private val participantEventsListeners = arrayListOf<DyteParticipantEventsListener>()
  private val cameraEventsListeners = arrayListOf<DyteCameraEventsListener>()
  private val microphoneEventsListeners = arrayListOf<DyteMicrophoneEventsListener>()
  private val permissionEventsListeners = arrayListOf<DytePermissionEventsListener>()
  private val pluginEventsListeners = arrayListOf<DytePluginEventsListener>()
  private val liveStreamEventsListeners = arrayListOf<DyteLiveStreamEventsListener>()
  private val waitlistEventsListeners = arrayListOf<DyteWaitlistEventsListener>()
  private val pollEventsListeners = arrayListOf<DytePollEventsListener>()
  private val recordingEventsListeners = arrayListOf<DyteRecordingEventsListener>()
  private val webinarEventsListeners = arrayListOf<DyteWebinarEventsListener>()
  private val chatEventsListeners = arrayListOf<DyteChatEventsListener>()
  private val dataUpdateListeners = arrayListOf<DyteDataUpdateListener>()

  private val participantUpdateListeners =
    hashMapOf<String, ArrayList<DyteParticipantUpdateListener>>()

  override fun triggerEvent(eventType: DyteEventType) {
    serialScope.launch {
      withContext(Dispatchers.Main) {
        triggerEventInternal(eventType)
      }
    }
  }

  private fun triggerEventInternal(eventType: DyteEventType) {
    when (eventType) {
      OnMeetingRoomInitStarted -> {
        ArrayList(roomEventsListeners).forEach { roomEventsListener ->
          roomEventsListener.onMeetingInitStarted()
        }
      }

      OnMeetingRoomInitCompleted -> {
        ArrayList(dataUpdateListeners).forEach { dataUpdateListener ->
          dataUpdateListener.onSelfPermissionsUpdate(controllerContainer.selfController.getSelf().permissions)
        }
        ArrayList(dataUpdateListeners).forEach { dataUpdateListener ->
          dataUpdateListener.onPluginsUpdates(controllerContainer.pluginsController.activePlugins)
        }
        ArrayList(roomEventsListeners).forEach { roomEventsListener ->
          roomEventsListener.onMeetingInitCompleted()
        }
        ArrayList(selfEventsListeners).forEach { selfEventsListener ->
          selfEventsListener.onUpdate(controllerContainer.selfController.getSelf())
        }
      }

      is OnMeetingRoomInitFailed -> {
        ArrayList(roomEventsListeners).forEach { roomEventsListener ->
          roomEventsListener.onMeetingInitFailed(eventType.exception)
        }
      }

      OnMeetingRoomLeaveStarted -> {
        ArrayList(roomEventsListeners).forEach { roomEventsListener ->
          roomEventsListener.onMeetingRoomLeaveStarted()
        }
      }

      OnMeetingRoomLeave -> {
        ArrayList(roomEventsListeners).forEach { roomEventsListener ->
          roomEventsListener.onMeetingRoomLeaveCompleted()
        }
      }

      is OnMeetingStateChanged -> {
        controllerContainer.connectionController.onSocketStateChanged(eventType.state)
      }

      OnMeetingRoomJoinStarted -> {
        ArrayList(roomEventsListeners).forEach { roomEventsListener ->
          roomEventsListener.onMeetingRoomJoinStarted()
        }
      }

      is OnMeetingRoomJoined -> {
        ArrayList(roomEventsListeners).forEach { roomEventsListener ->
          roomEventsListener.onMeetingRoomJoinCompleted()
        }
        ArrayList(selfEventsListeners).forEach { selfEventsListener ->
          selfEventsListener.onUpdate(controllerContainer.selfController.getSelf())
        }
        ArrayList(participantEventsListeners).forEach { participantEventsListener ->
          participantEventsListener.onParticipantJoin(eventType.participant)
          participantEventsListener.onUpdate(controllerContainer.participantController.meetingRoomParticipants)
        }

        ArrayList(dataUpdateListeners).forEach { dataUpdateListener ->
          dataUpdateListener.onSelfPermissionsUpdate(controllerContainer.selfController.getSelf().permissions)
        }
      }

      OnMeetingRoomJoinedWithoutMicPermission -> {
        ArrayList(selfEventsListeners).forEach { selfEventsListener ->
          selfEventsListener.onMeetingRoomJoinedWithoutMicPermission()
        }
      }

      OnMeetingRoomJoinedWithoutCameraPermission -> {
        ArrayList(selfEventsListeners).forEach { selfEventsListener ->
          selfEventsListener.onMeetingRoomJoinedWithoutCameraPermission()
        }
      }

      is OnMeetingRoomJoinFailed -> {
        ArrayList(roomEventsListeners).forEach { roomEventsListener ->
          roomEventsListener.onMeetingRoomJoinFailed(eventType.exception)
        }
      }

      is OnSelfAudioUpdate -> {
        val updateListeners =
          participantUpdateListeners[controllerContainer.selfController.getSelf().id]
        updateListeners?.let {
          ArrayList(updateListeners).forEach { updateListener ->
            updateListener.onAudioUpdate(
              controllerContainer.selfController.getSelf().audioEnabled
            )
            updateListener.onUpdate(controllerContainer.selfController.getSelf())
          }
        }
        ArrayList(selfEventsListeners).forEach { selfEventsListener ->
          selfEventsListener.onAudioUpdate(
            controllerContainer.selfController.getSelf().audioEnabled
          )
          selfEventsListener.onUpdate(controllerContainer.selfController.getSelf())
        }
        ArrayList(participantEventsListeners).forEach { participantEventListener ->
          participantEventListener.onAudioUpdate(controllerContainer.selfController.getSelf().audioEnabled, controllerContainer.selfController.getSelf())
          participantEventListener.onUpdate(controllerContainer.participantController.meetingRoomParticipants)
        }
      }

      is OnSelfVideoUpdate -> {
        val updateListeners =
          participantUpdateListeners[controllerContainer.selfController.getSelf().id]
        updateListeners?.let {
          ArrayList(updateListeners).forEach { updateListener ->
            updateListener.onVideoUpdate(
              controllerContainer.selfController.getSelf().videoEnabled
            )
            updateListener.onUpdate(controllerContainer.selfController.getSelf())
          }
        }
        ArrayList(selfEventsListeners).forEach { selfEventsListener ->
          selfEventsListener.onVideoUpdate(
            controllerContainer.selfController.getSelf().videoEnabled
          )
          selfEventsListener.onUpdate(controllerContainer.selfController.getSelf())
        }
        ArrayList(participantEventsListeners).forEach { participantEventListener ->
          participantEventListener.onUpdate(controllerContainer.participantController.meetingRoomParticipants)
          participantEventListener.onVideoUpdate(controllerContainer.selfController.getSelf().videoEnabled, controllerContainer.selfController.getSelf())
        }
      }

      is OnParticipantsUpdate -> {
        ArrayList(participantEventsListeners).forEach { participantEventsListener ->
          participantEventsListener.onUpdate(eventType.participant)
        }
      }

      OnSelfUpdate -> {
        ArrayList(selfEventsListeners).forEach { selfEventsListener ->
          selfEventsListener.onUpdate(controllerContainer.selfController.getSelf())
        }
      }

      is OnPeerJoin -> {
        ArrayList(participantEventsListeners).forEach { participantEventsListener ->
          participantEventsListener.onParticipantJoin(eventType.participant)
          participantEventsListener.onUpdate(controllerContainer.participantController.meetingRoomParticipants)
        }
      }

      is OnPeerLeft -> {
        ArrayList(participantEventsListeners).forEach { participantEventsListener ->
          participantEventsListener.onParticipantLeave(eventType.participant)
          participantEventsListener.onUpdate(controllerContainer.participantController.meetingRoomParticipants)
        }
      }

      is OnPeerAudioUpdate -> {
        val updateListeners = participantUpdateListeners[eventType.participant.id]
        updateListeners?.let {
          ArrayList(updateListeners).forEach { updateListener ->
            updateListener.onAudioUpdate(eventType.participant.audioEnabled)
            updateListener.onUpdate(eventType.participant)
          }
        }
        ArrayList(participantEventsListeners).forEach { participantEventsListener ->
          participantEventsListener.onAudioUpdate(
            eventType.participant.audioEnabled,
            eventType.participant
          )
          participantEventsListener.onUpdate(controllerContainer.participantController.meetingRoomParticipants)
        }
      }

      is OnPeerVideoUpdate -> {
        val updateListeners = participantUpdateListeners[eventType.participant.id]
        updateListeners?.let {
          ArrayList(updateListeners).forEach { updateListener ->
            updateListener.onVideoUpdate(eventType.participant.videoEnabled)
            updateListener.onUpdate(eventType.participant)
          }
        }
        ArrayList(participantEventsListeners).forEach { participantEventsListener ->
          participantEventsListener.onVideoUpdate(
            eventType.participant.videoEnabled,
            eventType.participant
          )
          participantEventsListener.onUpdate(controllerContainer.participantController.meetingRoomParticipants)
        }
      }

      is OnPeerScreenShareStarted -> {
        val updateListeners = participantUpdateListeners[eventType.participant.id]
        updateListeners?.let {
          ArrayList(updateListeners).forEach { updateListener ->
            updateListener.onScreenShareStarted()
          }
        }
        ArrayList(participantEventsListeners).forEach { participantEventsListener ->
          participantEventsListener.onScreenShareStarted(eventType.participant)
          participantEventsListener.onScreenSharesUpdated()
        }
        ArrayList(dataUpdateListeners).forEach { dataUpdateListener ->
          dataUpdateListener.onScreenShareUpdate(controllerContainer.participantController.meetingRoomParticipants.screenshares)
        }
      }

      is OnPeerScreenShareEnded -> {
        val updateListeners = participantUpdateListeners[eventType.participant.id]
        updateListeners?.let {
          ArrayList(updateListeners).forEach { updateListener ->
            updateListener.onScreenShareEnded()
          }
        }
        ArrayList(participantEventsListeners).forEach { participantEventsListener ->
          participantEventsListener.onScreenShareEnded(eventType.participant)
          participantEventsListener.onScreenSharesUpdated()
        }
        ArrayList(dataUpdateListeners).forEach { dataUpdateListener ->
          dataUpdateListener.onScreenShareUpdate(controllerContainer.participantController.meetingRoomParticipants.screenshares)
        }
      }

      is OnPeerUpdate -> {
        val updateListeners = participantUpdateListeners[eventType.participant.id]
        updateListeners?.let {
          ArrayList(updateListeners).forEach { updateListener ->
            updateListener.onUpdate(eventType.participant)
          }
        }
        ArrayList(participantEventsListeners).forEach { participantEventsListener ->
          participantEventsListener.onUpdate(controllerContainer.participantController.meetingRoomParticipants)
        }
      }

      is OnActiveParticipantsChanged -> {
        ArrayList(participantEventsListeners).forEach { dyteParticipantEventsListener ->
          dyteParticipantEventsListener.onActiveParticipantsChanged(eventType.activeParticipants)
          dyteParticipantEventsListener.onUpdate(controllerContainer.participantController.meetingRoomParticipants)
        }
      }

      is OnActiveSpeakerChanged -> {
        ArrayList(participantEventsListeners).forEach { participantEventsListener ->
          participantEventsListener.onActiveSpeakerChanged(eventType.participant)
          participantEventsListener.onUpdate(controllerContainer.participantController.meetingRoomParticipants)
        }
      }

      OnNoActiveSpeaker -> {
        ArrayList(participantEventsListeners).forEach { participantEventsListener ->
          participantEventsListener.onNoActiveSpeaker()
          participantEventsListener.onUpdate(controllerContainer.participantController.meetingRoomParticipants)
        }
      }

      is OnPeerPinned -> {
        val updateListeners = participantUpdateListeners[eventType.participant.id]
        updateListeners?.let {
          ArrayList(updateListeners).forEach { updateListener ->
            updateListener.onPinned()
          }
        }
        ArrayList(participantEventsListeners).forEach { participantEventsListener ->
          participantEventsListener.onParticipantPinned(eventType.participant)
          participantEventsListener.onUpdate(controllerContainer.participantController.meetingRoomParticipants)
        }
      }

      is OnPeerUnpinned -> {
        val updateListeners = participantUpdateListeners[eventType.participant.id]
        updateListeners?.let {
          ArrayList(updateListeners).forEach { updateListener ->
            updateListener.onUnpinned()
          }
        }
        ArrayList(participantEventsListeners).forEach { participantEventsListener ->
          participantEventsListener.onParticipantUnpinned(eventType.participant)
          participantEventsListener.onUpdate(controllerContainer.participantController.meetingRoomParticipants)
        }
      }

      OnMeetingRecordingStarted -> {
        ArrayList(recordingEventsListeners).forEach { recordingEventsListener ->
          recordingEventsListener.onMeetingRecordingStarted()
        }
      }

      OnMeetingRecordingStopped -> {
        ArrayList(recordingEventsListeners).forEach { recordingEventsListener ->
          recordingEventsListener.onMeetingRecordingEnded()
        }
      }

      is OnMeetingRecordingError -> {
        ArrayList(recordingEventsListeners).forEach { recordingEventsListener ->
          recordingEventsListener.onMeetingRecordingStopError(eventType.error)
        }
      }

      is OnMeetingRecordingStateUpdate -> {
        ArrayList(recordingEventsListeners).forEach { recordingEventsListener ->
          recordingEventsListener.onMeetingRecordingStateUpdated(eventType.state)
        }
      }

      is OnMeetingMessagesReceived -> {
        ArrayList(chatEventsListeners).forEach { chatEventListener ->
          chatEventListener.onChatUpdates(eventType.messages)
        }
      }

      is OnNewMeetingMessageReceived -> {
        ArrayList(chatEventsListeners).forEach { chatEventListener ->
          chatEventListener.onNewChatMessage(eventType.message)
          chatEventListener.onChatUpdates(controllerContainer.chatController.messages)
        }
      }

      is OnMeetingPollsReceived -> {
        ArrayList(pollEventsListeners).forEach { pollEventsListeners ->
          pollEventsListeners.onPollUpdates(eventType.polls)
        }
      }

      is OnNewMeetingPollReceived -> {
        ArrayList(pollEventsListeners).forEach { pollEventsListeners ->
          pollEventsListeners.onNewPoll(
            eventType.poll
          )
          pollEventsListeners.onPollUpdates(controllerContainer.pollsController.polls)
        }
      }

      OnCameraClosed -> {
        // ArrayList(cameraEventsListeners).forEach { cameraEventsListener ->
        //   controllerContainer.platformUtilsProvider.getMediaSoupUtils().onCameraStreamKilled()
        //   controllerContainer.peerConnectionUtils.onCameraClosed()
        //   cameraEventsListener.onCameraClosed()
        // }
      }

      OnCameraDisconnected -> {
        ArrayList(cameraEventsListeners).forEach { cameraEventsListener ->
          (controllerContainer.sfuUtils as? IDyteMediaSoupUtils)?.onCameraStreamKilled()
          controllerContainer.platformUtilsProvider.getMediaUtils().onCameraClosed()
          cameraEventsListener.onCameraDisconnected()
        }
      }

      is OnCameraError -> {
        ArrayList(cameraEventsListeners).forEach { cameraEventsListener ->
          (controllerContainer.sfuUtils as? IDyteMediaSoupUtils)?.onCameraStreamKilled()
          controllerContainer.platformUtilsProvider.getMediaUtils().onCameraClosed()
          cameraEventsListener.onCameraError(eventType.message)
        }
      }

      is OnCameraFreezed -> {
        ArrayList(cameraEventsListeners).forEach { cameraEventsListener ->
          cameraEventsListener.onCameraFreezed(eventType.message)
        }
      }

      is OnCameraOpening -> {
        ArrayList(cameraEventsListeners).forEach { cameraEventsListener ->
          cameraEventsListener.onCameraOpening(eventType.message)
        }
      }

      OnFirstFrameAvailable -> {
        ArrayList(cameraEventsListeners).forEach { cameraEventsListener ->
          cameraEventsListener.onFirstFrameAvailable()
        }
      }

      is OnMicrophoneInitError -> {
        ArrayList(microphoneEventsListeners).forEach { microphoneEventsListener ->
          microphoneEventsListener.onAudioTrackInitError(eventType.message)
        }
      }

      is OnMicrophoneStartError -> {
        ArrayList(microphoneEventsListeners).forEach { microphoneEventsListener ->
          microphoneEventsListener.onAudioTrackStartError(eventType.message)
        }
      }

      is OnMicrophoneError -> {
        ArrayList(microphoneEventsListeners).forEach { microphoneEventsListener ->
          microphoneEventsListener.onAudioTrackError(eventType.message)
        }
      }

      OnAudioDevicesChanged -> {
        ArrayList(selfEventsListeners).forEach { selfEventsListener ->
          selfEventsListener.onAudioDevicesUpdated()
        }
      }

      is OnProximityChanged -> {
        ArrayList(selfEventsListeners).forEach { selfEventsListener ->
          selfEventsListener.onProximityChanged(eventType.isNear)
        }
      }

      OnMeetingRoomDisconnected -> {
        ArrayList(roomEventsListeners).forEach { roomEventsListener ->
          roomEventsListener.onMeetingRoomDisconnected()
        }
      }

      is OnPluginEnabled -> {
        pluginEventsListeners.forEach { pluginEventsListener ->
          pluginEventsListener.onPluginActivated(eventType.plugin)
        }
        dataUpdateListeners.forEach {
          it.onPluginsUpdates(controllerContainer.pluginsController.activePlugins)
        }
      }

      is OnPluginDisabled -> {
        pluginEventsListeners.forEach { pluginEventsListener ->
          pluginEventsListener.onPluginDeactivated(eventType.plugin)
        }
        dataUpdateListeners.forEach {
          it.onPluginsUpdates(controllerContainer.pluginsController.activePlugins)
        }
      }

      is OnPluginMessage -> {
        pluginEventsListeners.forEach { pluginEventsListener ->
          pluginEventsListener.onPluginMessage(eventType.message)
        }
      }

      is OnPluginFileRequest -> {
        pluginEventsListeners.forEach { pluginEventsListener ->
          pluginEventsListener.onPluginFileRequest(eventType.plugin)
        }
      }

      OnHostKicked -> {
        ArrayList(selfEventsListeners).forEach { selfEventsListener ->
          selfEventsListener.onRemovedFromMeeting()
        }
      }

      is OnWaitListPeerJoined -> {
        ArrayList(waitlistEventsListeners).forEach { waitlistEventListener ->
          waitlistEventListener.onWaitListParticipantJoined(eventType.participant)
        }
        ArrayList(participantEventsListeners).forEach { participantEventListener ->
          participantEventListener.onUpdate(controllerContainer.participantController.meetingRoomParticipants)
        }
      }

      is OnWaitListPeerAccepted -> {
        ArrayList(waitlistEventsListeners).forEach { waitlistEventListener ->
          waitlistEventListener.onWaitListParticipantAccepted(eventType.participant)
        }
        ArrayList(participantEventsListeners).forEach { participantEventListener ->
          participantEventListener.onUpdate(controllerContainer.participantController.meetingRoomParticipants)
        }
      }

      is OnWaitListPeerClosed -> {
        ArrayList(waitlistEventsListeners).forEach { waitlistEventListener ->
          waitlistEventListener.onWaitListParticipantClosed(eventType.participant)
        }
        ArrayList(participantEventsListeners).forEach { participantEventListener ->
          participantEventListener.onUpdate(controllerContainer.participantController.meetingRoomParticipants)
        }
      }

      is OnWaitListPeerRejected -> {
        ArrayList(waitlistEventsListeners).forEach { waitlistEventListener ->
          waitlistEventListener.onWaitListParticipantRejected(eventType.participant)
        }
        ArrayList(participantEventsListeners).forEach { participantEventListener ->
          participantEventListener.onUpdate(controllerContainer.participantController.meetingRoomParticipants)
        }
      }

      is OnSelfWaitListStatusUpdate -> {
        ArrayList(selfEventsListeners).forEach { selfEventsListener ->
          selfEventsListener.onWaitListStatusUpdate(controllerContainer.selfController.getSelf().waitListStatus)
          selfEventsListener.onUpdate(controllerContainer.selfController.getSelf())
        }
      }

      is DyteEventType.OnSelfStageStatusUpdate -> {
        ArrayList(selfEventsListeners).forEach { selfEventsListener ->
          selfEventsListener.onStageStatusUpdated(controllerContainer.stageController.stageStatus)
          selfEventsListener.onUpdate(controllerContainer.selfController.getSelf())
        }
      }

      OnWebinarPresentRequest -> {
        ArrayList(webinarEventsListeners).forEach { webinarEventListener ->
          webinarEventListener.onPresentRequestReceived()
        }
      }

      OnWebinarStoppedPresenting -> {
        ArrayList(webinarEventsListeners).forEach { webinarEventListener ->
          webinarEventListener.onRemovedFromStage()
        }
      }

      OnWebinarStartedPresenting -> {
        ArrayList(webinarEventsListeners).forEach { webinarEventListener ->
          webinarEventListener.onAddedToStage()
        }
      }

      is OnWebinarPresentRequestAdded -> {
        ArrayList(webinarEventsListeners).forEach { webinarEventListener ->
          webinarEventListener.onPresetRequestAdded(eventType.presentParticipant)
        }
      }

      is OnWebinarPresentRequestClosed -> {
        ArrayList(webinarEventsListeners).forEach { webinarEventListener ->
          webinarEventListener.onPresentRequestClosed(eventType.presentParticipant)
        }
      }

      is OnWebinarPresentRequestAccepted -> {
        ArrayList(webinarEventsListeners).forEach { webinarEventListener ->
          webinarEventListener.onPresetRequestAccepted(eventType.presentParticipant)
        }
      }

      is OnWebinarPresentRequestRejected -> {
        ArrayList(webinarEventsListeners).forEach { webinarEventListener ->
          webinarEventListener.onPresetRequestRejected(eventType.presentParticipant)
        }
      }

      is OnWebinarPresentRequestWithdrawn -> {
        ArrayList(webinarEventsListeners).forEach { webinarEventListener ->
          webinarEventListener.onPresetRequestWithdrawn(eventType.presentParticipant)
        }
      }

      OnLiveStreamStarting -> {
        ArrayList(liveStreamEventsListeners).forEach { liveStreamEventsListener ->
          liveStreamEventsListener.onLiveStreamStarting()
        }
      }

      OnLiveStreamStarted -> {
        ArrayList(liveStreamEventsListeners).forEach { liveStreamEventsListener -> liveStreamEventsListener.onLiveStreamStateUpdate(controllerContainer.liveStreamController.livestreamData) }
        ArrayList(liveStreamEventsListeners).forEach { liveStreamEventsListener ->
          liveStreamEventsListener.onLiveStreamStarted()
        }
      }

      OnLiveStreamEnding -> {
        ArrayList(liveStreamEventsListeners).forEach { liveStreamEventsListener ->
          liveStreamEventsListener.onLiveStreamEnding()
        }
      }

      OnLiveStreamEnded -> {
        ArrayList(liveStreamEventsListeners).forEach { liveStreamEventsListener -> liveStreamEventsListener.onLiveStreamStateUpdate(controllerContainer.liveStreamController.livestreamData) }
        ArrayList(liveStreamEventsListeners).forEach { liveStreamEventsListener ->
          liveStreamEventsListener.onLiveStreamEnded()
        }
      }

      OnLiveStreamErrored -> {
        ArrayList(liveStreamEventsListeners).forEach { liveStreamEventsListener -> liveStreamEventsListener.onLiveStreamStateUpdate(controllerContainer.liveStreamController.livestreamData) }

        ArrayList(liveStreamEventsListeners).forEach { liveStreamEventsListener ->
          liveStreamEventsListener.onLiveStreamErrored()
        }
      }

      is OnViewerCountUpdated -> {
        ArrayList(liveStreamEventsListeners).forEach { liveStreamEventsListener ->
          liveStreamEventsListener.onViewerCountUpdated(eventType.count)
        }
      }

      is OnStageCountUpdated -> {
        ArrayList(liveStreamEventsListeners).forEach { liveStreamEventsListener ->
          liveStreamEventsListener.onStageCountUpdated(eventType.count)
        }
      }

      /*is OnLiveStreamPresentRequestAccepted -> {
        ArrayList(dataUpdateListeners).forEach { dataEventsListener ->
          dataEventsListener.onLivestreamUpdate(controllerContainer.liveStreamController.livestreamData)
        }
        ArrayList(liveStreamEventsListeners).forEach { liveStreamEventsListener ->
          liveStreamEventsListener.onJoinRequestAccepted(eventType.peer)
        }
      }

      is OnLiveStreamPresentRequestRejected -> {
        ArrayList(dataUpdateListeners).forEach { dataEventsListener ->
          dataEventsListener.onLivestreamUpdate(controllerContainer.liveStreamController.livestreamData)
        }
        ArrayList(liveStreamEventsListeners).forEach { liveStreamEventsListener ->
          liveStreamEventsListener.onJoinRequestRejected(eventType.peer)
        }
      }*/

      is OnLiveStreamStageRequestsUpdated -> {
        ArrayList(liveStreamEventsListeners).forEach { liveStreamEventsListener ->
          liveStreamEventsListener.onStageRequestsUpdated(eventType.requests)
        }
      }
      OnLivestreamUpdate -> {
        ArrayList(dataUpdateListeners).forEach { dataUpdateListener -> dataUpdateListener.onLivestreamUpdate(
          controllerContainer.liveStreamController.livestreamData
        ) }
      }

      OnMetaUpdate -> {
        ArrayList(dataUpdateListeners).forEach { dataUpdateListener ->
          dataUpdateListener.onMetaUpdate(
            controllerContainer.metaController.getRoomName(),
            controllerContainer.metaController.getMeetingTitle(),
            controllerContainer.metaController.getMeetingStatedTimestamp(),
            controllerContainer.metaController.getMeetingType().name
          )
        }
      }

      OnConnectingToMeetingRoom -> {
        ArrayList(roomEventsListeners).forEach { roomEventsListener ->
          roomEventsListener.onConnectingToMeetingRoom()
        }
      }

      OnConnectedToMeetingRoom -> {
        ArrayList(roomEventsListeners).forEach { roomEventsListener ->
          roomEventsListener.onConnectedToMeetingRoom()
        }
      }

      OnDisconnectedFromMeetingRoom -> {
        ArrayList(roomEventsListeners).forEach { roomEventsListener ->
          roomEventsListener.onDisconnectedFromMeetingRoom()
        }
      }

      OnMeetingRoomConnectionFailed -> {
        ArrayList(roomEventsListeners).forEach { roomEventsListener ->
          roomEventsListener.onMeetingRoomConnectionFailed()
        }
      }

      OnReconnectingToMeetingRoom -> {
        ArrayList(roomEventsListeners).forEach { roomEventsListener ->
          roomEventsListener.onReconnectingToMeetingRoom()
        }
      }

      OnReconnectedToMeetingRoom -> {
        ArrayList(roomEventsListeners).forEach { roomEventsListener ->
          roomEventsListener.onReconnectedToMeetingRoom()
        }
      }

      OnMeetingRoomReconnectionFailed -> {
        ArrayList(roomEventsListeners).forEach { roomEventsListener ->
          roomEventsListener.onMeetingRoomReconnectionFailed()
        }
      }

      is OnRoomMessage -> {
        ArrayList(selfEventsListeners).forEach { selfEventListener ->
          selfEventListener.onRoomMessage(eventType.message)
        }
      }
    }
  }

  override fun addSelfEventListener(selfEventListener: DyteSelfEventsListener) {
    if (selfEventsListeners.contains(selfEventListener)) {
      return
    }
    this.selfEventsListeners.add(selfEventListener)
  }

  override fun removeSelfEventListener(selfEventListener: DyteSelfEventsListener) {
    this.selfEventsListeners.remove(selfEventListener)
  }

  override fun addRoomEventListener(roomEventsListener: DyteMeetingRoomEventsListener) {
    this.roomEventsListeners.add(roomEventsListener)
  }

  override fun removeRoomEventListener(roomEventsListener: DyteMeetingRoomEventsListener) {
    if (this.roomEventsListeners.contains(roomEventsListener)) {
      this.roomEventsListeners.remove(roomEventsListener)
    }
  }

  override fun addParticipantEventListener(participantEventsListener: DyteParticipantEventsListener) {
    this.participantEventsListeners.add(participantEventsListener)
  }

  override fun removeParticipantEventListener(participantEventsListener: DyteParticipantEventsListener) {
    if (this.participantEventsListeners.contains(participantEventsListener)) {
      this.participantEventsListeners.remove(participantEventsListener)
    }
  }

  override fun addCameraEventListener(cameraEventsListener: DyteCameraEventsListener) {
    this.cameraEventsListeners.add(cameraEventsListener)
  }

  override fun removeCameraEventListener(cameraEventsListener: DyteCameraEventsListener) {
    if (this.cameraEventsListeners.contains(cameraEventsListener)) {
      this.cameraEventsListeners.remove(cameraEventsListener)
    }
  }

  override fun addMicrophoneEventListener(microphoneEventsListener: DyteMicrophoneEventsListener) {
    this.microphoneEventsListeners.add(microphoneEventsListener)
  }

  override fun removeMicrophoneEventListener(microphoneEventsListener: DyteMicrophoneEventsListener) {
    if (this.microphoneEventsListeners.contains(microphoneEventsListener)) {
      this.microphoneEventsListeners.remove(microphoneEventsListener)
    }
  }

  override fun addPermissionEventListener(permissionEventsListener: DytePermissionEventsListener) {
    this.permissionEventsListeners.add(permissionEventsListener)
  }

  override fun removePermissionEventListener(permissionEventsListener: DytePermissionEventsListener) {
    if (this.permissionEventsListeners.contains(permissionEventsListener)) {
      this.permissionEventsListeners.remove(permissionEventsListener)
    }
  }

  override fun addPluginEventListener(pluginEventsListener: DytePluginEventsListener) {
    this.pluginEventsListeners.add(pluginEventsListener)
  }

  override fun removePluginEventListener(pluginEventsListener: DytePluginEventsListener) {
    if (this.pluginEventsListeners.contains(pluginEventsListener)) {
      this.pluginEventsListeners.remove(pluginEventsListener)
    }
  }

  override fun addLiveStreamEventListener(liveStreamEventsListener: DyteLiveStreamEventsListener) {
    this.liveStreamEventsListeners.add(liveStreamEventsListener)
  }

  override fun removeLiveStreamEventListener(liveStreamEventsListener: DyteLiveStreamEventsListener) {
    if (this.liveStreamEventsListeners.contains(liveStreamEventsListener)) {
      this.liveStreamEventsListeners.remove(liveStreamEventsListener)
    }
  }

  override fun addWaitlistEventListener(waitlistEventsListener: DyteWaitlistEventsListener) {
    if (this.waitlistEventsListeners.contains(waitlistEventsListener)) {
      return
    }

    this.waitlistEventsListeners.add(waitlistEventsListener)
  }

  override fun removeWaitlistEventListener(waitlistEventsListener: DyteWaitlistEventsListener) {
    if (this.waitlistEventsListeners.contains(waitlistEventsListener)) {
      waitlistEventsListeners.remove(waitlistEventsListener)
    }
  }

  override fun addWebinarEventListener(webinarEventsListener: DyteWebinarEventsListener) {
    if (this.webinarEventsListeners.contains(webinarEventsListener)) {
      return
    }

    this.webinarEventsListeners.add(webinarEventsListener)
  }

  override fun removeWebinarEventListener(webinarEventsListener: DyteWebinarEventsListener) {
    if (this.webinarEventsListeners.contains(webinarEventsListener)) {
      webinarEventsListeners.remove(webinarEventsListener)
    }
  }

  override fun addParticipantUpdateListener(
    dyteMeetingParticipant: DyteMeetingParticipant,
    dyteParticipantUpdateListener: DyteParticipantUpdateListener
  ) {
    val existingListeners: ArrayList<DyteParticipantUpdateListener>? =
      participantUpdateListeners[dyteMeetingParticipant.id]
    val updatedListeners = arrayListOf<DyteParticipantUpdateListener>()
    if (existingListeners == null) {
      updatedListeners.add(dyteParticipantUpdateListener)
    } else {
      updatedListeners.addAll(existingListeners)
      if (existingListeners.contains(dyteParticipantUpdateListener).not()) {
        updatedListeners.add(dyteParticipantUpdateListener)
      }
    }
    participantUpdateListeners[dyteMeetingParticipant.id] = updatedListeners
  }

  override fun removeParticipantUpdateListener(
    dyteMeetingParticipant: DyteMeetingParticipant,
    dyteParticipantUpdateListener: DyteParticipantUpdateListener
  ) {
    val existingListeners = participantUpdateListeners[dyteMeetingParticipant.id]
    existingListeners?.remove(dyteParticipantUpdateListener)
  }

  override fun removeParticipantUpdateListeners(dyteMeetingParticipant: DyteMeetingParticipant) {
    participantUpdateListeners[dyteMeetingParticipant.id] = arrayListOf()
  }

  override fun addChatEventsListener(chatEventsListener: DyteChatEventsListener) {
    if (chatEventsListeners.contains(chatEventsListener)) {
      return
    }

    chatEventsListeners.add(chatEventsListener)
  }

  override fun removeChatEventsListener(chatEventsListener: DyteChatEventsListener) {
    chatEventsListeners.remove(chatEventsListener)
  }

  override fun addPollEventsListener(pollEventsListener: DytePollEventsListener) {
    if (pollEventsListeners.contains(pollEventsListener)) {
      return
    }

    pollEventsListeners.add(pollEventsListener)
  }

  override fun removePollEventsListener(pollEventsListener: DytePollEventsListener) {
    pollEventsListeners.remove(pollEventsListener)
  }

  override fun addRecordingEventsListener(recordingEventsListener: DyteRecordingEventsListener) {
    if (recordingEventsListeners.contains(recordingEventsListener)) {
      return
    }

    recordingEventsListeners.add(recordingEventsListener)
  }

  override fun removeRecordingEventsListener(recordingEventsListener: DyteRecordingEventsListener) {
    recordingEventsListeners.remove(recordingEventsListener)
  }

  override fun addDataUpdateListener(dataUpdateListener: DyteDataUpdateListener) {
    dataUpdateListeners.add(dataUpdateListener)
  }

  override fun removeDataUpdateListener(dataUpdateListener: DyteDataUpdateListener) {
    dataUpdateListeners.remove(dataUpdateListener)
  }

  override fun dispose() {
    this.selfEventsListeners.clear()
    this.roomEventsListeners.clear()
    this.participantEventsListeners.clear()
    this.cameraEventsListeners.clear()
    this.microphoneEventsListeners.clear()
    this.permissionEventsListeners.clear()
    this.pluginEventsListeners.clear()
    this.participantUpdateListeners.clear()
  }
}

interface IEventController {
  fun addSelfEventListener(selfEventListener: DyteSelfEventsListener)
  fun removeSelfEventListener(selfEventListener: DyteSelfEventsListener)

  fun addRoomEventListener(roomEventsListener: DyteMeetingRoomEventsListener)
  fun removeRoomEventListener(roomEventsListener: DyteMeetingRoomEventsListener)

  fun addParticipantEventListener(participantEventsListener: DyteParticipantEventsListener)
  fun removeParticipantEventListener(participantEventsListener: DyteParticipantEventsListener)

  fun addCameraEventListener(cameraEventsListener: DyteCameraEventsListener)
  fun removeCameraEventListener(cameraEventsListener: DyteCameraEventsListener)

  fun addMicrophoneEventListener(microphoneEventsListener: DyteMicrophoneEventsListener)
  fun removeMicrophoneEventListener(microphoneEventsListener: DyteMicrophoneEventsListener)

  fun addPermissionEventListener(permissionEventsListener: DytePermissionEventsListener)
  fun removePermissionEventListener(permissionEventsListener: DytePermissionEventsListener)

  fun addPluginEventListener(pluginEventsListener: DytePluginEventsListener)
  fun removePluginEventListener(pluginEventsListener: DytePluginEventsListener)

  fun addLiveStreamEventListener(liveStreamEventsListener: DyteLiveStreamEventsListener)
  fun removeLiveStreamEventListener(liveStreamEventsListener: DyteLiveStreamEventsListener)
  fun addWaitlistEventListener(waitlistEventsListener: DyteWaitlistEventsListener)
  fun removeWaitlistEventListener(waitlistEventsListener: DyteWaitlistEventsListener)
  fun addWebinarEventListener(webinarEventsListener: DyteWebinarEventsListener)
  fun removeWebinarEventListener(webinarEventsListener: DyteWebinarEventsListener)
  fun addParticipantUpdateListener(
    dyteMeetingParticipant: DyteMeetingParticipant,
    dyteParticipantUpdateListener: DyteParticipantUpdateListener
  )

  fun removeParticipantUpdateListener(
    dyteMeetingParticipant: DyteMeetingParticipant,
    dyteParticipantUpdateListener: DyteParticipantUpdateListener
  )

  fun removeParticipantUpdateListeners(dyteMeetingParticipant: DyteMeetingParticipant)

  fun addChatEventsListener(chatEventsListener: DyteChatEventsListener)
  fun removeChatEventsListener(chatEventsListener: DyteChatEventsListener)

  fun addPollEventsListener(pollEventsListener: DytePollEventsListener)
  fun removePollEventsListener(pollEventsListener: DytePollEventsListener)

  fun addRecordingEventsListener(recordingEventsListener: DyteRecordingEventsListener)
  fun removeRecordingEventsListener(recordingEventsListener: DyteRecordingEventsListener)

  fun addDataUpdateListener(dataUpdateListener: DyteDataUpdateListener)
  fun removeDataUpdateListener(dataUpdateListener: DyteDataUpdateListener)
  fun dispose()

  fun triggerEvent(eventType: DyteEventType)
}

sealed class DyteEventType {
  object OnMeetingRoomInitStarted : DyteEventType()
  object OnMeetingRoomInitCompleted : DyteEventType()
  class OnMeetingRoomInitFailed(val exception: Exception) : DyteEventType()

  object OnMeetingRoomLeave : DyteEventType()
  object OnMeetingRoomLeaveStarted : DyteEventType()
  object OnMeetingRoomDisconnected : DyteEventType()

  object OnMeetingRoomJoinStarted : DyteEventType()
  class OnMeetingRoomJoined(val participant: DyteJoinedMeetingParticipant) : DyteEventType()
  class OnMeetingRoomJoinFailed(val exception: Exception) : DyteEventType()
  object OnMeetingRoomJoinedWithoutCameraPermission : DyteEventType()
  object OnMeetingRoomJoinedWithoutMicPermission : DyteEventType()

  class OnMeetingStateChanged(val state: String, val isReconnected: Boolean = false) :
    DyteEventType()

  // participant related event callbacks
  class OnPeerJoin(val participant: DyteJoinedMeetingParticipant) : DyteEventType()
  class OnPeerLeft(val participant: DyteJoinedMeetingParticipant) : DyteEventType()
  class OnPeerScreenShareStarted(val participant: DyteScreenShareMeetingParticipant) : DyteEventType()
  class OnPeerScreenShareEnded(val participant: DyteScreenShareMeetingParticipant) : DyteEventType()
  class OnPeerAudioUpdate(val  participant: DyteMeetingParticipant) : DyteEventType()
  class OnPeerVideoUpdate(val participant: DyteMeetingParticipant) : DyteEventType()
  class OnPeerPinned(val participant: DyteJoinedMeetingParticipant) : DyteEventType()
  class OnPeerUnpinned(val participant: DyteJoinedMeetingParticipant) : DyteEventType()
  class OnPeerUpdate(val participant: DyteJoinedMeetingParticipant) : DyteEventType()
  class OnActiveParticipantsChanged(val activeParticipants: List<DyteJoinedMeetingParticipant>) :
    DyteEventType()

  class OnParticipantsUpdate(val participant: DyteRoomParticipants) : DyteEventType()

  class OnWaitListPeerJoined(val participant: DyteWaitlistedParticipant) : DyteEventType()
  class OnWaitListPeerAccepted(val participant: DyteWaitlistedParticipant) : DyteEventType()
  class OnWaitListPeerRejected(val participant: DyteWaitlistedParticipant) : DyteEventType()
  class OnWaitListPeerClosed(val participant: DyteWaitlistedParticipant) : DyteEventType()
  class OnSelfWaitListStatusUpdate(val status: WaitListStatus) : DyteEventType()
  class OnSelfStageStatusUpdate(val status: StageStatus) : DyteEventType()
  object OnSelfAudioUpdate : DyteEventType()
  object OnSelfVideoUpdate : DyteEventType()
  object OnSelfUpdate : DyteEventType()

  class OnActiveSpeakerChanged(val participant: DyteJoinedMeetingParticipant) : DyteEventType()
  object OnNoActiveSpeaker : DyteEventType()

  object OnAudioDevicesChanged : DyteEventType()
  class OnProximityChanged(val isNear: Boolean) : DyteEventType()

  object OnMeetingRecordingStarted : DyteEventType()
  object OnMeetingRecordingStopped : DyteEventType()
  class OnMeetingRecordingError(val error: Exception) : DyteEventType()
  class OnMeetingRecordingStateUpdate(val state: DyteRecordingState) : DyteEventType()

  class OnMeetingPollsReceived(val polls: List<DytePollMessage>) : DyteEventType()
  class OnNewMeetingPollReceived(val poll: DytePollMessage) : DyteEventType()

  class OnMeetingMessagesReceived(val messages: List<DyteChatMessage>) : DyteEventType()
  class OnNewMeetingMessageReceived(val message: DyteChatMessage) : DyteEventType()

  class OnCameraError(val message: String?) : DyteEventType()
  object OnCameraDisconnected : DyteEventType()
  class OnCameraFreezed(val message: String?) : DyteEventType()
  class OnCameraOpening(val message: String?) : DyteEventType()
  object OnFirstFrameAvailable : DyteEventType()
  object OnCameraClosed : DyteEventType()

  class OnMicrophoneInitError(val message: String) : DyteEventType()
  class OnMicrophoneStartError(val message: String) : DyteEventType()
  class OnMicrophoneError(val message: String) : DyteEventType()

  object OnHostKicked : DyteEventType()

  class OnPluginEnabled(val plugin: DytePlugin) : DyteEventType()
  class OnPluginDisabled(val plugin: DytePlugin) : DyteEventType()
  class OnPluginMessage(val message: JsonObject) : DyteEventType()
  class OnPluginFileRequest(val plugin: DytePlugin) : DyteEventType()

  object OnWebinarPresentRequest : DyteEventType()
  object OnWebinarStoppedPresenting : DyteEventType()
  object OnWebinarStartedPresenting : DyteEventType()

  class OnWebinarPresentRequestAdded(val presentParticipant: RequestToPresentParticipant): DyteEventType()
  class OnWebinarPresentRequestClosed(val presentParticipant: RequestToPresentParticipant): DyteEventType()
  class OnWebinarPresentRequestAccepted(val presentParticipant: RequestToPresentParticipant): DyteEventType()
  class OnWebinarPresentRequestRejected(val presentParticipant: RequestToPresentParticipant): DyteEventType()
  class OnWebinarPresentRequestWithdrawn(val presentParticipant: RequestToPresentParticipant): DyteEventType()

  object OnLiveStreamStarting : DyteEventType()
  object OnLiveStreamStarted : DyteEventType()
  object  OnLiveStreamEnding : DyteEventType()
  object OnLiveStreamEnded : DyteEventType()
  object OnLiveStreamErrored : DyteEventType()
  class OnViewerCountUpdated(val count: Int) : DyteEventType()
  class OnStageCountUpdated(val count: Int) : DyteEventType()
  /*class OnLiveStreamPresentRequestAccepted(val peer: LiveStreamStagePeer): DyteEventType()
  class OnLiveStreamPresentRequestRejected(val peer: LiveStreamStagePeer): DyteEventType()*/
  class OnLiveStreamStageRequestsUpdated(val requests: List<LiveStreamStageRequestPeer>): DyteEventType()

  object OnLivestreamUpdate : DyteEventType()

  object OnMetaUpdate : DyteEventType()

  object OnConnectingToMeetingRoom: DyteEventType()

  object OnConnectedToMeetingRoom: DyteEventType()

  object OnDisconnectedFromMeetingRoom: DyteEventType()

  object OnMeetingRoomConnectionFailed : DyteEventType()

  object OnReconnectingToMeetingRoom : DyteEventType()

  object OnReconnectedToMeetingRoom : DyteEventType()

  object OnMeetingRoomReconnectionFailed : DyteEventType()

  class OnRoomMessage(val message: String): DyteEventType()
}