package io.dyte.core.controllers

import io.dyte.core.controllers.DyteEventType.OnViewerCountUpdated
import io.dyte.core.controllers.LiveStreamState.ERRORED
import io.dyte.core.controllers.LiveStreamState.NONE
import io.dyte.core.controllers.LiveStreamState.STARTED
import io.dyte.core.controllers.LiveStreamState.STOPPED
import io.dyte.core.socket.socketservice.SocketService
import io.dyte.core.socket.socketservice.SocketServiceEventListener
import io.dyte.core.socket.socketservice.SocketServiceUtils.RoomEvent
import kotlinx.coroutines.launch
import socket.livestreaming.LiveStreamingEvent
import socket.livestreaming.RoomPeerCountResponse

internal class LiveStreamController(
  controllerContainer: IControllerContainer,
  private val socketService: SocketService
) : BaseController(controllerContainer), ILiveStreamController {
  private var _liveStreamUrl: String? = null

  private var _state: LiveStreamState = NONE

  private var _viewerCount: Int = 0

  private var _roomName: String = ""

  override val livestreamData: DyteLivestreamData
    get() = DyteLivestreamData(_liveStreamUrl, _roomName, _state, _viewerCount)

  private val socketEventListener = object : SocketServiceEventListener {
    override fun onEvent(event: Int, eventId: String?, payload: ByteArray?) {
      when (event) {
        RoomEvent.STARTED.id -> {
          _state = STARTED
          val decodedPayload = LiveStreamingEvent.ADAPTER.decode(requireNotNull(payload))
          _roomName = decodedPayload.name
          _liveStreamUrl = decodedPayload.playback_url
          println("DyteMobileClient | LiveStreamController onEvent roomName $_roomName")
          controllerContainer.eventController.triggerEvent(DyteEventType.OnLiveStreamStarted)
        }

        RoomEvent.STOPPED.id -> {
          _state = STOPPED
          controllerContainer.eventController.triggerEvent(DyteEventType.OnLiveStreamEnded)
        }

        RoomEvent.ERRORED.id -> {
          _state = ERRORED
          controllerContainer.eventController.triggerEvent(DyteEventType.OnLiveStreamErrored)
        }

        RoomEvent.ROOM_PEER_COUNT.id -> {
          println("DyteMobileClient | LiveStreamController onEvent room peer count")
          val payload = RoomPeerCountResponse.ADAPTER.decode(requireNotNull(payload))
          println("DyteMobileClient | LiveStreamController onEvent ${payload.count}")
          _viewerCount = payload.count.toInt()
          controllerContainer.eventController.triggerEvent(OnViewerCountUpdated(_viewerCount))
        }
      }
    }
  }

  override fun init() {
    socketService.subscribe(RoomEvent.STARTED.id, socketEventListener)
    socketService.subscribe(RoomEvent.STOPPED.id, socketEventListener)
    socketService.subscribe(RoomEvent.ERRORED.id, socketEventListener)
    socketService.subscribe(RoomEvent.ROOM_PEER_COUNT.id, socketEventListener)
    loadLiveStreamUrl()
  }

  private fun loadLiveStreamUrl() {
    try {
      serialScope.launch {
        try {
          _liveStreamUrl = controllerContainer.apiClient.getLiveStreamUrl()
          _state = STARTED
          controllerContainer.eventController.triggerEvent(DyteEventType.OnLiveStreamStarted)
        } catch (e: Exception) {
          // ignore exception
        }
      }
    } catch (e: Exception) {
      _state = NONE
      e.printStackTrace()
    }
  }

  override fun start() {
    serialScope.launch {
      controllerContainer.apiClient.startLiveStream()
    }
  }

  override fun stop() {
    serialScope.launch {
      controllerContainer.apiClient.stopLiveStream()
    }
  }
}

interface ILiveStreamController {
  val livestreamData: DyteLivestreamData
  fun start()
  fun stop()
}

enum class LiveStreamState {
  NONE,
  STARTED,
  STOPPED,
  ERRORED;

  companion object {
    fun fromMap(stage: String): LiveStreamState {
      return when (stage) {
        "none" -> NONE
        "started" -> STARTED
        "stopped" -> STOPPED
        "errored" -> ERRORED
        else -> NONE
      }
    }
  }

  fun toMap(): Map<String, String> {
    val strState =
      when (this) {
        NONE -> "none"
        STARTED -> "started"
        STOPPED -> "stopped"
        ERRORED -> "errored"
      }

    val map: HashMap<String, String> = HashMap()

    map["state"] = strState

    return map;
  }
}

data class DyteLivestreamData(
  val liveStreamUrl: String?,
  val roomName: String,
  val state: LiveStreamState,
  val viewerCount: Int,
) {
  companion object {
    fun fromMap(map: Map<String, Any?>): DyteLivestreamData {

      val lvsUrl = map["livestreamUrl"] as String
      val roomName = map["roomName"] as String
      val viewerCount = map["viewerCount"] as Int
      val state = LiveStreamState.fromMap(map["state"] as String)

      return DyteLivestreamData(lvsUrl, roomName, state, viewerCount)
    }
  }

  fun toMap(): Map<String, Any?> {
    val map = HashMap<String, Any?>()
    map["livestreamUrl"] = liveStreamUrl
    map["roomName"] = roomName
    map["viewerCount"] = viewerCount
    map["state"] = state.toMap()["state"]
    return map;
  }
}