package io.dyte.core.controllers

import io.dyte.core.events.JoinRoomEventPublisher
import io.dyte.core.media.IDyteHive
import io.dyte.core.network.models.IceServerData
import io.dyte.core.socket.events.payloadmodel.inbound.MeetingPeerFlags
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketMeetingPeerUser
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketPeerLeftModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketPeerMuteModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketPeerPinnedModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketSelectedPeersModel
import io.dyte.core.socket.events.payloadmodel.outbound.RoomState
import io.dyte.core.socket.events.payloadmodel.outbound.WebSocketRoomStateModel
import io.dyte.core.socket.socketservice.SocketService
import io.dyte.core.socket.socketservice.SocketServiceEventListener
import io.dyte.core.socket.socketservice.SocketServiceUtils
import io.dyte.media.hive.HiveConsumerOptions
import io.dyte.media.hive.HiveTransport
import io.dyte.webrtc.MediaStreamTrackKind
import io.dyte.webrtc.RtpTransceiverDirection
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.ExperimentalCoroutinesApi
import kotlinx.coroutines.launch
import kotlinx.coroutines.newSingleThreadContext
import kotlinx.coroutines.runBlocking
import kotlinx.coroutines.sync.Mutex
import kotlinx.coroutines.sync.withLock
import media.ProducerKind
import media.ProducerState
import media.edge.*

internal class HiveController(
  controllerContainer: IControllerContainer,
  private val socketService: SocketService,
  private val joinRoomEventPublisher: JoinRoomEventPublisher? = null
) : IRoomNodeController, BaseController(controllerContainer) {

  private val logger = controllerContainer.loggerController

  private lateinit var iceServers: List<IceServerData>

  private val hiveUtils = controllerContainer.sfuUtils as IDyteHive

  // FIXME(itzmanish): hardcoding room's maxPreferredStreams to 6 for now.
  // Though it should not be use after completely shifting to hive node
  private val maxPreferredStreams = 6

  private var activatedProducingPeerIds = mutableSetOf<String>()

  private var roomJoined: Boolean = false

  private var peerProducers = mutableListOf<PeerProducerMeta>()

  private val peerId: String = controllerContainer.selfController.getSelf().id

  private var roomJoiningInProgress = false

  private val consumerMutex = Mutex()

  @OptIn(ExperimentalCoroutinesApi::class)
  private val coroutineScope = CoroutineScope(newSingleThreadContext("HiveController"))

  init {
    coroutineScope.launch {
      handleHiveInternalEvents()
    }

    handleSocketEvents()
  }

  override fun init() {
    //TODO("Socket service on state events")
  }

  override suspend fun joinRoom() {
    hiveUtils.stopAllTransports()

    this.roomJoiningInProgress = true

    controllerContainer.loggerController.traceLog("join room started")
    controllerContainer.eventController.triggerEvent(DyteEventType.OnMeetingRoomJoinStarted)

    try {
      val peerJoinRequest = PeerJoinRequest(
        display_name = controllerContainer.selfController.getSelf().name,
        prejoined = false,
        room_uuid = "",
      )

      socketService.send(
        event = SocketServiceUtils.MediaEvent.JOIN_ROOM.id,
        payload = PeerJoinRequest.ADAPTER.encode(peerJoinRequest)
      )

//      val peerJoinResponse = PeerJoinResponse.ADAPTER.decode(
//        socketService.requestResponse(
//          event = SocketServiceUtils.MediaEvent.JOIN_ROOM.id,
//          payload = PeerJoinRequest.ADAPTER.encode(peerJoinRequest)
//        )!!
//      )

      hiveUtils.setupTransports()

      completeJoinRoom()
    } catch (e: Error) {
      logger.traceLog("HiveDebug: Error on sending join room request: $e")
    }
  }

  private suspend fun completeJoinRoom() {
    try {
      val req = PeerJoinCompleteRequest()

      val peerJoinResponse = PeerJoinCompleteResponse.ADAPTER.decode(
        socketService.requestResponse(
          event = SocketServiceUtils.MediaEvent.SELF_JOIN_COMPLETE.id,
          payload = PeerJoinCompleteRequest.ADAPTER.encode(req)
        )!!
      )
      //var pluginResponseRaw: ByteArray? = ByteArray(0)
//      runBlocking {
//        peerJoinResponseRaw = socketService.requestResponse(
//          event = SocketServiceUtils.MediaEvent.JOIN_ROOM.id,
//          payload = PeerJoinRequest.ADAPTER.encode(peerJoinRequest)
//        )
//
////                pluginResponseRaw = socketService.requestResponse(
////                    event = SocketServiceEvents.PluginEvent.GET_PLUGINS.id,
////                    payload = null
////                )
//      }
//
//      val peerJoinResponse = PeerJoinResponse.ADAPTER.decode(peerJoinResponseRaw!!)
      //val pluginResponse = EnablePluginsResponse.ADAPTER.decode(pluginResponseRaw!!)
      // TODO: update proto files
      // val is_recorder = peerJoinResponse.room_state.recorder_participant_id == it.peer_id
      val oldPeers = peerJoinResponse.participants.map {
        WebSocketMeetingPeerUser(
          id = it.peer_id,
          isHost = false,
          name = it.display_name,
          flags = MeetingPeerFlags(recordere = false),
          userId = it.user_id,
          audioMuted = false,
        )
      }
//            val oldPlugins = pluginResponse.plugins.map {
//                RoomStatePlugin(
//                    id= it.plugin_id,
//                    enabledBy = it.enabled_by,
//                    staggered = false,
//                    stores = listOf(),
//                    aclAccessGrantedUsers = listOf(),
//                    aclAccessRemovedUsers = listOf(),
//                    aclType = ""
//                )
//            }

      val oldRoomState = WebSocketRoomStateModel(
        roomState= RoomState(
          displayTitle = peerJoinResponse.room_state?.display_title,
          roomUUID =  peerJoinResponse.room_state?.room_uuid,
          roomName = peerJoinResponse.room_state?.room_name,
          peers = oldPeers,
          //plugins = oldPlugins
        )
      )

      // check for remote peers joined before local user
      controllerContainer.participantController.handleRoomState(oldRoomState)

      // Pick out active plugins, if any
      // controllerContainer.pluginsController.handleRoomState(oldRoomState)
      // Notify other components who are listening to join-room event
      joinRoomEventPublisher?.publish()

      logger.traceLog("HiveDebug: Setting up transports")
//            CoroutineScope(Dispatchers.Default).launch {
      //hiveUtils.setupTransports()
//            }
      logger.traceLog("HiveDebug: Transports are setup")

      // TODO: Set Meeting Started Time
      // controllerContainer.metaController.setMeetingStartedTimestamp()

      // TODO: Handle Waitlisted State
      // controllerContainer.selfController.getSelf()._waitListStatus = WAITING
      //        controllerContainer.eventController.triggerEvent(
      //          DyteEventType.OnSelfWaitListStatusUpdate(
      //            WAITING
      //          )
      //        )


      this.roomJoiningInProgress = false
      this.roomJoined = true

      val audioEnableByDeveloper = controllerContainer.metaController.isAudioEnabled()

      if (controllerContainer.presetController.canPublishAudio() && audioEnableByDeveloper) {
        logger.traceLog("DyteMobileCLient | connectAudioProducerTransport is calling HP")
        hiveUtils.shareMic(controllerContainer.platformUtilsProvider.getMediaUtils().createAudioTrack())

        if (controllerContainer.selfController.getSelf()._audioEnabled) {
          controllerContainer.selfController.getSelf()._audioEnabled = true
          controllerContainer.eventController.triggerEvent(DyteEventType.OnSelfAudioUpdate)
        } else {
          hiveUtils.pauseMic()
        }
      }

      val videoEnableByDeveloper = controllerContainer.metaController.isVideoEnabled()

      if (controllerContainer.presetController.canPublishVideo()
        && videoEnableByDeveloper && controllerContainer.selfController.getSelf()._videoEnabled) {
        logger.traceLog("DyteMobileCLient | connectCameraProducerTransport is calling HP")
        hiveUtils.shareCam(controllerContainer.platformUtilsProvider.getMediaUtils().getVideoTrack())
      }

      val selfTracks = hiveUtils.getSelfTrack()
      val selfParticipant = controllerContainer.selfController.getSelf()
      // selfParticipant.audioTrack = selfTracks.first
      selfParticipant._videoTrack = selfTracks.second

//      controllerContainer.participantController.onSelfJoined()

      controllerContainer.participantController.onSelectedPeers(
        WebSocketSelectedPeersModel(
          peerIds = peerJoinResponse.selected_peers?.audio_peers,
          compulsoryPeers = peerJoinResponse.selected_peers?.compulsory_peers
        )
      )

      controllerContainer.selfController.onEnteredInRoom()

      controllerContainer.eventController.triggerEvent(
        DyteEventType.OnMeetingRoomJoined(
          controllerContainer.selfController.getSelf()
        )
      )

      peerJoinResponse.participants.forEach { participant ->
        participant.producer_states.forEach { state ->
          val producerMeta = PeerProducerMeta(
            producerId = state.producer_id,
            kind = if (state.kind == ProducerKind.AUDIO) "audio" else "video",
            paused = state.pause,
            screenShare = state.screen_share,
            peerId = participant.peer_id
          )

          createConsumer(producerMeta)
        }
      }

      controllerContainer.participantController.refreshGridParticipantsHive()

      if (controllerContainer.permissionController.isPermissionGrated(PermissionType.CAMERA).not()) {
        controllerContainer.loggerController.traceLog("joined without camera permission")
        controllerContainer.eventController.triggerEvent(
          DyteEventType.OnMeetingRoomJoinedWithoutCameraPermission
        )
      }
      if (controllerContainer.permissionController.isPermissionGrated(PermissionType.MICROPHONE).not()) {
        controllerContainer.loggerController.traceLog("joined without microphone permission")
        controllerContainer.eventController.triggerEvent(
          DyteEventType.OnMeetingRoomJoinedWithoutMicPermission
        )
      }

//            controllerContainer.pollsController.loadPolls()
//            controllerContainer.chatController.loadChatMessages()
//
//            controllerContainer.loggerController.traceLog("can produce audio ? ${controllerContainer.presetController.canPublishAudio()} } video ? ${controllerContainer.presetController.canPublishVideo()}")
//
//            if (controllerContainer.presetController.permissions.hostPermissions.canPresent && (controllerContainer.presetController.canPublishAudio() || controllerContainer.presetController.canPublishVideo())) {
//                connectProducerTransport()
//            }
//

      logger.traceLog("HiveDebug: join room flow completed")
    } catch (e:Exception) {
      e.printStackTrace()
      controllerContainer.eventController.triggerEvent(DyteEventType
        .OnMeetingRoomJoinFailed(e))
    }
  }

  private fun createConsumer(producerMeta: PeerProducerMeta) {
    peerProducers.add(producerMeta)

    if (producerMeta.kind == "audio" || producerMeta.screenShare ||
      controllerContainer.participantController.currentActiveGridParticipantsHive.contains(producerMeta.peerId)) {
      logger.traceLog("HiveDebug: handling new consumer")
      coroutineScope.launch {
        consumerMutex.withLock {
          hiveUtils.handleNewConsumer(
            HiveConsumerOptions(
              id = null,
              paused = producerMeta.paused,
              producerId = producerMeta.producerId,
              kind = if (producerMeta.kind == "audio") MediaStreamTrackKind.Audio else MediaStreamTrackKind.Video,
              producingPeerId = producerMeta.peerId,
              appData = mutableMapOf("screenShare" to producerMeta.screenShare)
            )
          )
        }
      }
    }
  }

  private fun handleSocketEvents() {
    val onPeerJoinedBroadcast = object : SocketServiceEventListener {
      override fun onEvent(event: Int, eventId: String?, payload: ByteArray?) {
        logger.traceLog("HiveDebug: Peer Joined Broadcast")
        if (!roomJoined) return

        try {
          val participant = PeerJoinBroadcastResponse.ADAPTER.decode(
            payload!!
          ).participant
          logger.traceLog("HiveController: Got peer joined with Id: ${participant?.peer_id}")

          if (peerId == participant?.peer_id) return

          val dyteParticipant = WebSocketMeetingPeerUser(
            id = participant?.peer_id,
            isHost = false,
            name = participant?.display_name,
            flags = MeetingPeerFlags(),
            userId = participant?.user_id, // Setting peer id as userid for now
//                        audioMuted = true,
            // videoEnabled
          )

          controllerContainer.participantController.onPeerJoined(dyteParticipant)

          serialScope.launch {
            controllerContainer.participantController.refreshGridParticipantsHive()
          }

        } catch (e: Error) {
          logger.traceLog("HiveController: Error in peer joined broadcast")
        }
      }
    }

    socketService.subscribe(
      SocketServiceUtils.MediaEvent.PEER_JOINED_BROADCAST.id,
      onPeerJoinedBroadcast
    )

    val onPeerProducerCreateBroadcast = object : SocketServiceEventListener {
      override fun onEvent(event: Int, eventId: String?, payload: ByteArray?) {
        if (!roomJoined) return

        try {
          val response = PeerProducerCreateBroadcastResponse.ADAPTER.decode(payload!!)

          if (response.participant_id == peerId) return

          logger.traceLog(
            "HiveController: Producer created broadcast: " +
              response.participant_id +
              "Producer id: ${response.producer_state?.producer_id}"
          )

          val participant =
            controllerContainer.participantController.meetingRoomParticipants.joined
              .find { it.id == response.participant_id }

          if (participant == null) {
            logger.traceLog("Peer not found for producer ${response.producer_state?.producer_id} ")
          }

          val producerMeta = PeerProducerMeta(
            producerId = response.producer_state!!.producer_id,
            kind = if (response.producer_state.kind == ProducerKind.AUDIO) "audio" else "video",
            paused = response.producer_state.pause,
            screenShare = response.producer_state.screen_share,
            peerId = response.participant_id
          )

          logger.traceLog("HiveDebug: Producer meta ${producerMeta.kind}")

          createConsumer(producerMeta)
        } catch (e: Error) {
          logger.traceLog("HiveController: Error in peer producer create broadcast")
        }
      }
    }

    socketService.subscribe(
      SocketServiceUtils.MediaEvent.PEER_PRODUCER_CREATE_BROADCAST.id,
      onPeerProducerCreateBroadcast
    )

    val onPeerProducerToggleBroadcast = object : SocketServiceEventListener {
      override fun onEvent(event: Int, eventId: String?, payload: ByteArray?) {
        if (!roomJoined) return

        try {
          val response = PeerProducerToggleBroadcastResponse.ADAPTER.decode(payload!!)

          if (response.participant_id == peerId) return

          val convertedKind: MediaStreamTrackKind =
            if (response.producer_state?.kind == ProducerKind.AUDIO)
              MediaStreamTrackKind.Audio
            else MediaStreamTrackKind.Video

          logger.traceLog(
            "HiveController: Producer toggle broadcast: " +
              "${response.participant_id} producer id: " +
              "${response.producer_state?.producer_id}"
          )
          val participant =
            controllerContainer.participantController.meetingRoomParticipants.joined
              .find { it.id == response.participant_id } ?: return

          if (response.producer_state!!.kind == ProducerKind.AUDIO) {
            if (response.producer_state.pause) {
              controllerContainer.participantController
                .onPeerAudioMuted(
                  WebSocketPeerMuteModel(participant.id)
                )
            } else {
              controllerContainer.participantController
                .onPeerAudioUnmuted(WebSocketPeerMuteModel(participant.id))
            }
          }

          val producer = peerProducers.find { it.peerId == response.participant_id }
          if (producer != null) {
            producer.paused = response.producer_state.pause
          }

          val consumersForThisProducer = hiveUtils.getConsumers().values.filter {
            it.getProducerId() == response.producer_state.producer_id && it.getKind() == convertedKind
          }

          consumersForThisProducer.forEach {
            if (it.getPaused() != response.producer_state.pause) {
              logger.traceLog("HiveController: Consumer state is mismatched for $")

              CoroutineScope(Dispatchers.Default).launch {
                if (response.producer_state.pause) {
                  it.pause()
                  hiveUtils.handlePauseConsumer(
                    it.getId()
                  )
                } else {
                  it.resume()
                  hiveUtils.handleResumeConsumer(
                    it.getId()
                  )
                }
              }
            }
          }
        } catch (e: Error) {
          logger.traceLog("HiveController: Error in producer toggle broadcast handler")
        }
      }
    }

    socketService.subscribe(
      SocketServiceUtils.MediaEvent.PEER_PRODUCER_TOGGLE_BROADCAST.id,
      onPeerProducerToggleBroadcast
    )

    val onSelectedPeer = object : SocketServiceEventListener {
      override fun onEvent(event: Int, eventId: String?, payload: ByteArray?) {
        if (!roomJoined) return

        try {
          val response = SelectedPeersResponse.ADAPTER.decode(payload!!)
          logger.traceLog(
            "HiveController: Selected peers: ${response.audio_peers} " +
              "${response.compulsory_peers}"
          )

          controllerContainer.participantController.onSelectedPeers(
            WebSocketSelectedPeersModel(
              peerIds = response.audio_peers,
              compulsoryPeers = response.compulsory_peers
            )
          )
        } catch (e: Error) {
          logger.traceLog("HiveController: Error in selected peer $e")
        }
      }
    }

    socketService.subscribe(SocketServiceUtils.MediaEvent.SELECTED_PEER.id, onSelectedPeer)

    val onSelectedPeerDiff = object : SocketServiceEventListener {
      override fun onEvent(event: Int, eventId: String?, payload: ByteArray?) {
        if (!roomJoined) return

        try {
          val entries = SelectedPeersDiffResponse.ADAPTER.decode(payload!!).entries

          val peerIdsWithPriorities = entries.map {
            PeerWithPriority(
              peerId = it.peer_id,
              priority = it.priority,
            )
          }

          controllerContainer.participantController.onSelectedPeersDiff(
            peerIdsWithPriorities
          )
        } catch (e: Error) {
          logger.traceLog("HiveController: Error in selected peer diff $e")
        }
      }
    }

    socketService.subscribe(
      SocketServiceUtils.MediaEvent.SELECTED_PEER_DIFF.id,
      onSelectedPeerDiff
    )

    val onPeerLeaveBroadcast = object : SocketServiceEventListener {
      override fun onEvent(event: Int, eventId: String?, payload: ByteArray?) {
        if (!roomJoined) return

        try {
          val participantId = PeerLeaveBroadcastResponse.ADAPTER.decode(
            payload!!
          ).participant_id

          if (participantId == peerId) return

          logger.traceLog("HiveController: Peer left broadcast: $participantId")

          runBlocking {
            hiveUtils.cleanupConsumers(participantId)
          }

          controllerContainer.participantController.onPeerLeft(
            WebSocketPeerLeftModel(
              id = participantId
            )
          )
        } catch (e: Error) {
          logger.traceLog("HiveController: Error in peer left broadcast $e")
        }
      }
    }

    socketService.subscribe(
      SocketServiceUtils.MediaEvent.PEER_LEAVE_BROADCAST.id,
      onPeerLeaveBroadcast
    )

    val onPeerProducerCloseBroadcast = object : SocketServiceEventListener {
      override fun onEvent(event: Int, eventId: String?, payload: ByteArray?) {
        if (!roomJoined) return

        try {
          val response = PeerProducerCloseBroadcastResponse.ADAPTER.decode(payload!!)

          if (response.participant_id == peerId) return

          logger.traceLog("HiveController: Producer closed broadcast ${response.participant_id}")

          peerProducers =
            peerProducers.filter { it.producerId != response.producer_state!!.producer_id }
              .toMutableList()

          val producerId = response.producer_state!!.producer_id

          val consumerId = hiveUtils.producerIdToConsumerIdMap[producerId]

          if (consumerId == null) {
            logger.traceLog("HiveController: No consumer found for producer: $producerId")
            //return
          }

          logger.traceLog("HiveController: Closing consumer $consumerId for $producerId")

          runBlocking {
            consumerId?.let {
              hiveUtils.closeConsumer(it, null)
            }
            //hiveUtils.closeConsumer(consumerId, null)
          }

          if (response.producer_state.screen_share) {
            val participant =
              controllerContainer.participantController.meetingRoomParticipants.joined
                .find { it.id == response.participant_id }

            participant?.let {
              val screenShareParticipant = controllerContainer.participantController.meetingRoomParticipants.screenshares.find { ss ->
                ss.id == response.participant_id
              }

              screenShareParticipant?.let {
                controllerContainer.participantController.onPeerScreenSharedEnded(screenShareParticipant)
              }
            }
          }

          logger.traceLog("HiveController: Closed consumer $consumerId")

          hiveUtils.producerIdToConsumerIdMap.remove(producerId)
          // todo hive

//          CoroutineScope(Dispatchers.Default).async {
//            hiveUtils.closeConsumer(consumerId)
//          }
        } catch (e: Error) {
          logger.traceLog("HiveController: Error on Producer close broadcast $e")
        }
      }
    }

    socketService.subscribe(
      SocketServiceUtils.MediaEvent.PEER_PRODUCER_CLOSE_BROADCAST.id,
      onPeerProducerCloseBroadcast
    )

    val onGlobalPeerPinBroadcast = object : SocketServiceEventListener {
      override fun onEvent(event: Int, eventId: String?, payload: ByteArray?) {
        if (!roomJoined) return

        try {
          if (payload != null) {
            val peerId = GlobalPeerPinningBroadcastResponse.ADAPTER.decode(
              payload
            ).participant_id
            logger.traceLog("HiveController: Pinning peerId: $peerId")

            controllerContainer.participantController.onPeerPinned(
              WebSocketPeerPinnedModel(
                peerId = peerId
              )
            )
          } else {
            controllerContainer.participantController.onPeerUnpinned()
          }
        } catch (e: Error) {
          logger.traceLog("HiveController: Error on global peer pin broadcast $e")
        }
      }
    }

    socketService.subscribe(
      SocketServiceUtils.MediaEvent.GLOBAL_PEER_PIN_BROADCAST.id,
      onGlobalPeerPinBroadcast
    )
  }

  override suspend fun closeConsumers(peers: List<String>) {
    val consumerIds = hiveUtils.getConsumers().filter {
      peers.contains(it.value.getPeerId())
    }.map { itt -> itt.value.getId() }

    peers.forEach {
      activatedProducingPeerIds.remove(it)
    }

    return hiveUtils.closeConsumers(consumerIds)
  }

  override suspend fun consumePeer(producingPeerId: String, force: Boolean?) {
    if (!activatedProducingPeerIds.contains(producingPeerId) || force!!) {
      try {
        hiveUtils.consumePeer(producingPeerId)
        this.activatedProducingPeerIds.add(producingPeerId)
      } catch (e: Error) {
        logger.traceLog("HiveController: consumePeer failed $e")
      }
    } else {
      logger.traceLog("HiveController: Not creating consumer for peerId = $producingPeerId")
    }
  }

  override suspend fun pinPeer(peerId: String?) {
    val req = GlobalPeerPinningRequest(
      participant_id = peerId ?: ""
    )

    try {
      socketService.send(
        event = SocketServiceUtils.MediaEvent.GLOBAL_PIN_PEER.id,
        payload = GlobalPeerPinningRequest.ADAPTER.encode(req)
      )
    } catch (e:Error) {
      logger.traceLog("HiveController: Error in pinning peer $e")
    }
  }

  override fun leaveRoom() {
    roomJoined = false
  }

  suspend fun activatePeer(peerId: String, force: Boolean? = false) {
    logger.traceLog("HiveDebug: in activate peer")
    val producers = peerProducers.filter { it.peerId == peerId }
    if (producers.isEmpty()) {
      logger.traceLog("HiveUtils: activatePeer $peerId: Not an active peer")

      return
    }

    if (force == true) {
      consumerMutex.withLock {
        this.hiveUtils.consumePeer(peerId)
      }
      return
    }

    val consumersToResume = arrayListOf<String>()

    val allConsumersExist = producers.all { producer ->
      val consumerId = this.hiveUtils.producerIdToConsumerIdMap.get(producer.producerId) ?: return@all false
      val consumer = this.hiveUtils.getConsumers()[consumerId]

      if (consumer != null && !producer.paused && consumer.getPaused()) {
        consumersToResume.add(consumerId)
      }

      return@all true
    }

    if (!allConsumersExist) {
      consumerMutex.withLock {
        this.hiveUtils.consumePeer(peerId)
      }
      return
    }

    consumersToResume.forEach {
      this.hiveUtils.handleResumeConsumer(it)
    }
  }


  suspend fun deactivatePeer(peerId: String) {
    val producers = peerProducers.filter { it.peerId == peerId }
    logger.traceLog("HiveUtils: deactivatePeer $peerId")

    producers.forEach {
      if (it.kind == "audio" ||
        (it.kind == "video" && it.screenShare)) {
        logger.traceLog("HiveUtils: We don't want to pause audio or screenshare")

        return
      }

      val consumerId = this.hiveUtils.producerIdToConsumerIdMap[it.producerId]

      if (consumerId == null) {
        logger.traceLog("HiveUtils: Consumer not found in deactivate producers ${it.producerId}")

        return
      }

      this.hiveUtils.closeConsumer(consumerId)
      this.activatedProducingPeerIds.remove(peerId)
    }
  }

  override suspend fun reconnectTransport(transport: HiveTransport) {
    // TODO: Socket is connected or not
    if (this.roomJoiningInProgress) return

    val iceServerData = controllerContainer.apiClient.getICEServers().iceServers

    transport.close()

    if (transport.getDirection() == RtpTransceiverDirection.SendOnly) {
      this.hiveUtils.createWebRtcTransportProd(null, iceServerData)
      logger.traceLog("HiveUtils: Transport connected now need to start creating producers")

      this.hiveUtils.resetVideoProducers(
        controllerContainer.platformUtilsProvider.getMediaUtils().getVideoTrack(),
        null
      )

      return
    }

    this.hiveUtils.cleanupConsumers(null)

    this.hiveUtils.createWebRtcTransportRecv(null, iceServerData)

//    val lastActivePeers = this.activatedProducingPeerIds
    this.activatedProducingPeerIds = mutableSetOf()

//    lastActivePeers.forEach {
//      this.consumePeer(it, true)
//    }
    controllerContainer.participantController.refreshGridParticipantsHive(true)
  }

  private suspend fun handleHiveInternalEvents() {
    this.hiveUtils.getObserverFlow().collect {
      if (it.eventName == "reconnect_transport") {
        val transport = it.data as HiveTransport

        try {
          this.reconnectTransport(transport)
          logger.traceLog("HiveController: Transport reconnected ${transport.getId()}")
        } catch (e: Error) {
          logger.traceLog("HiveController: Error on reconnecting transports: $e")
        }
      }
    }
  }
}

interface IRoomNodeController {
  fun init() {}
  suspend fun joinRoom()

  fun connectProducerTransport() {}

  // Hive specific methods
  suspend fun reconnectTransport(transport: HiveTransport) {}

  suspend fun activatePeer(peerId: String, producers: List<ProducerState>, force: Boolean?) {}

  suspend fun deactivatePeer(peerId: String, producers: List<ProducerState>) {}

  suspend fun closeConsumers(peers: List<String>) {}

  suspend fun consumePeer(producingPeerId: String, force: Boolean? = null) {}

  suspend fun pinPeer(peerId: String? = null) {}

  fun leaveRoom()
}

data class PeerProducerMeta(
  var producerId: String,
  var kind: String,
  var paused: Boolean,
  var screenShare: Boolean,
  var peerId: String,
)