package io.dyte.core.polls

import io.dyte.core.chat.DefaultChatSocketHandler
import io.dyte.core.controllers.DyteEventType
import io.dyte.core.controllers.IControllerContainer
import io.dyte.core.feat.DytePollMessage
import io.dyte.core.feat.DytePollOption
import io.dyte.core.feat.DytePollVote
import io.dyte.core.socket.events.OutboundMeetingEventType.GET_POLLS
import io.dyte.core.socket.events.OutboundMeetingEventType.NEW_POLL
import io.dyte.core.socket.events.OutboundMeetingEventType.VOTE_POLL
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketPoll
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketPollsModel
import io.dyte.core.socket.socketservice.SocketService
import io.dyte.core.socket.socketservice.SocketServiceEventListener
import kotlinx.serialization.json.JsonArray
import kotlinx.serialization.json.JsonElement
import kotlinx.serialization.json.JsonObject
import kotlinx.serialization.json.JsonPrimitive
import socket.polls.GetPollsResponse
import socket.polls.NewPollRequest
import socket.polls.Poll
import socket.polls.UpdatePollResponse
import socket.polls.VotePollRequest

internal class PollsSocketHandler(
  private val pollsSocketServer: String,
  private val socketService: SocketService,
  private val controllerContainer: IControllerContainer
) : IPollsSocketHandler {
  private var _polls: ArrayList<DytePollMessage> = arrayListOf()

  override val isUsingSocketService: Boolean
    get() = pollsSocketServer == DefaultChatSocketHandler.MEETING_SOCKET_SOCKET_SERVICE

  override val polls: List<DytePollMessage>
    get() = _polls

  private val pollSocketListener = object : SocketServiceEventListener {
    override fun onEvent(event: Int, eventId: String?, payload: ByteArray?) {
      when (event) {
        PollSocketEvent.CREATE_POLL.id -> {
          val decodedPayload = UpdatePollResponse.ADAPTER.decode(requireNotNull(payload))
          handleNewPoll(decodedPayload)
        }

        PollSocketEvent.UPDATE_POLL.id -> {
          val decodedPayload = UpdatePollResponse.ADAPTER.decode(requireNotNull(payload))
          decodedPayload.poll?.let {
            handleNewPoll(decodedPayload.poll, true)
          }        }
      }
    }
  }

  init {
    if (isUsingSocketService) {
      socketService.subscribe(PollSocketEvent.CREATE_POLL.id, pollSocketListener)
      socketService.subscribe(PollSocketEvent.UPDATE_POLL.id, pollSocketListener)
    }
  }

  override suspend fun loadPolls() {
    if (isUsingSocketService) {
      try {
        val payload = socketService.requestResponse(PollSocketEvent.GET_POLLS.id, null)
        payload?.let {
          val decodedPayload = GetPollsResponse.ADAPTER.decode(payload)
          decodedPayload.polls.forEach {
            handleNewPoll(it, false)
          }
          if (polls.isNotEmpty()) {
            controllerContainer.eventController.triggerEvent(DyteEventType.OnMeetingPollsReceived(polls))
          }
        }
      } catch (e: Exception) {
        // ignore exception
      }
    } else {
      val pollsResponse = controllerContainer.socketController.sendMessage(GET_POLLS, null)
      val socketParsedResponse =
        controllerContainer.socketMessageResponseParser.parseResponse(pollsResponse)
      val webSocketPollsModel = socketParsedResponse.payload as WebSocketPollsModel
      webSocketPollsModel.polls?.values?.forEach {
        handlePoll(it, false)
      }
      if (polls.isNotEmpty()) {
        controllerContainer.eventController.triggerEvent(DyteEventType.OnMeetingPollsReceived(polls))
      }
    }
  }

  override suspend fun createPoll(
    question: String,
    options: List<String>,
    anonymous: Boolean,
    hideVotes: Boolean
  ) {
    if (isUsingSocketService) {
      val newPollRequest = NewPollRequest(question, options, anonymous)
      socketService.send(
        event = PollSocketEvent.CREATE_POLL.id,
        payload = NewPollRequest.ADAPTER.encode(newPollRequest)
      )
    } else {
      val content = HashMap<String, JsonElement>()
      content["question"] = JsonPrimitive(question)
      content["options"] = JsonArray(options.map { JsonPrimitive(it) })
      content["anonymous"] = JsonPrimitive(anonymous)
      content["hideVotes"] = JsonPrimitive(hideVotes)
      controllerContainer.socketController.sendMessage(NEW_POLL, JsonObject(content))
    }
  }

  private fun handleNewPoll(updatePollResponse: UpdatePollResponse) {
    updatePollResponse.poll?.let { poll ->
      handleNewPoll(poll, true)
    }
  }

  private fun handleNewPoll(poll: Poll, updateUI: Boolean) {
    val options = ArrayList<DytePollOption>()
    poll.options.forEach { option ->
      val votes = ArrayList<DytePollVote>()
      val updateOption = DytePollOption(option.text, votes, option.count?.toInt() ?: 0)
      option.votes.forEach { vote ->
        votes.add(DytePollVote(vote.user_id, vote.name))
      }
      options.add(updateOption)
    }
    val existingPoll = polls.find { it.id == poll.poll_id }
    if (existingPoll == null) {
      val newPoll = DytePollMessage(
        poll.poll_id,
        poll.question,
        poll.anonymous,
        poll.hide_votes,
        poll.created_by,
        options
      )
      _polls.add(newPoll)
      if (updateUI) {
        controllerContainer.eventController.triggerEvent(
          DyteEventType.OnNewMeetingPollReceived(
            newPoll
          )
        )
        controllerContainer.eventController.triggerEvent(DyteEventType.OnMeetingPollsReceived(polls))
      }
    } else {
      val updatedPoll = existingPoll.copy(options = options)
      val existingIndex = polls.indexOf(existingPoll)
      _polls.removeAt(existingIndex)
      _polls.add(existingIndex, updatedPoll)
      if (updateUI) {
        controllerContainer.eventController.triggerEvent(DyteEventType.OnMeetingPollsReceived(polls))
      }
    }
  }

  override fun handlePoll(webSocketPoll: WebSocketPoll, updateUI: Boolean) {
    if (!controllerContainer.presetController.canViewPoll()) {
      throw UnsupportedOperationException("not allowed to view poll")
    }

    val options = ArrayList<DytePollOption>()
    webSocketPoll.options.forEach { option ->
      val votes = ArrayList<DytePollVote>()
      val updateOption = DytePollOption(option.text, votes, option.count)
      option.votes.forEach { vote ->
        votes.add(DytePollVote(vote.id, vote.name))
      }
      options.add(updateOption)
    }
    val ifPollExists = polls.find { it.id == webSocketPoll.id } != null
    if (ifPollExists) {
      val existingPoll = requireNotNull(polls.find { it.id == webSocketPoll.id })
      val updatedPoll = existingPoll.copy(options = options)
      val existingIndex = polls.indexOf(existingPoll)
      _polls.removeAt(existingIndex)
      _polls.add(existingIndex, updatedPoll)
      if (updateUI) {
        controllerContainer.eventController.triggerEvent(DyteEventType.OnMeetingPollsReceived(polls))
      }
    } else {
      val newPoll = DytePollMessage(
        webSocketPoll.id,
        webSocketPoll.question,
        webSocketPoll.anonymous,
        webSocketPoll.hideVotes,
        webSocketPoll.createdBy,
        options
      )
      _polls.add(newPoll)
      if (updateUI) {
        controllerContainer.eventController.triggerEvent(
          DyteEventType.OnNewMeetingPollReceived(
            newPoll
          )
        )
        controllerContainer.eventController.triggerEvent(DyteEventType.OnMeetingPollsReceived(polls))
      }
    }
  }

  override suspend fun vote(
    dytePollMessage: DytePollMessage,
    dytePollOption: DytePollOption
  ) {
    if (isUsingSocketService) {
      socketService.send(PollSocketEvent.VOTE_POLL.id, VotePollRequest.ADAPTER.encode(VotePollRequest(dytePollMessage.id, dytePollMessage.options.indexOf(dytePollOption).toLong())))
    } else {
      val content = HashMap<String, JsonElement>()
      content["index"] = JsonPrimitive(dytePollMessage.options.indexOf(dytePollOption))
      content["pollId"] = JsonPrimitive(dytePollMessage.id)
      controllerContainer.socketController.sendMessage(VOTE_POLL, JsonObject(content))
    }
  }
}

interface IPollsSocketHandler {
  val isUsingSocketService: Boolean
  val polls: List<DytePollMessage>

  suspend fun loadPolls()

  suspend fun vote(dytePollMessage: DytePollMessage, dytePollOption: DytePollOption)

  suspend fun createPoll(
    question: String,
    options: List<String>,
    anonymous: Boolean,
    hideVotes: Boolean
  )

  fun handlePoll(webSocketPoll: WebSocketPoll, updateUI: Boolean)
}