package io.dyte.core.plugins.socketservice

import io.dyte.core.controllers.DyteEventType
import io.dyte.core.controllers.IControllerContainer
import io.dyte.core.events.JoinRoomEventReceiver
import io.dyte.core.feat.DytePlugin
import io.dyte.core.models.DyteJoinedMeetingParticipant
import io.dyte.core.models.DyteSelfParticipant
import io.dyte.core.network.IApiClient
import io.dyte.core.plugins.LocalPluginEvent
import io.dyte.core.plugins.PluginsController
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.ExperimentalCoroutinesApi
import kotlinx.coroutines.flow.launchIn
import kotlinx.coroutines.flow.onEach
import kotlinx.coroutines.newSingleThreadContext
import kotlinx.serialization.json.JsonArray
import kotlinx.serialization.json.JsonElement
import kotlinx.serialization.json.JsonNull
import kotlinx.serialization.json.JsonObject
import kotlinx.serialization.json.addJsonObject
import kotlinx.serialization.json.buildJsonArray
import kotlinx.serialization.json.buildJsonObject
import kotlinx.serialization.json.int
import kotlinx.serialization.json.jsonArray
import kotlinx.serialization.json.jsonObject
import kotlinx.serialization.json.jsonPrimitive
import kotlinx.serialization.json.put
import kotlinx.serialization.json.putJsonObject

internal class SocketServicePluginsController(
  private val meetingTitle: String,
  private val roomName: String,
  private val organizationId: String,
  private val v1Plugins: Set<String>,
  private val apiClient: IApiClient,
  private val pluginsSocketHandler: PluginsSocketHandler,
  private val joinRoomEventReceiver: JoinRoomEventReceiver,
  private val controllerContainer: IControllerContainer
) : PluginsController(controllerContainer) {

  @OptIn(ExperimentalCoroutinesApi::class)
  private var serialCoroutineScope =
    CoroutineScope(newSingleThreadContext("SSPluginsControllerScope"))

  override suspend fun init() {
    fetchAllPlugins()

    joinRoomEventReceiver.joinRoomEvent.onEach {
      loadActivePlugins()
    }.launchIn(serialCoroutineScope)

    pluginsSocketHandler.listenToPluginSocketMessages().onEach { message ->
      handlePluginSocketMessage(message)
    }.launchIn(serialCoroutineScope)
  }

  override suspend fun launchPlugin(pluginId: String, staggered: Boolean) {
    pluginsSocketHandler.addPlugin(pluginId, staggered)
  }

  override suspend fun closePlugin(pluginId: String) {
    pluginsSocketHandler.removePlugin(pluginId)
  }

  override suspend fun handlePluginMessage(pluginId: String, message: JsonObject) {
    val type = message["type"]?.jsonPrimitive?.int
    val uuid = message["uuid"]?.jsonPrimitive?.content
    val payload = message["payload"]

    when (type) {
      PluginsSocketEvent.CUSTOM_PLUGIN_EVENT_TO_ROOM.id -> {
        if (payload == null) {
          return
        }
        pluginsSocketHandler.sendCustomPluginEventToRoom(pluginId, payload, uuid)
      }

      PluginsSocketEvent.CUSTOM_PLUGIN_EVENT_TO_PEERS.id -> {
        if (payload == null || payload !is JsonObject) {
          return
        }
        val peerIds = extractPeerIdsFromPluginMessagePayload(payload) ?: return
        pluginsSocketHandler.sendCustomPluginEventToPeers(pluginId, peerIds, payload, uuid)
      }

      PluginsSocketEvent.ENABLE_PLUGIN_FOR_ROOM.id -> {
        pluginsSocketHandler.enablePluginForRoom(pluginId, uuid)
      }

      PluginsSocketEvent.ENABLE_PLUGIN_FOR_PEERS.id -> {
        if (payload == null || payload !is JsonObject) {
          return
        }
        val peerIds = extractPeerIdsFromPluginMessagePayload(payload) ?: return
        pluginsSocketHandler.enablePluginForPeers(pluginId, peerIds, uuid)
      }

      PluginsSocketEvent.DISABLE_PLUGIN_FOR_ROOM.id -> {
        pluginsSocketHandler.disablePluginForRoom(pluginId, uuid)
      }

      PluginsSocketEvent.DISABLE_PLUGIN_FOR_PEERS.id -> {
        if (payload == null || payload !is JsonObject) {
          return
        }
        val peerIds = extractPeerIdsFromPluginMessagePayload(payload) ?: return
        pluginsSocketHandler.disablePluginForPeers(pluginId, peerIds, uuid)
      }

      PluginsSocketEvent.STORE_INSERT_KEYS.id -> {
        if (payload == null || payload !is JsonObject) {
          return
        }
        val store = payload["store"]?.jsonPrimitive?.content ?: return
        val insertKeysJsonArray = payload["insertKeys"]?.jsonArray ?: return
        val insertKeys = mutableListOf<PluginsSocketHandler.PluginStoreKeyEntry>()
        for (insertKey in insertKeysJsonArray) {
          val key = insertKey.jsonObject["key"]?.jsonPrimitive?.content ?: continue
          val insertKeyPayload = insertKey.jsonObject["payload"]
          insertKeys.add(PluginsSocketHandler.PluginStoreKeyEntry(key, insertKeyPayload))
        }
        pluginsSocketHandler.storeInsertKeys(pluginId, store, insertKeys, uuid)
      }

      PluginsSocketEvent.STORE_GET_KEYS.id -> {
        if (payload == null || payload !is JsonObject) {
          return
        }
        val store = payload["store"]?.jsonPrimitive?.content ?: return
        val getKeysJsonArray = payload["getKeys"]?.jsonArray ?: return
        val getKeys = mutableListOf<String>()
        for (getKey in getKeysJsonArray) {
          val key = getKey.jsonObject["key"]?.jsonPrimitive?.content ?: continue
          getKeys.add(key)
        }
        pluginsSocketHandler.storeGetKeys(pluginId, store, getKeys, uuid)
      }

      PluginsSocketEvent.STORE_DELETE_KEYS.id -> {
        if (payload == null || payload !is JsonObject) {
          return
        }
        val store = payload["store"]?.jsonPrimitive?.content ?: return
        val deleteKeysJsonArray = payload["deleteKeys"]?.jsonArray ?: return
        val deleteKeys = mutableListOf<String>()
        for (deleteKey in deleteKeysJsonArray) {
          val key = deleteKey.jsonObject["key"]?.jsonPrimitive?.content ?: continue
          deleteKeys.add(key)
        }
        pluginsSocketHandler.storeDeleteKeys(pluginId, store, deleteKeys, uuid)
      }

      PluginsSocketEvent.STORE_DELETE_KEYS.id -> {
        if (payload == null || payload !is JsonObject) {
          return
        }
        val store = payload["store"]?.jsonPrimitive?.content ?: return
        val deleteKeysJsonArray = payload["deleteKeys"]?.jsonArray ?: return
        val deleteKeys = mutableListOf<String>()
        for (deleteKey in deleteKeysJsonArray) {
          val key = deleteKey.jsonObject["key"]?.jsonPrimitive?.content ?: continue
          deleteKeys.add(key)
        }
        pluginsSocketHandler.storeDeleteKeys(pluginId, store, deleteKeys, uuid)
      }

      PluginsSocketEvent.STORE_DELETE.id -> {
        if (payload == null || payload !is JsonObject) {
          return
        }
        val store = payload["store"]?.jsonPrimitive?.content ?: return
        pluginsSocketHandler.deleteStore(pluginId, store, uuid)
      }

      LocalPluginEvent.GET_PLUGIN_INITIATOR.id -> {
        val enabledBy = _activePlugins[pluginId]?.enabledBy ?: return
        val data = buildJsonObject {
          put("enabledBy", enabledBy)
        }
        submitMessageToPlugin(pluginId, type, uuid, data)
      }

      LocalPluginEvent.GET_DISPLAY_TITLE.id -> {
        val data = buildJsonObject {
          put("displayTitle", meetingTitle)
        }
        submitMessageToPlugin(pluginId, type, uuid, data)
      }

      LocalPluginEvent.GET_ROOM_NAME.id -> {
        val data = buildJsonObject {
          put("roomName", roomName)
        }
        submitMessageToPlugin(pluginId, type, uuid, data)
      }

      LocalPluginEvent.CHAT_MESSAGE.id -> {
        if (payload == null || payload !is JsonObject) {
          return
        }

        val messagePayload = payload["messagePayload"]?.jsonObject
        val peerIds = try {
          payload["peerIds"]?.jsonArray
        } catch (e: Exception) {
          return
        }

        // TODO: Once we re-structure our features, check if Chat feature is enabled, here

        if (!peerIds.isNullOrEmpty()) {
          return
        }

        // TODO: Once we have support for Private chat, use that API here.
        val messageType = messagePayload?.get("type")?.jsonPrimitive?.content

        /*
        * note(swapnil): Only sending text type chat messages for now.
        *
        * { type: 'image', image: File }, { type: 'file', file: File }
        * TODO: Check the data coming in File object as part of messagePayload for file and image
        *  type payload and find a way to use it with mobile Chat APIs.
        * */
        if (messageType == "text") {
          try {
            val textMessage = messagePayload["message"]?.jsonPrimitive?.content ?: return
            controllerContainer.chatController.sendMessage(textMessage)
            val data = buildJsonObject { put("success", true) }
            submitMessageToPlugin(pluginId, type, uuid, data)
          } catch (e: Exception) {
            val data = buildJsonObject { put("error", e.message) }
            submitMessageToPlugin(pluginId, type, uuid, data)
          }
        }
      }

      LocalPluginEvent.GET_PEER.id -> {
        val peerId = payload?.jsonObject?.get("peerId")?.jsonPrimitive?.content
        val selfParticipant = controllerContainer.selfController.getSelf()
        val peer: JsonElement = if (peerId == null) {
          selfParticipant.toJsonObject(organizationId)
        } else {
          val joinedMeetingParticipant =
            controllerContainer.participantController.meetingRoomParticipants.joined.find { it.id == peerId }
          joinedMeetingParticipant?.let {
            if (it.id == selfParticipant.id) {
              selfParticipant.toJsonObject(organizationId)
            } else {
              it.toJsonObject()
            }
          } ?: JsonNull
        }

        val data = buildJsonObject {
          put("peer", peer)
        }
        submitMessageToPlugin(pluginId, type, uuid, data)
      }

      LocalPluginEvent.GET_PEERS.id -> {
        val serializedPeers =
          controllerContainer.participantController.meetingRoomParticipants.joined.map {
            it.toJsonObject()
          }
        val data = buildJsonObject {
          put("peers", JsonArray(serializedPeers))
        }
        submitMessageToPlugin(pluginId, type, uuid, data)
      }

      LocalPluginEvent.CUSTOM_PLUGIN_EVENT_TO_PARENT.id -> {
      }
    }
  }

  private suspend fun loadActivePlugins() {
    try {
      val activePlugins = pluginsSocketHandler.getActivePlugins()
      activePlugins.forEach { activePlugin ->
        val plugin = _allPlugins[activePlugin.id]
        plugin?.let {
          enablePluginForLocalUser(it.id, activePlugin.enabledBy)
        }
      }
    } catch (e: Exception) {
      // TODO: Log getActivePlugins failure
    }
  }

  private suspend fun fetchAllPlugins() {
    val getPluginsResponse = apiClient.getPlugins()
    val socketServicePlugins = getPluginsResponse.data.plugins.filter {
      !v1Plugins.contains(it.id)
    }
    socketServicePlugins.forEach {
      val plugin = DytePlugin(
        id = it.id,
        name = it.name,
        picture = it.picture,
        description = it.description,
        isPrivate = it.private,
        staggered = it.staggered,
        baseURL = it.baseURL,
        controller = controllerContainer
      )
      _allPlugins[plugin.id] = plugin
    }
  }

  private fun extractPeerIdsFromPluginMessagePayload(payload: JsonObject): List<String>? {
    val peerIdsJsonArray = payload["peerIds"]?.jsonArray ?: return null
    val peerIds = mutableListOf<String>()
    for (peerId in peerIdsJsonArray) {
      peerIds.add(peerId.jsonPrimitive.content)
    }
    return peerIds
  }

  private suspend fun handlePluginSocketMessage(message: PluginSocketMessage) {
    when (message) {
      is PluginSocketMessage.AddPlugin -> {
        handleAddPluginSocketMessage(message)
      }

      is PluginSocketMessage.RemovePlugin -> {
        handleRemovePluginSocketMessage(message)
      }

      is PluginSocketMessage.EnablePluginForPeers -> {
        val data = buildJsonObject {
          put("enabledBy", message.enabledBy)
        }
        submitMessageToPlugin(message.pluginId, message.event, message.id, data)
      }

      is PluginSocketMessage.EnablePluginForRoom -> {
        val data = buildJsonObject {
          put("enabledBy", message.enabledBy)
        }
        submitMessageToPlugin(message.pluginId, message.event, message.id, data)
      }

      is PluginSocketMessage.DisablePluginForPeers -> {
        val data = buildJsonObject {
          put("disabledBy", message.disabledBy)
        }
        submitMessageToPlugin(message.pluginId, message.event, message.id, data)
      }

      is PluginSocketMessage.DisablePluginForRoom -> {
        val data = buildJsonObject {
          put("disabledBy", message.disabledBy)
        }
        submitMessageToPlugin(message.pluginId, message.event, message.id, data)
      }

      is PluginSocketMessage.CustomPluginEventToPeers -> {
        val data = buildJsonObject {
          put("data", message.pluginData)
        }
        submitMessageToPlugin(message.pluginId, message.event, message.id, data)
      }

      is PluginSocketMessage.CustomPluginEventToRoom -> {
        val data = buildJsonObject {
          put("data", message.pluginData)
        }
        submitMessageToPlugin(message.pluginId, message.event, message.id, data)
      }

      is PluginSocketMessage.StoreInsertKeys -> {
        val data = buildJsonObject {
          put("storeName", message.pluginStore.storeName)
          put("storeItems", message.pluginStore.getStoreItemsAsJsonArray())
        }
        submitMessageToPlugin(message.pluginStore.pluginId, message.event, message.id, data)
      }

      is PluginSocketMessage.StoreGetKeys -> {
        val data = buildJsonObject {
          put("storeName", message.pluginStore.storeName)
          put("storeItems", message.pluginStore.getStoreItemsAsJsonArray())
        }
        submitMessageToPlugin(message.pluginStore.pluginId, message.event, message.id, data)
      }

      is PluginSocketMessage.StoreDeleteKeys -> {
        val data = buildJsonObject {
          put("storeName", message.pluginStore.storeName)
          put("storeItems", message.pluginStore.getStoreItemsAsJsonArray())
        }
        submitMessageToPlugin(message.pluginStore.pluginId, message.event, message.id, data)
      }

      is PluginSocketMessage.StoreDelete -> {
        val data = buildJsonObject {
          put("storeName", message.storeName)
        }
        submitMessageToPlugin(message.pluginId, message.event, message.id, data)
      }
    }
  }

  private suspend fun handleAddPluginSocketMessage(message: PluginSocketMessage.AddPlugin) {
    var pluginActive = false
    _activePlugins[message.pluginId]?.let { plugin ->
      if (plugin.isActive) {
        pluginActive = true
      }
    }

    if (!pluginActive) {
      enablePluginForLocalUser(message.pluginId, message.enabledBy)
      _activePlugins[message.pluginId]?.let { plugin ->
        if (plugin.isActive) {
          controllerContainer.eventController.triggerEvent(
            DyteEventType.OnPluginEnabled(plugin)
          )
        }
      }
    }
  }

  private fun handleRemovePluginSocketMessage(message: PluginSocketMessage.RemovePlugin) {
    _activePlugins[message.pluginId]?.let { plugin ->
      plugin.disableLocal()
      _activePlugins.remove(plugin.id)
      controllerContainer.eventController.triggerEvent(DyteEventType.OnPluginDisabled(plugin))
    }
  }

  private fun submitMessageToPlugin(
    pluginId: String,
    messageType: Int,
    messageId: String?,
    data: JsonObject
  ) {
    val eventPayload = buildJsonObject {
      put("type", messageType)
      put("uuid", messageId)
      put("payload", data)
    }
    submitEventToPlugin(pluginId, eventPayload)
  }

  companion object {
    private fun PluginStore.getStoreItemsAsJsonArray(): JsonArray {
      return buildJsonArray {
        storeItems.forEach { pluginStoreItem ->
          addJsonObject {
            put("timestamp", pluginStoreItem.timestamp)
            put("peerId", pluginStoreItem.peerId)
            put("payload", pluginStoreItem.payload)
            put("key", pluginStoreItem.key)
          }
        }
      }
    }

    /*
    * note(swapnil): The following properties are not present in the DyteJoinedMeetingParticipant of mobile-core:
    * 1. device: DeviceConfig
    * 2. screenShareEnabled: Boolean
    * 3. supportsRemoteControl: Boolean
    * 4. presetName: String?
    * 5. webinarStageStatus: WebinarStageStatus
    *
    * These properties won't be sent to the Plugin.
    * */
    private fun DyteJoinedMeetingParticipant.toJsonObject(): JsonObject {
      return buildJsonObject {
        put("id", id)
        put("userId", userId)
        put("name", name)
        put("picture", picture)
        put("isHost", isHost)
        put("clientSpecificId", clientSpecificId)
        putJsonObject("flags") {
          put("recorder", flags.recorder)
          put("hidden_participant", flags.hiddenParticipant)
          // web-core has both hidden_participant & hiddenParticipant. So covering both the cases.
          put("hiddenParticipant", flags.hiddenParticipant)
        }
        put("videoEnabled", videoEnabled)
        put("audioEnabled", audioEnabled)
      }
    }

    /*
    * note(swapnil): The following properties are not present in the DyteSelfParticipant of mobile-core:
    * 1. device: DeviceConfig
    * 2. role: Any
    * 3. supportsRemoteControl: Boolean
    * 4. presetName: String?
    * 5. stageStatus: StageStatus
    * 6. isRecorder: Boolean
    *
    * These properties won't be sent to the Plugin.
    *
    * We could have passed the presetName from MetaController but we are deliberately skipping it until
    * it is absolutely required by a Plugin to work.
    * Reason: Uniformity. We are not including the presetName in serialization of DyteJoinedMeetingParticipant as well.
    *
    * We are also not passing isRecorder because currently we are not parsing it from API response and
    * it is also not exposed by SelfController.
    * */
    private fun DyteSelfParticipant.toJsonObject(organizationId: String): JsonObject {
      return buildJsonObject {
        put("id", id)
        put("userId", userId)
        put("name", name)
        put("picture", picture)
        put("isHost", isHost)
        put("clientSpecificId", clientSpecificId)
        putJsonObject("flags") {
          put("recorder", flags.recorder)
          put("hidden_participant", flags.hiddenParticipant)
          // web-core has both hidden_participant & hiddenParticipant. So covering both the cases.
          put("hiddenParticipant", flags.hiddenParticipant)
        }
        put("videoEnabled", videoEnabled)
        put("audioEnabled", audioEnabled)

        val waitlistStatus = waitListStatus.name.lowercase()
        put("waitlistStatus", waitlistStatus)

        put("organizationId", organizationId)
        // put("isRecorder", permissions.host.canTriggerRecording)
      }
    }
  }
}