package io.dyte.core.controllers

import io.dyte.core.controllers.DyteEventType.OnWebinarStartedPresenting
import io.dyte.core.controllers.DyteEventType.OnWebinarStoppedPresenting
import io.dyte.core.models.DyteMediaPermission.ALLOWED
import io.dyte.core.models.DyteMediaPermission.CAN_REQUEST
import io.dyte.core.socket.events.OutboundMeetingEventType.ACCEPT_PRESENTING_REQUEST
import io.dyte.core.socket.events.OutboundMeetingEventType.REJECT_PRESENTING_REQUEST
import io.dyte.core.socket.events.OutboundMeetingEventType.REQUEST_TO_JOIN_STAGE
import io.dyte.core.socket.events.OutboundMeetingEventType.START_PRESENTING
import io.dyte.core.socket.events.OutboundMeetingEventType.STOP_PRESENTING
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketPeerLeftModel
import io.dyte.core.socket.events.payloadmodel.inbound.WebSocketWebinarStagePeer
import io.dyte.core.socket.events.payloadmodel.outbound.RoomState
import io.dyte.core.socket.events.payloadmodel.outbound.WebSocketJoinRoomModel
import io.dyte.core.socket.events.payloadmodel.outbound.WebSocketRoomStateModel
import kotlinx.coroutines.launch
import kotlinx.serialization.json.JsonPrimitive
import kotlinx.serialization.json.addJsonObject
import kotlinx.serialization.json.buildJsonArray
import kotlinx.serialization.json.buildJsonObject

internal class WebinarController(controllerContainer: IControllerContainer) :
  BaseController(controllerContainer), IWebinarController {
  private var isPresenting = false

  override val requestedParticipants: List<RequestToPresentParticipant>
    get() = requestsToPresent

  private val requestsToPresent = arrayListOf<RequestToPresentParticipant>()

  override suspend fun joinStage() {
    if (controllerContainer.presetController.permissions.media.video.permission == CAN_REQUEST || controllerContainer.presetController.permissions.media.audioPermission == CAN_REQUEST) {
      controllerContainer.socketController.sendMessage(
        REQUEST_TO_JOIN_STAGE,
        JsonPrimitive("REQUEST_TO_PRESENT")
      )
    } else if (controllerContainer.presetController.permissions.media.video.permission == ALLOWED || controllerContainer.presetController.permissions.media.audioPermission == ALLOWED) {
      controllerContainer.socketController.sendMessage(
        START_PRESENTING,
        null
      )
    } else {
      controllerContainer.loggerController.traceError("DyteWebinar::unauthorized_joinStage")
      throw UnsupportedOperationException("not allowed to join stage")
    }
  }

  override suspend fun leaveStage() {
    controllerContainer.socketController.sendMessage(STOP_PRESENTING, null)
  }

  override fun onRequestedToPresent(requestType: String) {
    if (requestType == "REQUEST_TO_PRESENT") {
      controllerContainer.eventController.triggerEvent(DyteEventType.OnWebinarPresentRequest)
    }
  }

  override suspend fun acceptRequestToPresent() {
    isPresenting = true
    controllerContainer.socketController.sendMessage(START_PRESENTING, null)
  }

  override suspend fun rejectRequestToPresent() {
    isPresenting = false
    controllerContainer.socketController.sendMessage(STOP_PRESENTING, null)
  }

  override fun onPeerAddedToStage(peerId: String) {
    val addedPeer = requestsToPresent.filter { it.id == peerId }
    if (addedPeer.isNotEmpty()) {
      requestsToPresent.removeAll(addedPeer.toSet())
      controllerContainer.eventController.triggerEvent(
        DyteEventType.OnWebinarPresentRequestAccepted(
          addedPeer.first()
        )
      )
    }
  }

  override fun onPeerRejectedToStage(peerId: String) {
    val removedPeers = requestsToPresent.filter { it.id == peerId }
    if(removedPeers.isNotEmpty()) {
      requestsToPresent.removeAll(removedPeers.toSet())
      controllerContainer.eventController.triggerEvent(
        DyteEventType.OnWebinarPresentRequestRejected(
          removedPeers.first()
        )
      )
    }
  }

  override fun isPresenting(): Boolean {
    return isPresenting
  }

  override fun onStoppedPresenting() {
    isPresenting = false
    controllerContainer.eventController.triggerEvent(OnWebinarStoppedPresenting)
  }

  override fun onStartedPresenting() {
    isPresenting = true
    controllerContainer.eventController.triggerEvent(OnWebinarStartedPresenting)
    serialScope.launch {
      controllerContainer.sfuUtils.produceMedia()
    }
  }

  override fun canJoinStage(): Boolean {
    return controllerContainer.presetController.permissions.media.audioPermission == CAN_REQUEST
      || controllerContainer.presetController.permissions.media.video.permission == CAN_REQUEST
  }

  override fun shouldJoinStage(): Boolean {
    return controllerContainer.presetController.permissions.media.audioPermission == ALLOWED
      || controllerContainer.presetController.permissions.media.video.permission == ALLOWED
  }

  override fun onRequestToPresentPeerAdded(requestToPresentParticipant: RequestToPresentParticipant) {
    requestsToPresent.add(requestToPresentParticipant)
    controllerContainer.eventController.triggerEvent(
      DyteEventType.OnWebinarPresentRequestAdded(
        requestToPresentParticipant
      )
    )
  }

  override fun onRequestToPresentPeerClosed(requestToPresentParticipant: WebSocketPeerLeftModel) {
    val addedPeer = requestsToPresent.filter { it.id == requestToPresentParticipant.id }
    if (addedPeer.isNotEmpty()) {
      requestsToPresent.removeAll(addedPeer.toSet())
      controllerContainer.eventController.triggerEvent(
        DyteEventType.OnWebinarPresentRequestClosed(
          addedPeer.first()
        )
      )
    }
  }

  override fun onRequestToPresentPeerWithdrawn(requestToPresentParticipant: WebSocketWebinarStagePeer) {
    val withdrawnPeer = requestsToPresent.filter { it.id == requestToPresentParticipant.id }
    if (withdrawnPeer.isNotEmpty()) {
      requestsToPresent.removeAll(withdrawnPeer.toSet())
      controllerContainer.eventController.triggerEvent(
        DyteEventType.OnWebinarPresentRequestWithdrawn(
          withdrawnPeer.first()
        )
      )
    }
  }

  override suspend fun acceptRequest(id: String) {
    if (controllerContainer.presetController.permissions.host.canAcceptRequests.not()) {
      controllerContainer.loggerController.traceError("DyteWebinar::unauthorized_acceptRequest")
      throw UnsupportedOperationException("not allowed to accept webinar requests")
    }
    val arrayIds = buildJsonArray {
      addJsonObject {
        put("id", JsonPrimitive(id))
      }
    }
    controllerContainer.socketController.sendMessage(ACCEPT_PRESENTING_REQUEST, arrayIds)
  }

  override suspend fun acceptAllRequest() {
    if (controllerContainer.presetController.permissions.host.canAcceptRequests.not()) {
      controllerContainer.loggerController.traceError("DyteWebinar::unauthorized_acceptAllRequest")
      throw UnsupportedOperationException("not allowed to accept all webinar requests")
    }
    val arrayIds = buildJsonArray {
      requestsToPresent.forEach {
        addJsonObject {
          put("id", JsonPrimitive(it.id))
        }
      }
    }
    controllerContainer.socketController.sendMessage(ACCEPT_PRESENTING_REQUEST, arrayIds)
  }

  override suspend fun rejectRequest(id: String) {
    if (controllerContainer.presetController.permissions.host.canAcceptRequests.not()) {
      controllerContainer.loggerController.traceError("DyteWebinar::unauthorized_rejectRequest")
      throw UnsupportedOperationException("not allowed to reject webinar requests")
    }
    val idToReject = buildJsonObject {
      put("id", JsonPrimitive(id))
    }
    controllerContainer.socketController.sendMessage(REJECT_PRESENTING_REQUEST, idToReject)
  }

  override suspend fun rejectAllRequest() {
    if (controllerContainer.presetController.permissions.host.canAcceptRequests.not()) {
      controllerContainer.loggerController.traceError("DyteWebinar::unauthorized_rejectAllRequest")
      throw UnsupportedOperationException("not allowed to reject all webinar requests")
    }
    requestsToPresent.forEach {
      val idToReject = buildJsonObject {
        put("id", JsonPrimitive(it.id))
      }
      controllerContainer.socketController.sendMessage(REJECT_PRESENTING_REQUEST, idToReject)
    }
  }

  override fun handleRoomJoined(roomState: WebSocketJoinRoomModel) {
    requestsToPresent.clear()
    requestsToPresent.addAll(roomState.requestToJoinPeersList?: emptyList())
  }
}

interface IWebinarController {
  val requestedParticipants: List<RequestToPresentParticipant>
  suspend fun joinStage()
  suspend fun leaveStage()

  fun onRequestedToPresent(requestType: String)
  suspend fun acceptRequestToPresent()
  suspend fun rejectRequestToPresent()

  fun onRequestToPresentPeerAdded(requestToPresentParticipant: RequestToPresentParticipant)
  fun onRequestToPresentPeerClosed(requestToPresentParticipant: WebSocketPeerLeftModel)
  fun onRequestToPresentPeerWithdrawn(requestToPresentParticipant: WebSocketWebinarStagePeer)

  suspend fun acceptRequest(id: String)
  suspend fun acceptAllRequest()
  suspend fun rejectRequest(id: String)
  suspend fun rejectAllRequest()

  fun isPresenting(): Boolean

  fun onPeerAddedToStage(peerId: String)
  fun onPeerRejectedToStage(peerId: String)
  fun onStoppedPresenting()
  fun onStartedPresenting()
  fun canJoinStage(): Boolean
  fun shouldJoinStage(): Boolean

  fun handleRoomJoined(roomState: WebSocketJoinRoomModel)
}

@kotlinx.serialization.Serializable
data class RequestToPresentParticipant(
  val id: String,
  val name: String
)